<?php

namespace WordpressModels\DataModels;

use WordpressModels\DataModels\Attribute\DateTimeFormat;
use WordpressModels\DataModels\Attribute\NoSerialize;
use WordpressModels\DataModels\Attribute\Serialize;
use DateTime;
use DateTimeInterface;
use Exception;
use Jawira\CaseConverter\CaseConverterException;
use Jawira\CaseConverter\Convert;
use JetBrains\PhpStorm\ArrayShape;
use ReflectionClass;
use ReflectionException;
use ReflectionProperty;

trait Hydratable
{


    /**
     * @throws CaseConverterException|ReflectionException
     */
    public static function hydrate(self $obj = null, array $data = [])
    {
        $obj = $obj ?? new static();

        // early exit when the data array is not associative
        if (array_is_list($data)) {
            return $obj;
        }

        $reflectionClass = new ReflectionClass($obj);

        foreach ($data as $key => $value) {
            if (is_serialized($value)) {
                $value = unserialize($value);
            }

            $cc = (new Convert($key))->toCamel();
            if ($reflectionClass->hasProperty($cc) && $reflectionProperty = $reflectionClass->getProperty($cc)) {
                if ($reflectionProperty->getAttributes(NoSerialize::class)) {
                    // skip NoSerialize attributed properties
                    continue;
                }

                $propType = $reflectionProperty->getType();
                $childClass = $propType->getName();

                if ($childClass === 'array') {
                    $value = self::hydrateArray($reflectionProperty, $value);
                } else {
                    $value = self::hydrateValue($childClass, $value);
                }

                $setterMethodName = 'set' . ucfirst($cc);
                if ($reflectionClass->hasMethod($setterMethodName)
                    && $reflectionMethod = $reflectionClass->getMethod($setterMethodName)) {
                    $reflectionMethod->invoke($obj, $value);
                } else {
                    $reflectionProperty->setValue($obj, $value);
                }
            }
        }

        return $obj;
    }

    /**
     * @param $childClass
     * @param $value
     * @return DateTime|mixed
     * @throws Exception
     */
    private static function hydrateValue($childClass, $value)
    {
        if (is_object($value) && in_array(Hydratable::class, class_uses($childClass))) {
            // create a new instance and hydrate using the childClass reference
            $value = $childClass::hydrate(null, $value);
        } elseif (is_subclass_of($childClass, DateTimeInterface::class) && $value && is_string($value)) {
            // create DateTime instance when the childClass is valid, and value is non-falsy
            $value = new DateTime('@' . strtotime($value));
        } elseif (is_serialized($value)) {
            $value = unserialize($value);
        }

        return $value;
    }

    /**
     * Hydrate an array typed property. `$reflectionProperty` is assumed to have the 'array' ReflectionNamedType.
     *
     * Depending on what the property allows (null/non-null), a return value is early-exit when the `$value` is falsy.
     *
     * PHPDoc is scanned using ReflectionProperty::getDocComment(), where some type or class is attempted to be
     * determined, using the `var` attribute. If no item type is defined in doc comments, generic mapping is applied.
     *
     * @throws Exception
     */
    private static function hydrateArray(ReflectionProperty $reflectionProperty, mixed $value): array|null
    {
        if (!$value) {
            // early exit falsy values, map to null if allowed, else empty array
            return $reflectionProperty->getType()->allowsNull() ? null : [];
        }

        $arrayClass = null;
        if (preg_match('/@var\s+(\S+)/', $reflectionProperty->getDocComment(), $matches)) {
            // some match was found, get the first group (class name)
            [, $type] = $matches;

            // omit array [] (E.g. stdClass[] -> stdClass)
            $arrayClass = substr($type, 0, strlen($type) - 2);
        }

        // map all items using either the found array item class, or null for generic items
        return array_map(fn($item) => self::hydrateValue($arrayClass, $item), $value);
    }

    /**
     * @param array{serialize_arrays: bool, date_format: string, casing: string} $options
     * @return array
     * @throws ReflectionException
     */
    public function dehydrate(#[ArrayShape(['serialize_arrays' => 'bool', 'date_format' => 'string', 'casing' => 'string', 'depth' => 'int', 'omit_additional' => 'bool'])]
                              array $options = []): array
    {
        $out = [];
        $reflectionClass = new ReflectionClass($this);
        $casing = $options['casing'] ?? 'snake';
        $caseMethod = 'to' . ucfirst($casing);

        foreach ($reflectionClass->getProperties() as $reflectionProperty) {
            $caseConverter = new Convert($reflectionProperty->getName());
            $outputCase = $caseConverter->{$caseMethod}();
            $camelCase = $casing === 'camel' ? $outputCase : $caseConverter->toCamel();

            $propOptions = $this->getPropOptions($reflectionProperty, $options);

            $getter = 'get' . ucfirst($camelCase);
            $isser = 'is' . ucfirst($camelCase);
            if (!$reflectionClass->hasMethod($getter) && !$reflectionClass->getMethod($getter = $isser)) {
                continue;
            }
            $value = $reflectionClass->getMethod($getter)->invoke($this);
            $out[$outputCase] = $this->dehydrateValue($value, $propOptions);
        }

        if (!($options['omit_additional'] ?? false)) {
            // get public getter and is-er methods
            $publicGetters = $this->getGetterMethods($reflectionClass);
            foreach ($publicGetters as $method) {
                // check for Serialize attributes
                $attrs = $method->getAttributes(Serialize::class);

                if ($attr = ($attrs[0] ?? false)) {
                    /** @var Serialize $serializeAttribute */
                    $serializeAttribute = $attr->newInstance();

                    if ($serializeAttribute->getMaxDepth() !== -1 && ($options['depth'] ?? 0) > $serializeAttribute->getMaxDepth()) {
                        continue;
                    }

                    // either the attribute prop name or the method name is used as serialized name
                    $outName = $serializeAttribute->getPropertyName() ?? $method->getName();
                    // convert to correct casing
                    $outName = (new Convert($outName))->{$caseMethod}();
                    // invoke the method
                    $value = $method->invoke($this);
                    // store the value
                    $out[$outName] = $this->dehydrateValue($value, $options);
                }
            }
        }

        return $out;
    }

    /**
     * @param $value
     * @param array $options
     * @return mixed
     */
    private function dehydrateValue($value,
                                    #[ArrayShape(['serialize_arrays' => 'bool', 'date_format' => 'string', 'casing' => 'string', 'depth' => 'int', 'omit_additional' => 'bool'])]
                                    array $options = []): mixed
    {
        if (is_object($value) && in_array(Hydratable::class, class_uses($childClass))) {
            // dehydrate hydratable model
            return $value->dehydrate(array_merge($options, ['depth' => ($options['depth'] ?? 0) + 1]));
        } elseif ($value instanceof DateTimeInterface) {
            // format date
            return $value->format($options['date_format'] ?? 'Y-m-d H:i:s');
        } elseif (is_array($value)) {
            // dehydrate array values
            foreach ($value as $key => $v) {
                $vs[$key] = $this->dehydrateValue($v, $options);
            }

            $vs ??= [];

            return ($options['serialize_arrays'] ?? false) ? serialize($vs) : $vs;
        } else {
            return $value;
        }
    }

    /**
     * @param ReflectionProperty $reflectionProperty
     * @param array $options
     * @return array
     */
    #[ArrayShape(['serialize_arrays' => 'bool', 'date_format' => 'string', 'casing' => 'string', 'depth' => 'int', 'omit_additional' => 'bool'])]
    private function getPropOptions(ReflectionProperty $reflectionProperty, array $options): array
    {
        $childClass = $reflectionProperty->getType()?->getName();
        // check for DateTimeFormat serialization attributes
        if (is_subclass_of($childClass, DateTimeInterface::class)
            && $reflectionAttribute = ($reflectionProperty->getAttributes(DateTimeFormat::class)[0] ?? false)) {
            /** @var DateTimeFormat $dateTimeFormatAttribute */
            $dateTimeFormatAttribute = $reflectionAttribute->newInstance();
            $options['date_format'] = $dateTimeFormatAttribute->getFormat();
        }

        return $options;
    }

    /**
     * Get all public Getter and Is-er methods for the given reflection class.
     *
     * @param ReflectionClass $reflectionClass
     * @return \ReflectionMethod[]
     */
    private function getGetterMethods(ReflectionClass $reflectionClass): array
    {
        return array_filter($reflectionClass->getMethods(\ReflectionMethod::IS_PUBLIC),
            fn(\ReflectionMethod $m) => str_starts_with($m->getName(), 'get') || str_starts_with($m->getName(), 'is'));
    }
}
