<?php

namespace RTMCustomMails\Rest;

use RTMCustomMails\Email\RTMCustomEmail;
use RTMCustomMails\EmailSections\SectionValidation;
use RTMCustomMails\PDFInvoices\WoocommercePDFInvoices;
use RTMCustomMails\Preview\PreviewEmails;
use RTMCustomMails\Preview\PreviewTriggerArguments;
use RTMCustomMails\Util;
use RTMMailComposer\Model\EmailSection;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\Serializer;
use Throwable;
use Timber\Loader;
use WC_Emails;
use WordpressModels\Rest\AbstractRestApi;
use WordpressModels\Rest\Attribute\RestRoute;
use WP_Error;
use WP_Http;
use WP_REST_Request;
use WP_REST_Response;

/**
 * REST API Endpoints for modifying emails.
 *
 */
class EmailApi extends AbstractRestApi
{

    const ARGS_EMAIL_SECTION = [
        'id' => [
            'required' => true,
            'type' => 'string'
        ],
        'section' => [
            'type' => 'string'
        ],
        'sort-order' => [
            'required' => true,
            'type' => 'integer'
        ],
        'title' => [
            'type' => 'string'
        ],
        'enable' => [
            'type' => 'boolean'
        ],
        'show-title' => [
            'type' => 'boolean'
        ],
        'condition' => [
            'type' => 'array',
        ],
        'text' => [
            'type' => 'string'
        ]
    ];

    private Serializer $serializer;

    public function __construct()
    {
        $this->serializer = new Serializer([new ObjectNormalizer()], [new JsonEncoder()]);
        parent::__construct('wrce');
    }

    #[RestRoute(
        route: 'preview',
        methods: ['GET'],
        permission: "role('administrator')",
        arguments: [
            'id' => [
                'required' => true,
                'type' => 'string',
            ],
            'objectId' => [
                'required' => true,
                'type' => 'number',
            ]
        ],
        public: false),
    ]
    public function preview(WP_REST_Request $request): array
    {
        PreviewTriggerArguments::instance();

        $emailId = $request->get_param('id');
        $objectId = $request->get_param('objectId');

        $e = array_filter(WC_Emails::instance()->get_emails(), fn($e) => $e->id === $emailId);
        $key = current(array_keys($e));
        $email = RTMCustomEmail::asCustomEmail(current($e));
        $content = PreviewEmails::instance()->renderEmail($key, $objectId, $email);

        return [
            'content' => $content,
            'context' => $email->getRenderContext()
        ];
    }

    #[RestRoute('settings', ['GET'], "role('administrator')", [
        'id' => [
            'required' => true,
            'type' => 'string'
        ]
    ],
        public: false)]
    public function settings(WP_REST_Request $request): array
    {
        $emailId = $request->get_param('id');

        return [
            'content' => Util::getEmailById($emailId)->generate_settings_html([], false)
        ];
    }

    #[RestRoute('sections', ['GET'], "role('administrator')", [
        'id' => [
            'required' => true,
            'type' => 'string'
        ]
    ],
        public: false)]
    public function sections(WP_REST_Request $request): array
    {
        $emailId = $request->get_param('id');

        $sections = Util::getEmailById($emailId)->getSections();
        usort($sections, fn(EmailSection $a, EmailSection $b) => $a->getSortOrder() <=> $b->getSortOrder());

        return [
            'content' => array_map(fn($section) => $this->serializer->normalize($section), $sections)
        ];
    }

    #[RestRoute('search', ['POST'], "role('administrator')", [
        'type' => [
            'required' => true,
            'type' => 'string'
        ],
        'search' => [
            'required' => true,
            'type' => 'string'
        ]
    ],
        public: false)]
    public function searchObject(WP_REST_Request $request)
    {
        $type = $request->get_param('type');
        $search = $request->get_param('search');

        return apply_filters('wrce_preview_find_objects_' . $type, [], $search);
    }

    #[RestRoute('attachments', ['GET', 'POST'], "role('administrator')", [
        'id' => ['type' => 'string', 'required' => true]
    ], public: false)]
    public function documents(WP_REST_Request $request): WP_Error|array
    {
        if (!class_exists('\WPO_WCPDF')) {
            return new WP_Error('This endpoint requires wpovernight\'s "WooCommerce PDF Invoices & Packing Slips" plugin.');
        }
        $emailId = $request->get_param('id');

        if ($request->get_method() === 'GET') {
            return ['content' => WoocommercePDFInvoices::getDocumentsForEmail($emailId)];
        } elseif ($request->get_method() === 'POST' && $emailId) {
            $documents = $request->get_param('documents');

            // validate document types
            $instance = forward_static_call(['\WPO_WCPDF', 'instance']);
            /** @var WPO\WC\PDF_Invoices\Documents\Order_Document[] $objs */
            $objs = $instance->documents->get_documents();
            $emailDocs = array_filter($objs, fn($doc) => in_array($doc->get_type(), $documents));

            $result = [];

            foreach ($emailDocs as $document) {
                $docType = $document->get_type();
                $options = get_option('wpo_wcpdf_document_settings_' . $docType);
                if (!$options) {
                    continue;
                }
                $options['attach_to_email_ids'] = array_merge($options['attach_to_email_ids'], [$emailId => '1']);
                $result[$docType] = update_option('wpo_wcpdf_document_settings_' . $docType, $options);
            }

            return ['content' => $result];
        }

        return new WP_Error("Invalid request");
    }

    #[RestRoute('document/toggle', ['POST'], 'role(administrator', [
        'id' => [
            'required' => true,
            'type' => 'string'
        ],
        'document' => [
            'required' => true,
            'type' => 'string'
        ]
    ],
        public: false)]
    public function toggleDocument(WP_REST_Request $request): WP_Error|array
    {
        $emailId = $request->get_param('id');
        $document = $request->get_param('document');
        $propable_slug = str_replace('-', '_', $document);
        $options = get_option('wpo_wcpdf_documents_settings_' . $propable_slug);
        if (!$options) {
            return new WP_Error("Could not find settings for document \"$document\".");
        }
        $attached = $options['attach_to_email_ids'] ?? [];

        if ($attached[$emailId] ?? false) {
            unset($attached[$emailId]);
        } else {
            $attached[$emailId] = '1';
        }

        $options['attach_to_email_ids'] = $attached;

        if (!update_option('wpo_wcpdf_documents_settings_' . $propable_slug, $options)) {
            return new WP_Error("Could not update settings for document \"$document\".");
        }

        return ['result' => true];
    }

    #[RestRoute('add-section', ['POST'], "role('administrator')", self::ARGS_EMAIL_SECTION, public: false)]
    #[RestRoute('set-section', ['PATCH'], "role('administrator')", self::ARGS_EMAIL_SECTION, public: false)]
    public function addSection(WP_REST_Request $request): WP_REST_Response
    {
        $emailId = $request->get_param('id');
        $sortOrder = $request->get_param('sort-order');

        $sectionId = $request->get_param('section');
        $enabled = $request->get_param('enable') ?: false;
        $displayTitle = $request->get_param('show-title') ?: false;
        $condition = $request->get_param('condition');
        $title = $request->get_param('title');
        $text = $request->get_param('text');

        $email = Util::getEmailById($emailId);

        $sections = $email->getSections();
        $existing = current(array_filter($sections, fn($s) => $s->getId() === $sectionId));
        $safeText = preg_replace('/(\{[\%\{][^‘’]+)[‘’]+([^‘’]+)[‘’]+([^‘’]+[\%\}]\})/', '$1\'$2\'$3', $text);

        $sectionObject = new EmailSection(
            $sectionId,
            $enabled,
            $title,
            $displayTitle,
            $safeText,
            $condition ?: [],
            $sortOrder
        );

        try {
            (new Loader())->get_twig()->createTemplate($safeText);
        } catch (Throwable $t) {
            return new WP_REST_Response(['errors' => [[
                'property' => 'text',
                'message' => 'Invalid Template, please check variable syntax.<br/><strong>Details:</strong> ' . $t->getMessage()
            ]]], WP_Http::BAD_REQUEST);
        }

        if ($errors = SectionValidation::instance()->validateEmailWithSections($emailId, [$sectionObject])) {
            return new WP_REST_Response(['errors' => $errors], WP_Http::BAD_REQUEST);
        }

        if (!$existing) {
            $sections[] = $sectionObject;

            usort($sections, fn($a, $b) => $a->getSortOrder() <=> $b->getSortOrder());
        } else {
            $idx = array_search($existing, $sections);

            $section = array_merge($this->serializer->normalize($existing), $this->serializer->normalize($sectionObject));
            $sections[$idx] = new EmailSection($section['id'], $section['enable'], $section['title'], $section['showTitle'], $section['text'], $section['condition'], $section['sortOrder']);
        }

        $email->setSections($sections);

        return new WP_REST_Response($sections);
    }

    #[RestRoute('sections', ['POST'], "role('administrator')", [
        'id' => [
            'type' => 'string',
            'required' => true
        ],
        'sections' => [
            'type' => 'array',
            'required' => true
        ]
    ], public: false)]
    public function setSections(WP_REST_Request $request): array
    {
        $emailId = $request->get_param('id');
        $sections = $request->get_param('sections');

        $sections = array_map(fn($section) => new EmailSection($section['id'], $section['enable'], $section['title'], $section['showTitle'], $section['text'], $section['condition'], $section['sortOrder']), $sections);

        $email = Util::getEmailById($emailId);
        $result = $email->setSections($sections);
        $message = $result ? 'Succesfully saved sections' : 'Could not save settings';

        return [
            'result' => $result,
            'message' => $message
        ];
    }

    #[RestRoute('validate-template', ['POST'], "role('administrator')", [
        'id' => [
            'type' => 'string',
            'required' => true
        ],
        'sections' => [
            'type' => 'array',
            'required' => true
        ]
    ], public: false)]
    public function validateEmailSections(WP_REST_Request $request): array
    {
        $emailId = $request->get_param('id');
        $sections = $request->get_param('sections');
        $sections = array_map(fn($s) => new EmailSection($s['id'], $s['enable'], $s['title'], $s['showTitle'], $s['text'], $s['condition'], $s['sortOrder']), $sections);

        if ($errors = SectionValidation::instance()->validateEmailWithSections($emailId, $sections)) {
            return [
                'result' => false,
                'errors' => $errors
            ];
        }

        return [
            'result' => true
        ];
    }

    #[RestRoute('export', ['POST'], "role('administrator')", [
        'id' => [
            'type' => 'string',
            'required' => true
        ]
    ], public: false)]
    public function exportEmailSettings(WP_REST_Request $request)
    {
        $emailId = $request->get_param('id');
        if (!$email = Util::getEmailById($emailId)) {
            return new WP_REST_Response([
                'result' => false,
                'message' => 'Invalid Email ID'
            ], WP_Http::BAD_REQUEST);
        }

        $optionKey = $email->get_option_key();
        return get_option($optionKey);
    }

    #[RestRoute('import', ['POST'], "role('administrator')", [
        'id' => [
            'type' => 'string',
            'required' => true
        ],
        'settings' => [
            'type' => 'object',
            'required' => true
        ]
    ], public: false)]
    public function importEmailSettings(WP_REST_Request $request): WP_REST_Response|array
    {
        $emailId = $request->get_param('id');
        $settings = $request->get_param('settings');

        if (!$email = Util::getEmailById($emailId)) {
            return new WP_REST_Response([
                'result' => false,
                'message' => 'Invalid Email ID'
            ], WP_Http::BAD_REQUEST);
        }

        $optionKey = $email->get_option_key();
        $options = update_option($optionKey, $settings);

        return ['result' => $options];
    }
}