<?php

namespace RTMCustomMails\Pages;

use RTMCustomMails\Core;
use RTMCustomMails\Email\RTMCustomEmail;
use RTMCustomMails\EmailSections\EmailSections;
use RTMCustomMails\PDFInvoices\WoocommercePDFInvoices;
use RTMCustomMails\Preview\EmailObjects;
use RTMCustomMails\Schedule\ScheduledEmails;
use RTMCustomMails\Util;
use WC_Email;
use WC_Emails;
use WordpressModels\Page;
use WordpressModels\Traits\SingletonTrait;
use WP_Term;

/**
 *
 */
class EmailOverviewPage extends Page
{

    use SingletonTrait;

    private ?string $emailId;
    private ?string $view;
    private ?array $emails;

    public function __construct()
    {
        parent::__construct(
            'email-overview',
            'rtm_custom_emails',
            __('Emails', 'wrce'),
            'toplevel',
            'administrator',
            58,
            'dashicons-email-alt',
            assets: Core::instance()->assets()
        );
    }

    public function init()
    {
        parent::init();

        $this->emails = array_map([RTMCustomEmail::class, 'asCustomEmail'], WC_Emails::instance()->get_emails());
        $this->view = $_GET['view'] ?? 'sections';
        $this->emailId = $_GET['emailId'] ?? current($this->emails)->id;
    }

    public function doEnqueue()
    {
        $supportsAttachments = class_exists('\WPO_WCPDF');
        if ($supportsAttachments && ($instance = forward_static_call(['\WPO_WCPDF', 'instance']))) {
            $documents = array_reduce(
                $instance->documents->get_documents('all'),
                fn($carry, $doc) => $carry + [
                        $doc->get_type() => [
                            'title' => $doc->get_title(),
                            'type' => $doc->get_type(),
                            'attached' => array_keys($doc->get_setting('attach_to_email_ids') ?: [])
                        ]
                    ], []);
        } else {
            $documents = [];
        }

        $sectionPresets = EmailSections::getSectionPresets();

        usort($this->emails, fn(RTMCustomEmail $a, RTMCustomEmail $b) => strcmp($a->title, $b->title));
        // generate view data
        $email = Util::getEmailById($this->emailId);

        $data = match ($this->view) {
            'sections' => $email->getSections(),
            'attachments' => WoocommercePDFInvoices::getDocumentsForEmail($this->emailId),
            'settings' => $email->generate_settings_html([], false),
            default => [],
        };

        $objectType = $email->getObjectType();
        $registeredObjectSupport = EmailObjects::instance()->getSupported();
        $objectSupport = $registeredObjectSupport[$objectType] ?? null;

        /**
         * Create the default object selector configuration.
         *
         * Used in the search input field for previewing emails using existing objects.
         *
         * Configuration is injected in the javascript and follows mostly the
         * <a href="https://tom-select.js.org/docs/api/">TomSelect</a> configuration. Note that the following configuration
         * keys are always overwritten: `{
         *  maxOptions: null,
         *  loadThrottle: 1000,
         *  placeHolder: 'Search {CurrentObjectType}...'
         *  shouldLoad: function,
         *  load: function,
         *  onFocus: function,
         *  onItemAdd: function,
         *  options: array<CurrentObjectType>,
         *  render: {loading: function}
         * }`.
         * Also, no functions can be passed from PHP to JS, avoid settings configuration keys which demand functions. There
         * are some workarounds, but don't expect support.
         *
         * The render configuration can be setup using templated strings, where properties of the given object may be
         * used as template variables.
         * For example, order:
         *  Orders are remapped as associative arrays with the following keys: `orderId`, `orderNumber`, `text`.
         *  We can create a render template for `option`: `<div>{orderId}</div>`. The {orderId} param is mapped from
         *  the `order` object at runtime.
         *
         * @param array $config
         * @return array
         */
        $objectSelectorConfig = apply_filters('wrce_preview_object_selector_config', []);

        $allowedObjectVariables = array_reduce(
            array_keys($registeredObjectSupport),
            fn($carry, $type) => $carry + [
                    $type => apply_filters('wrce_email_allowed_variables/object_type=' . $type, [])
                ],
            []);

        $allowedEmailVariables = array_map(fn(WC_Email $email) => [
            ...apply_filters('wrce_email_allowed_variables/email_id=' . $email->id, []),
        ], $this->emails);

        // tinymce
        wp_enqueue_editor();
        // add rest params
        wp_localize_script('email-overview', 'wpRest', [
            'url' => get_rest_url(),
            'nonce' => wp_create_nonce('wp_rest')
        ]);
        wp_localize_script('email-overview', 'overview', [
            'initialEmail' => ['id' => $this->emailId, 'name' => $email->get_title()],
            'initialView' => $this->view,
            'initialViewData' => $data,
            'documents' => $documents,
            'allSections' => $sectionPresets,
            'objectSelectorConfig' => $objectSelectorConfig,
            'latestObjects' => $objectSupport ? $objectSupport->getLatestObjects() : [],
            'allowedVariables' => [
                'object' => $allowedObjectVariables,
                'email' => $allowedEmailVariables
            ],
            'allowedPreviewObjectTypes' => apply_filters('wrce_preview_object_selector_types', ["order"]),
            'allowedAttachmentObjectTypes' => apply_filters('wrce_allowed_attachment_object_types', []),
            'defaultSections' => apply_filters('wrce_email_section_hooks', []),
        ]);
    }

    /**
     * @inheritDoc
     */
    public function buildContext(array $context = []): array
    {
        $allSections = EmailSections::getSectionPresets();

        $tabs = [
            'sections' => __('Sections', 'wrce')
        ];

        $supportsAttachments = class_exists('\WPO_WCPDF');
        if ($supportsAttachments && ($instance = forward_static_call(['\WPO_WCPDF', 'instance']))) {
            $documents = $instance->documents->get_documents('all');
            $emailDocuments = array_reduce($documents, function ($carry, $doc) {
                $email_ids = $doc->get_setting('attach_to_email_ids') ?: [];
                foreach ($email_ids as $email_id => $enabled) {
                    if ($enabled) {
                        $carry[$email_id][] = $doc;
                    }
                }
                return $carry;
            }, []);

            $tabs['attachments'] = __('Attachments', 'wrce');
        } else {
            $documents = [];
            $emailDocuments = [];
        }

        $tabs['settings'] = __('Settings', 'wrce');

        /** @var WP_Term[] $terms */
        $terms = apply_filters('wrce_email_section_filter_terms', []);
        usort($terms, fn($a, $b) => strcmp($a->taxonomy, $b->taxonomy));

        $enabledColors = array_reduce($this->emails, function ($carry, $e) {
            $conditional = apply_filters('wrce_email_enabled_conditional', false, $e);

            $carry[$e->id] = is_bool($conditional) && $conditional
                ? 'warning'
                : ($e->is_enabled() ? 'success' : 'danger'); // not conditional,
            return $carry;
        }, []);

        return array_merge($context, [
            'emails' => $this->emails,
            'emailId' => $this->emailId,
            'view' => $this->view,
            'emailDocs' => $emailDocuments,
            'allSections' => $allSections,
            'documents' => $documents,
            'tabs' => $tabs,
            'filterTerms' => $terms,
            'enabledColors' => $enabledColors,
            'schedulableEmails' => ScheduledEmails::getSchedulableEmails(),
            'supportsAttachments' => $supportsAttachments
        ]);
    }
}