<?php

namespace RTMCustomMails\ConditionProcessor;

use Symfony\Component\ExpressionLanguage\ExpressionFunction;
use Symfony\Component\ExpressionLanguage\ExpressionFunctionProviderInterface;
use WC_Abstract_Order;
use WC_Order_Item;
use WC_Order_Item_Product;

class OrderFunctionProvider implements ExpressionFunctionProviderInterface
{

    public function __construct()
    {
    }

    public function compile(string $value): string
    {
        return '';
    }


    /**
     * Verifies if the order contains given productId
     *
     * @param array $context
     * @param string $productId
     *
     * @return bool Whether the order contains given product id
     */
    public static function evaluateOrderHasProduct(array $context, string $productId) : bool
    {
        /** @var WC_Abstract_Order|null $order */
        $order = $context['order'] ?? null;
        if ($order) {
            /** @var WC_Order_Item|null $item */
            foreach ($order->get_items() as $item) {
                $item = $item->get_data();
                if ($item['product_id'] == $productId) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Verifies if the order contains a product wih given attribute id
     *
     * @param array $context
     * @param string $attributeId
     *
     * @return bool Whether the order contains a product wih given attribute id
     */
    public static function evaluateOrderHasProductWithAttribute(array $context, string $attributeId) : bool
    {
        /** @var WC_Abstract_Order|null $order */
        $order = $context['order'] ?? null;
        if (! $order) {
            return false;
        }
        /** @var WC_Order_Item|null $item */
        foreach ($order->get_items() as $item) {
            if (! $item instanceof WC_Order_Item_Product) {
                continue;
            }
            $product = $item->get_product();
            if ($product) {
                $attributes = $product->get_attributes();
                foreach ( $attributes as $attribute ) {
                    return in_array( $attributeId, $attribute->get_options() );
                }
            }
        }
        return false;
    }

    /**
     * Verifies if the order has the given status string
     *
     * @param array $context
     * @param string $statusString
     *
     * @return bool Whether the order has the given status string
     */
    public static function evaluateOrderHasStatus(array $context, string $statusString) : bool
    {
        /** @var WC_Abstract_Order|null $order */
        $order = $context['order'] ?? null;
        if ($order) {
            return $order->get_status() == $statusString;
        }
        return false;
    }

    /**
     * Verifies if the order was made by a guest
     *
     * @param array $context
     *
     * @return bool Whether the order was made by a guest
     */
    public static function evaluateOrderIsGuest(array $context) : bool
    {
        /** @var WC_Abstract_Order|null $order */
        $order = $context['order'] ?? null;
        if ($order) {
            return $order->get_user() == false;
        }
        return false;
    }

    /**
     * @return ExpressionFunction[] Returns the functions associated with this function provider
     */
    public function getFunctions(): array
    {
        return [
            new ExpressionFunction('order_has_product', $this->compile(...), $this->evaluateOrderHasProduct(...)),
            new ExpressionFunction('order_has_product_with_attribute', $this->compile(...), $this->evaluateOrderHasProductWithAttribute(...)),
            new ExpressionFunction('order_has_status', $this->compile(...), $this->evaluateOrderHasStatus(...)),
            new ExpressionFunction('order_is_guest', $this->compile(...), $this->evaluateOrderIsGuest(...)),
        ];
    }
}
