<?php

namespace RTMCustomMails\Pages;

use ActionScheduler;
use ActionScheduler_Action;
use ActionScheduler_NullAction;
use Throwable;
use WC_Emails;
use WordpressModels\Page;
use WordpressModels\Traits\SingletonTrait;

class ScheduledEmailsPage extends Page
{

    use SingletonTrait;

    /**
     * @var array|mixed
     */
    private array $filter = [];
    private string $orderBy = 'date';
    private string $order = 'DESC';
    private int $limit = 20;
    private int $offset = 0;

    public function __construct()
    {
        parent::__construct(
            'scheduled-emails',
            'scheduled_emails',
            __('Scheduled Emails'),
            __('Scheduled Emails'),
            EmailOverview::instance()->getMenuSlug(),
        );
    }

    public function init()
    {
        try {
            if (($action_id = $_GET['cancel'] ?? false)
                && wp_verify_nonce($_GET['nonce'] ?? '', 'cancel::' . $action_id)
                && !(ActionScheduler::store()->fetch_action($action_id) instanceof ActionScheduler_NullAction)) {
                ActionScheduler::store()->cancel_action($action_id);
            }
            if (($action_id = $_GET['run'] ?? false)
                && wp_verify_nonce($_GET['nonce'] ?? '', 'run::' . $action_id)
                && !(ActionScheduler::store()->fetch_action($action_id) instanceof ActionScheduler_NullAction)) {
                ActionScheduler::runner()->process_action($action_id, 'WRCE Scheduled Emails');
            }
        } catch (Throwable $throwable) {
            // todo: log error while processing action
        }
        // get pagination and filter options from query args
        $this->filter = $_GET['filter'] ?? ['status' => \ActionScheduler_Store::STATUS_PENDING];
        $this->orderBy = array_key_first($_GET['sort'] ?? []) ?? 'date';
        $this->order = ($_GET['sort'] ?? ['date' => 'ASC'])[$this->orderBy] ?? null;
        $this->limit = abs($_GET['limit'] ?? 20);
        $this->offset = (max(1, ($_GET['paged'] ?? 1)) - 1) * $this->limit;
    }

    /**
     * @param $context
     * @return array
     */
    public function buildContext($context = [])
    {
        $type_arguments = apply_filters('wrce_get_scheduled_emails_and_args', []);

        $emails = WC_Emails::instance()->get_emails();

        $schedulerStore = ActionScheduler::store();
        if ($actionIds = ($filter['action_id'] ?? [])) {
            $actionIds = is_array($actionIds) ? $actionIds : [$actionIds];

            $actions = array_reduce($actionIds, function ($c, $id) use ($schedulerStore) {
                $action = $schedulerStore->fetch_action($id);
                $c[$id] = [
                    'status' => $schedulerStore->get_status($id),
                    'finished' => $action->is_finished(),
                    'args' => $action->get_args(),
                    'group' => $action->get_group(),
                    'hook' => $action->get_hook(),
                    'schedule' => $action->get_schedule(),
                ];
                return $c;
            }, []);
            $totalItems = count($actions);
        } else {
            $args = $this->generateActionArgs($this->limit, $this->offset, $this->orderBy, $this->order, $this->filter);

            // get total item count
            $totalItems = ActionScheduler::store()->query_actions(array_merge($args, ['per_page' => -1, 'offset' => 0]), 'count');

            // get all actions
            $actions = $this->getScheduledEmailActions($args);
        }

        // some static data
        $labels = $this->generateStatusLabels();
        $statusIcons = [
            \ActionScheduler_Store::STATUS_PENDING => ['clock-o', 'primary'],
            \ActionScheduler_Store::STATUS_RUNNING => ['times', 'secondary'],
            \ActionScheduler_Store::STATUS_COMPLETE => ['check', 'success'],
            \ActionScheduler_Store::STATUS_FAILED => ['times', 'warning'],
            \ActionScheduler_Store::STATUS_CANCELED => ['ban', 'danger'],
        ];

        return array_merge($context, [
            'title' => 'Scheduled Emails',
            'type_arguments' => $type_arguments,
            'next_scheduled_action_date' => $next_scheduled_action_date ?? null,
            'emails' => $emails,
            'schedule' => $schedule ?? null,
            'individual_scheduled_actions' => $actions,
            'statusIcons' => $statusIcons,
            'totalPages' => ceil($totalItems / $this->limit),
            'allStatuses' => $labels,
            'currentFilter' => $this->filter
        ]);
    }

    public function getScheduledEmailActions(array $args): array
    {
        /** @var ActionScheduler_Action[] $individual_scheduled_actions */
        $individual_scheduled_actions = as_get_scheduled_actions($args);
        $actions = [];
        foreach ($individual_scheduled_actions as $action_id => $action) {
            $actions[$action_id] = [
                'status' => ActionScheduler::store()->get_status($action_id),
                'finished' => $action->is_finished(),
                'args' => $action->get_args(),
                'group' => $action->get_group(),
                'hook' => $action->get_hook(),
                'schedule' => $action->get_schedule(),
            ];
        }
        return $actions;
    }

    public function generateStatusLabels(): array
    {
        $labels = ActionScheduler::store()->get_status_labels();
        foreach ($labels as $status => $label) {
            $count = ActionScheduler::store()->query_actions(['hook' => 'wrce_send_scheduled_email', 'per_page' => -1, 'status' => $status], 'count');
            $labels[$status] .= " ($count)";
        }
        return $labels;
    }

    public function generateActionArgs($limit, $offset, $orderBy, $order, $filter): array
    {
        $args = [
            'hook' => 'wrce_send_scheduled_email',
            'per_page' => $limit,
            'offset' => $offset,
            'orderby' => $orderBy,
            'order' => $order,
        ];

        if ($statuses = ($filter['status'] ?? false)) {
            $args['status'] = is_string($statuses) ? explode(',', $statuses) : $statuses;
        } else {
            $args['status'] = \ActionScheduler_Store::STATUS_PENDING;
        }
        return $args;
    }

    public function doEnqueue()
    {
        // TODO: Implement doEnqueue() method.
    }
}