<?php

namespace RTMCustomMails\EmailSections;

use RTMCustomMails\Util;
use Timber\Timber;
use WordpressModels\Traits\SingletonTrait;
use WP_Error;
use WP_Term;

/**
 *
 */
class EmailSections
{
    use SingletonTrait;

    private function __construct()
    {
        Timber::$twig_cache = WP_DEBUG;

        add_filter('timber/locations', [$this, 'addTimberTemplateLocations']);
        add_filter('timber/cache/location', [$this, 'setTimberCacheLocation']);

        SectionValidation::instance();
    }

    /**
     * Get all email sections associated with the Woocommerce Email ID.
     * @param string $emailId
     * @return int|WP_Error|WP_Term[]
     */
    public static function getSectionsForEmail(string $emailId)
    {
        // try fetch from cache
        $sections = wp_cache_get('wrce_email_sections_' . $emailId);

        if (!$sections) {
            $section_opt = Util::getEmailById($emailId)->get_option('sections', []);
            $section_terms = $section_opt
                ? get_terms([
                    'taxonomy' => EmailSectionsTaxonomy::TAXONOMY_IDENTIFIER,
                    'hide_empty' => false,
                    'slug' => array_keys($section_opt)
                ])
                : [];

            $sections = array_reduce($section_terms, fn($carry, $term) => array_merge($carry, [$term->slug => [
                'id' => $term->slug,
                'slug' => $term->slug,
                'title' => $term->name,
                'text' => get_field('text', EmailSectionsTaxonomy::TAXONOMY_IDENTIFIER . '_' . $term->term_id),
                'showTitle' => $section_opt[$term->slug]['show_title'] ?? false,
                'enable' => $section_opt[$term->slug]['active'] ?? false,
                'condition' => $section_opt[$term->slug]['display_for'] ?? [],
                'sortOrder' => intval($section_opt[$term->slug]['sort_order'] ?? 1)
            ]]), []);

            usort($sections, fn($a, $b) => $a['sortOrder'] > $b['sortOrder'] ? 1 : -1);


            wp_cache_set('wrce_email_sections_' . $emailId, $sections);
        }

        return $sections;
    }

    public static function getSectionPresets()
    {
        return apply_filters('wrce_email_section_presets', get_option('wrce_email_section_presets', []));
    }

    /**
     * @param string $location
     * @return string
     */
    public function setTimberCacheLocation(string $location)
    {
        return WP_DEBUG ? (WRCE_ROOT_PATH . 'cache/') : $location;
    }

    /**
     * @param string[] $locations
     * @return string[]
     */
    public function addTimberTemplateLocations(array $locations = [])
    {
        $locations[] = WRCE_ROOT_PATH . 'templates/emails/';

        return $locations;
    }

}