<?php
/**
 * Class WC_Email_Customer_Reset_Password file.
 *
 * @package WooCommerce\Emails
 */

namespace RTMCustomMails\Email;


/**
 * Customer Reset Password.
 *
 * An email sent to the customer when they reset their password.
 *
 * @class       WC_Email_Customer_Reset_Password
 * @version     3.5.0
 * @package     WooCommerce\Classes\Emails
 * @extends     RTMCustomEmail
 */
class RTMCustomEmail_Customer_Reset_Password extends RTMCustomEmail
{

    /**
     * User ID.
     *
     * @var int
     */
    public $user_id;

    /**
     * User login name.
     *
     * @var string
     */
    public $user_login;

    /**
     * User email.
     *
     * @var string
     */
    public $user_email;

    /**
     * Reset key.
     *
     * @var string
     */
    public $reset_key;

    /**
     * Constructor.
     */
    public function __construct()
    {

        $this->id = 'customer_reset_password';
        $this->customer_email = true;

        $this->title = __('Reset password', 'woocommerce');
        $this->description = __('Customer "reset password" emails are sent when customers reset their passwords.', 'woocommerce');

        $this->template_html = '';
        $this->template_plain = '';

        // Call parent constructor.
        parent::__construct();

        $this->object_type = 'user';
    }

    /**
     * Get email subject.
     *
     * @return string
     * @since  3.1.0
     */
    public function get_default_subject()
    {
        return __('Password Reset Request for {site_title}', 'woocommerce');
    }

    /**
     * Get email heading.
     *
     * @return string
     * @since  3.1.0
     */
    public function get_default_heading()
    {
        return __('Password Reset Request', 'woocommerce');
    }

    /**
     * Trigger.
     *
     * @param string $user_login User login.
     * @param string $reset_key Password reset key.
     */
    public function trigger($user_login = '', $reset_key = '')
    {
        $this->setup_locale();

        if ($user_login && $reset_key) {
            $this->object = get_user_by('login', $user_login);
            $this->user_id = $this->object->ID;
            $this->user_login = $user_login;
            $this->reset_key = $reset_key;
            $this->user_email = stripslashes($this->object->user_email);
            $this->recipient = $this->user_email;
        }

        if ($this->is_enabled() && $this->get_recipient()) {
            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
        }

        $this->restore_locale();
    }

    public function prepareContext($baseContext)
    {
        return parent::prepareContext($baseContext) + [
                'user_id' => $this->user_id,
                'user_login' => $this->user_login,
                'reset_key' => $this->reset_key,
                'user' => new \WP_User($this->user_id),
                'recipient_name' => $this->user_login,
                'woocommerce_password_reset_url' => esc_url(add_query_arg(['key' => $this->reset_key, 'id' => $this->user_id], wc_get_endpoint_url('lost-password', '', wc_get_page_permalink('myaccount'))))
            ];
    }

    /**
     * Get content plain.
     *
     * @return string
     */
    public function get_content_plain()
    {
        return wc_get_template_html(
            $this->template_plain,
            [
                'email_heading' => $this->get_heading(),
                'user_id' => $this->user_id,
                'user_login' => $this->user_login,
                'reset_key' => $this->reset_key,
                'blogname' => $this->get_blogname(),
                'additional_content' => $this->get_additional_content(),
                'sent_to_admin' => false,
                'plain_text' => true,
                'email' => $this,
            ]
        );
    }

    /**
     * Default content to show below main email content.
     *
     * @return string
     * @since 3.7.0
     */
    public function get_default_additional_content()
    {
        return __('Thanks for reading.', 'woocommerce');
    }

    public function getTriggerHooks(): array
    {
        return [
            'woocommerce_reset_password_notification'
        ];
    }
}

