<?php
/**
 * Class WC_Email_Customer_Refunded_Order file.
 *
 * @package WooCommerce\Emails
 */

namespace RTMCustomMails\Email;


use WC_Order;

/**
 * TODO: triggers
 * Customer Refunded Order Email.
 *
 * Order refunded emails are sent to the customer when the order is marked refunded.
 *
 * @class    WC_Email_Customer_Refunded_Order
 * @version  3.5.0
 * @package  WooCommerce\Classes\Emails
 * @extends  RTMCustomEmail
 */
class RTMCustomEmail_Customer_Refunded_Order extends RTMCustomEmail
{

    /**
     * Refund order.
     *
     * @var WC_Order|bool
     */
    public $refund;

    /**
     * Is the order partial refunded?
     *
     * @var bool
     */
    public $partial_refund;

    /**
     * Constructor.
     */
    public function __construct()
    {
        $this->customer_email = true;
        $this->id = 'customer_refunded_order';
        $this->title = __('Refunded order', 'woocommerce');
        $this->description = __('Order refunded emails are sent to customers when their orders are refunded.', 'woocommerce');
        $this->template_html = '';
        $this->template_plain = '';
        $this->placeholders = [
            '{order_date}' => '',
            '{order_number}' => '',
        ];

        // Call parent constructor.
        parent::__construct();
    }

    /**
     * Get email subject.
     *
     * @param bool $partial Whether it is a partial refund or a full refund.
     * @return string
     * @since  3.1.0
     */
    public function get_default_subject($partial = false)
    {
        if ($partial) {
            return __('Your {site_title} order #{order_number} has been partially refunded', 'woocommerce');
        } else {
            return __('Your {site_title} order #{order_number} has been refunded', 'woocommerce');
        }
    }

    /**
     * Get email heading.
     *
     * @param bool $partial Whether it is a partial refund or a full refund.
     * @return string
     * @since  3.1.0
     */
    public function get_default_heading($partial = false)
    {
        if ($partial) {
            return __('Partial Refund: Order {order_number}', 'woocommerce');
        } else {
            return __('Order Refunded: {order_number}', 'woocommerce');
        }
    }

    /**
     * Get email subject.
     *
     * @return string
     */
    public function get_subject()
    {
        if ($this->partial_refund) {
            $subject = $this->get_option('subject_partial', $this->get_default_subject(true));
        } else {
            $subject = $this->get_option('subject_full', $this->get_default_subject());
        }
        return apply_filters('woocommerce_email_subject_customer_refunded_order', $this->format_string($subject), $this->object, $this);
    }

    /**
     * Get email heading.
     *
     * @return string
     */
    public function get_heading()
    {
        if ($this->partial_refund) {
            $heading = $this->get_option('heading_partial', $this->get_default_heading(true));
        } else {
            $heading = $this->get_option('heading_full', $this->get_default_heading());
        }
        return apply_filters('woocommerce_email_heading_customer_refunded_order', $this->format_string($heading), $this->object, $this);
    }

    /**
     * Set email strings.
     *
     * @param bool $partial_refund Whether it is a partial refund or a full refund.
     * @deprecated 3.1.0 Unused.
     */
    public function set_email_strings($partial_refund = false)
    {
    }

    /**
     * Full refund notification.
     *
     * @param int $order_id Order ID.
     * @param int $refund_id Refund ID.
     */
    public function trigger_full($order_id, $refund_id = null)
    {
        $this->trigger($order_id, false, $refund_id);
    }

    /**
     * Partial refund notification.
     *
     * @param int $order_id Order ID.
     * @param int $refund_id Refund ID.
     */
    public function trigger_partial($order_id, $refund_id = null)
    {
        $this->trigger($order_id, true, $refund_id);
    }

    /**
     * Trigger.
     *
     * @param int $order_id Order ID.
     * @param bool $partial_refund Whether it is a partial refund or a full refund.
     * @param int $refund_id Refund ID.
     */
    public function trigger($order_id = 0, $partial_refund = false, $refund_id = null)
    {
        $this->setup_locale();
        $this->partial_refund = $partial_refund;
        $this->id = $this->partial_refund ? 'customer_partially_refunded_order' : 'customer_refunded_order';

        if ($order_id) {
            $this->object = wc_get_order($order_id);
            $this->recipient = $this->object->get_billing_email();
            $this->placeholders['{order_date}'] = wc_format_datetime($this->object->get_date_created());
            $this->placeholders['{order_number}'] = $this->object->get_order_number();
        }

        if (!empty($refund_id)) {
            $this->refund = wc_get_order($refund_id);
        } else {
            $this->refund = false;
        }

        if ($this->is_enabled() && $this->get_recipient()) {
            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
        }

        $this->restore_locale();
    }

    public function prepareContext($baseContext)
    {
        return parent::prepareContext($baseContext) + [
                'partial_refund' => $this->partial_refund,
            ];
    }

    /**
     * Get content plain.
     *
     * @return string
     */
    public function get_content_plain()
    {
        return wc_get_template_html(
            $this->template_plain,
            [
                'order' => $this->object,
                'refund' => $this->refund,
                'partial_refund' => $this->partial_refund,
                'email_heading' => $this->get_heading(),
                'additional_content' => $this->get_additional_content(),
                'sent_to_admin' => false,
                'plain_text' => true,
                'email' => $this,
            ]
        );
    }

    /**
     * Default content to show below main email content.
     *
     * @return string
     * @since 3.7.0
     */
    public function get_default_additional_content()
    {
        return __('We hope to see you again soon.', 'woocommerce');
    }

    /**
     * Initialise settings form fields.
     */
    public function init_form_fields()
    {
        /* translators: %s: list of placeholders */
        $placeholder_text = sprintf(__('Available placeholders: %s', 'woocommerce'), '<code>' . esc_html(implode('</code>, <code>', array_keys($this->placeholders))) . '</code>');
        $this->form_fields = [
            'enabled' => [
                'title' => __('Enable/Disable', 'woocommerce'),
                'type' => 'checkbox',
                'label' => __('Enable this email notification', 'woocommerce'),
                'default' => 'yes',
            ],
            'subject_full' => [
                'title' => __('Full refund subject', 'woocommerce'),
                'type' => 'text',
                'desc_tip' => true,
                'description' => $placeholder_text,
                'placeholder' => $this->get_default_subject(),
                'default' => '',
            ],
            'subject_partial' => [
                'title' => __('Partial refund subject', 'woocommerce'),
                'type' => 'text',
                'desc_tip' => true,
                'description' => $placeholder_text,
                'placeholder' => $this->get_default_subject(true),
                'default' => '',
            ],
            'heading_full' => [
                'title' => __('Full refund email heading', 'woocommerce'),
                'type' => 'text',
                'desc_tip' => true,
                'description' => $placeholder_text,
                'placeholder' => $this->get_default_heading(),
                'default' => '',
            ],
            'heading_partial' => [
                'title' => __('Partial refund email heading', 'woocommerce'),
                'type' => 'text',
                'desc_tip' => true,
                'description' => $placeholder_text,
                'placeholder' => $this->get_default_heading(true),
                'default' => '',
            ],
            'additional_content' => [
                'title' => __('Additional content', 'woocommerce'),
                'description' => __('Text to appear below the main email content.', 'woocommerce') . ' ' . $placeholder_text,
                'css' => 'width:400px; height: 75px;',
                'placeholder' => __('N/A', 'woocommerce'),
                'type' => 'textarea',
                'default' => $this->get_default_additional_content(),
                'desc_tip' => true,
            ],
            'email_type' => [
                'title' => __('Email type', 'woocommerce'),
                'type' => 'select',
                'description' => __('Choose which format of email to send.', 'woocommerce'),
                'default' => 'html',
                'class' => 'email_type wc-enhanced-select',
                'options' => $this->get_email_type_options(),
                'desc_tip' => true,
            ],
        ];
    }

    public function getTriggerHooks(): array
    {
        return [
            ['woocommerce_order_fully_refunded_notification', 'trigger_full', 10, 2],
            ['woocommerce_order_partially_refunded_notification', 'trigger_partial', 10, 2],
        ];
    }
}

