<?php

namespace RTMCustomMails;

use RTMCustomMails\Email\RTMCustomEmail;
use RTMCustomMails\Email\RTMCustomEmail_Cancelled_Order;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Bank_Reminder;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Completed_Order;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Invoice;
use RTMCustomMails\Email\RTMCustomEmail_Customer_New_Account;
use RTMCustomMails\Email\RTMCustomEmail_Customer_New_Order;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Note;
use RTMCustomMails\Email\RTMCustomEmail_Customer_On_Hold_Order;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Processing_Order;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Refunded_Order;
use RTMCustomMails\Email\RTMCustomEmail_Customer_Reset_Password;
use RTMCustomMails\Email\RTMCustomEmail_Failed_Order;
use RTMCustomMails\Email\RTMCustomEmail_New_Order;
use RTMCustomMails\Schedule\ScheduledEmails;
use WC_Email;
use WC_Emails;
use WordpressModels\Assets;
use WordpressModels\Traits\SingletonTrait;

/**
 *
 */
class Core
{

    private Assets $assets;

    use SingletonTrait;

    private function __construct()
    {
        add_filter('woocommerce_email_classes', [$this, 'addCustomEmails']);
        add_filter('woocommerce_locate_template', [$this, 'tryLocateWoocommerceTemplate'], 10, 2);

        add_filter('woocommerce_email', [$this, 'addTriggerHookOption']);

        add_action('admin_enqueue_scripts', [$this, 'enqueueBootstrapForOrderEdit']);

        add_action('woocommerce_email_sent', [$this, 'addMetaOnEmailSent'], 10, 3);

        add_filter('wrce_email_enabled_conditional', [$this, 'checkEmailConditionallyEnabled']);

        add_filter('wrce_email_trigger_hooks', [$this, 'applyFilterHookOptionValue'], 10, 2);

        $this->aliasWoocommerceEmailClasses();
        OrderEmailLogMetaBox::instance();

        $this->assets->addTimberTemplateLocation(WRCE_ROOT_PATH . 'view/');
    }

    /**
     * Get the plugin assets instance.
     * @return Assets
     */
    public function assets()
    {
        return $this->assets ?? new Assets(WRCE_ASSETS_PATH . '/build', WRCE_ASSETS_URL . '/build');
    }

    /**
     * Add 'trigger_hooks' option to all available email types.
     *
     * @param WC_Emails $emails
     * @return void
     */
    public function addTriggerHookOption(WC_Emails $emails)
    {
        $instances = $emails->get_emails();

        foreach ($instances as $instance) {
            $instance = RTMCustomEmail::asCustomEmail($instance);
            $defaultTriggers = $instance->getTriggerHooks();
            if (!$instance->get_option('trigger_hooks')) {
                $instance->update_option('trigger_hooks', $defaultTriggers);
            }

            $hooks = apply_filters('wrce_email_trigger_hook_options', [], $instance);
            $hooks = apply_filters('wrce_email_trigger_hook_options/email_id=' . $instance->id, $hooks, $instance);
            $hooks = apply_filters('wrce_email_trigger_hook_options/object_type=' . $instance->getObjectType(), $hooks, $instance);

            foreach ($hooks as $group => $array) {
                if (is_array($array)) {
                    continue;
                }

                // set the trigger to an empty named group.
                $hooks[''][$group] = $array;
                unset($hooks[$group]);
            }

            $instance->form_fields['trigger_hooks'] = [
                'title' => __('Trigger Hooks', 'wrce'),
                'description' => __('This email will sent when specific event(s) occurs.', 'wrce'),
                'type' => 'multiselect',
                'default' => apply_filters('wrce_email_trigger_hooks', $defaultTriggers, $instance),
                'options' => $hooks
            ];
        }
    }

    /**
     * @param $template
     * @param $template_name
     * @return mixed|string
     */
    public function tryLocateWoocommerceTemplate($template, $template_name)
    {
        $t = WRCE_ROOT_PATH . 'templates/' . $template_name;
        return is_file($t) ? $t : $template;
    }

    /**
     *  Add a custom email to the list of emails WooCommerce should load
     *
     * @param array $email_classes available email classes
     * @return array filtered available email classes
     * @since 0.1
     */
    public function addCustomEmails($email_classes)
    {
        // add the email classes to the list of email classes that WooCommerce loads
        $email_classes['RTMCustomEmail_Customer_Bank_Reminder'] = new RTMCustomEmail_Customer_Bank_Reminder();
        $email_classes['RTMCustomEmail_Customer_New_Order'] = new RTMCustomEmail_Customer_New_Order();

        return $email_classes;
    }

    public function enqueueBootstrapForOrderEdit($hook)
    {
        global $post_type;
        if (in_array($hook, ['post.php', 'post-new.php']) && $post_type === 'shop_order') {
            Assets::instance()->enqueueCompiledScript('bootstrap-util');
        }
    }

    public function addMetaOnEmailSent($return, $id, $emailInstance)
    {
        if ($emailInstance instanceof RTMCustomEmail) {
            $renderContext = $emailInstance->getRenderContext();;
            $atomDate = (new \DateTime())->format(\DateTimeInterface::ATOM);
            if ($emailInstance->getObjectType() === 'order' && ($order = $renderContext['order'] ?? false)) {
                /** @var \WC_Abstract_Order $order */
                if ($return) {
                    // add sent metadata to order
                    $order->add_meta_data("_wrce_{$id}_sent", $atomDate);

                    $currentActionId = ScheduledEmails::instance()->getCurrentActionId();
                    if ($currentActionId && method_exists($order, 'add_order_note')) {
                        $order->add_order_note('Email "' . $emailInstance->get_title() . '" was sent successfully.');
                    }
                } else {
                    // failed to send email (or at least `wp_mail` hook returned `false`
                    if (method_exists($order, 'add_order_note')) {
                        $rtmMailSettings = get_option('rtm_mail_settings', []);
                        $msg = 'Failed to send email <strong>' . $emailInstance->get_title() . '</strong>';

                        if (class_exists('\RtmMail\Core') && ($rtmMailSettings['block_emails'] ?? false) === true) {
                            $msg .= " (RTM Mail plugin is blocking emails)";
                        } else {
                            $msg .= " (something blocking emails / transport failed)";
                        }

                        $order->add_order_note($msg);
                    }

                    $failures = $order->get_meta("_wrce_{$id}_send_failure") ?: [];
                    $failures[] = $atomDate;
                    $order->update_meta_data("_wrce_{$id}_send_failure", $failures);
                    $order->save_meta_data();
                }
            } else {
                do_action('wrce_add_meta_on_sent', $return, $id, $emailInstance);
            }
        }
    }

    public function applyFilterHookOptionValue(array $hooks, WC_Email $email)
    {
        return $email->get_option('trigger_hooks', $hooks);
    }

    /**
     * Alias default WooCommerce email classes, such that they are not registered.
     * @return void
     */
    private function aliasWoocommerceEmailClasses()
    {
        // load WC_Email and WC_Settings_Api classes if not loaded
        if (!class_exists('WC_Email')) {
            $woocommerceIncludes = realpath(WRCE_ROOT_PATH . '../woocommerce/includes');
            $settingsApiFile = 'abstracts/abstract-wc-settings-api.php';
            $emailsFile = 'emails/class-wc-email.php';

            if (!is_file("$woocommerceIncludes/$settingsApiFile") || !is_file("$woocommerceIncludes/$emailsFile")) {
                // skip email loading if woocommerce files are not found
                return;
            }
            require_once "$woocommerceIncludes/$settingsApiFile";
            require_once "$woocommerceIncludes/$emailsFile";
        }

        // alias woocommerce default emails, such that their `require` calls will immediately return
        class_alias(RTMCustomEmail_Cancelled_Order::class, 'WC_Email_Cancelled_Order');
        class_alias(RTMCustomEmail_Customer_Completed_Order::class, 'WC_Email_Customer_Completed_Order');
        class_alias(RTMCustomEmail_Customer_Invoice::class, 'WC_Email_Customer_Invoice');
        class_alias(RTMCustomEmail_Customer_New_Account::class, 'WC_Email_Customer_New_Account');
        class_alias(RTMCustomEmail_Customer_Note::class, 'WC_Email_Customer_Note');
        class_alias(RTMCustomEmail_Customer_On_Hold_Order::class, 'WC_Email_Customer_On_Hold_Order');
        class_alias(RTMCustomEmail_Customer_Processing_Order::class, 'WC_Email_Customer_Processing_Order');
        class_alias(RTMCustomEmail_Customer_Refunded_Order::class, 'WC_Email_Customer_Refunded_Order');
        class_alias(RTMCustomEmail_Customer_Reset_Password::class, 'WC_Email_Customer_Reset_Password');
        class_alias(RTMCustomEmail_Failed_Order::class, 'WC_Email_Failed_Order');
        class_alias(RTMCustomEmail_New_Order::class, 'WC_Email_New_Order');
    }


    public function checkEmailConditionallyEnabled(\WC_Email $email)
    {
        global $wp_filter;
        $filters = $wp_filter['woocommerce_email_enabled_' . $email->id] ?? [];
        // callbacks not empty
        return !empty($filters->callbacks);
    }

}