<?php

namespace WordpressModels;

use WordpressModels\Traits\AbstractSingletonTrait;
use WP_Post;

abstract class PostMetaBox
{

    use AbstractSingletonTrait;

    protected const CONTEXTS = ['normal', 'side', 'advanced'];
    protected const PRIORITIES = ['high', 'core', 'default', 'low'];

    private string $id;
    private string $title;
    private string $context;
    private string $priority;

    private ?string $postType;
    private array $enqueueScripts;
    private ?Assets $assets;

    /**
     * @param string $id The metabox identifier
     * @param string $title This metabox title
     * @param string|null $postType The posttype for which to diplay this meta-box
     * @param value-of<static::CONTEXTS> $context Section in which the metabox is displayed ('normal', 'side', or 'advanced')
     * @param value-of<static::PRIORITIES> $priority 'high', 'core', 'default', or 'low'
     * @param string[] $enqueueScripts list of to enqueue scripts (via Assets util)
     */
    public function __construct(
        string $id,
        string $title,
        ?string $postType = null,
        string $context = 'advanced',
        string $priority = 'default',
        array $enqueueScripts = [],
        ?Assets $assets = null
    ) {
        $this->id             = $id;
        $this->title          = $title;
        $this->postType       = $postType;
        $this->context        = $context;
        $this->priority       = $priority;
        $this->enqueueScripts = $enqueueScripts;
        $this->assets = $assets;

        add_action('add_meta_boxes' . ($postType ? "_$postType" : ''), [$this, 'registerMetaBox']);

        add_action('load-post.php', [$this, 'enqueueScriptAction']);
        add_action('load-post-new.php', [$this, 'enqueueScriptAction']);

        add_filter('should_render_metabox_' . $id, [$this, 'shouldRender'], 10, 2);
    }

    /**
     * Add the enqueue_admin_script action for 'load-post[-new].php' action on 'shop_order' post types.
     * @return void
     */
    public function enqueueScriptAction()
    {
        global $typenow;
        if ($typenow === $this->postType) {
            add_action('admin_enqueue_scripts', [$this, 'enqueueScripts'], 10, 1);
        }
    }

    /**
     * Enqueue $this->enqueueScripts via Assets util (compiled via webpack)
     * @return void
     */
    public function enqueueScripts()
    {
        if (!$this->assets) {
            return;
        }
        foreach ($this->enqueueScripts as $script) {
            $this->assets->enqueueCompiledScript($script);
        }
    }

    /**
     * Add the additional order info meta box.
     * @return void
     */
    public function registerMetaBox($post)
    {
        if (apply_filters('should_render_metabox_' . $this->id, true, $post)) {
            add_meta_box(
                $this->id,
                $this->title,
                [$this, 'renderMetaBox'],
                $this->postType,
                $this->context,
                $this->priority
            );
        }
    }

    /**
     * Whether the metabox should be rendered.
     *
     * @param bool $value
     * @param WP_Post $post
     *
     * @return bool
     */
    public function shouldRender(bool $value, WP_Post $post)
    {
        return $value;
    }

    /**
     * Render function.
     *
     * Should echo the contents of the metabox.
     *
     * @param WP_Post $post
     *
     * @return void
     */
    public abstract function renderMetaBox(WP_Post $post);

}
