<?php

namespace WordpressModels;

use Timber\Timber;

/**
 *
 */
abstract class Page
{

    protected string $pageId;
    protected string $parentMenuSlug;
    protected string $menuSlug;
    protected string $capability;
    protected string $title;
    protected int|float|null $position;
    protected string $icon;
    /**
     * @var string[]
     */
    private array $additionalScripts;
    private ?Assets $assets;

    public function __construct(
        $pageId,
        $menuSlug,
        $title,
        $parentMenuSlug = 'toplevel',
        $capability = 'administrator',
        $position = null,
        $icon = '',
        $additionalScripts = [],
        ?Assets $assets = null
    )
    {
        $this->pageId = $pageId;
        $this->menuSlug = $menuSlug;
        $this->parentMenuSlug = $parentMenuSlug;
        $this->capability = $capability;
        $this->title = $title;
        $this->position = $position;
        $this->icon = $icon;
        $this->additionalScripts = $additionalScripts;
        $this->assets = $assets;

        add_action('admin_menu', [$this, 'addAdminMenuItem']);

        if (isset($this->menuSlug)) {
            $prefix = str_ends_with($this->parentMenuSlug, '.php')
                ? str_replace('.php', '', $this->parentMenuSlug)
                : $this->parentMenuSlug;
            add_action('load-' . $prefix . '_page_' . $this->menuSlug, [$this, 'init']);
        }
    }

    /**
     * Initializes the page.
     * @return void
     * @hooked load-{parent}_page_{slug}
     */
    public function init()
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueueScripts']);
    }

    /**
     * Renders the page.
     *
     * @return void
     */
    public function renderPage()
    {
        $context = $this->buildContext(
            array_merge(Timber::context(), [
                'pageId' => $this->pageId,
                'title' => $this->title
            ])
        );
        Timber::render("page-$this->pageId.twig", $context);
    }

    /**
     * Build the context for the Twig template.
     *
     * @param array $context Base context provided with Timber::context(), the page ID and title.
     * @return mixed
     */
    abstract public function buildContext(array $context = []);

    public function enqueueScripts()
    {
        if ($this->assets) {
            $this->assets->enqueueCompiledScript($this->pageId);

            foreach ($this->additionalScripts as $script) {
                $this->assets->enqueueCompiledScript($script);
            }
        }

        $this->doEnqueue();
    }


    abstract public function doEnqueue();

    public function addAdminMenuItem()
    {
        if ($this->parentMenuSlug !== 'toplevel') {
            add_submenu_page(
                $this->parentMenuSlug,
                $this->title,
                $this->title,
                $this->capability,
                $this->menuSlug,
                [$this, 'renderPage'],
                $this->position ?: '0'
            );
        } else {
            add_menu_page(
                $this->title,
                $this->title,
                $this->capability,
                $this->menuSlug,
                [$this, 'renderPage'],
                $this->icon,
                $this->position ?: '0'
            );
        }
    }

    /**
     * @return string
     */
    public function getPageId(): string
    {
        return $this->pageId;
    }

    /**
     * @return string
     */
    public function getParentMenuSlug(): string
    {
        return $this->parentMenuSlug;
    }

    /**
     * @return string
     */
    public function getMenuSlug(): string
    {
        return $this->menuSlug;
    }

    /**
     * @return string
     */
    public function getCapability(): string
    {
        return $this->capability;
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @return float|int|null
     */
    public function getPosition(): float|int|null
    {
        return $this->position;
    }

    /**
     * @return string
     */
    public function getIcon(): string
    {
        return $this->icon;
    }
    
}
