<?php

namespace WordpressModels\Rest;

use WordpressModels\Rest\Attribute\RestRoute;
use WordpressModels\Traits\AbstractSingletonTrait;
use WP_REST_Response;

/**
 * @template T
 */
abstract class AbstractRestApi
{

    use AbstractSingletonTrait;

    /**
     * @var mixed|string
     */
    private string $namespace;

    /**
     * @param string $namespace
     */
    protected function __construct(string $namespace)
    {
        add_action('rest_api_init', [$this, 'registerRoutes']);
        add_action('rest_api_init', [$this, 'registerRoutesAttributes']);

        $this->namespace = $namespace;
    }

    /**
     * Autoregister REST routes attributed with the `WordpressModels\Route` attribute.
     * @return void
     */
    public function registerRoutesAttributes()
    {
        $reflectionClass = new \ReflectionClass($this);
        $reflectionMethods = $reflectionClass->getMethods(\ReflectionMethod::IS_PUBLIC);

        foreach ($reflectionMethods as $method) {
            foreach ($method->getAttributes(RestRoute::class) as $attribute) {
                /** @var RestRoute $route */
                $route = $attribute->newInstance();

                $callback = null;
                if ($permission = $route->permission) {
                    $parts = explode(":", $permission);

                    if (count($parts) > 1) {
                        [$permissionHandler, $validateValue] = $parts;
                        $callback = apply_filters('rest_validate_callback_' . $permissionHandler, null, $validateValue, $this);
                    } elseif (method_exists(Validation::class, $permission)) {
                        $callback = [Validation::class, $permission];
                    }
                }

                register_rest_route($this->namespace, $route->route, [
                    'methods' => $route->methods ?? [],
                    'callback' => [$this, $method->getName()],
                    'permission_callback' => $callback,
                    'args' => $route->arguments ?? [],
                    'show_in_index' => $route->public
                ]);
            }
        }
    }

    public function getNamespace(): string
    {
        return $this->namespace;
    }


    /**
     * Manually register REST routes. You might prefer to use this with custom parameter validations.
     *
     * @return void
     */
    public function registerRoutes(): void
    {
    }

    /**
     * @param $route
     * @param array{methods: string, callback: callable, permission_callback: callable } $setup
     * @param array{validate_callback: callable, sanitize_callback: callable, required: bool, default: mixed}[] $arguments
     * @param bool $override
     * @return void
     */
    public function registerRoute($route, $setup, $arguments = [], $override = false)
    {
        register_rest_route($this->getNamespace(), $route, array_merge($setup, ['args' => $arguments]), $override);
    }

    /**
     * Prevent cloning.
     */
    private function __clone()
    {
    }

    /**
     * Prevent unserializing.
     */
    final public function __wakeup()
    {
        _doing_it_wrong(__FUNCTION__, __('Unserializing instances of this class is forbidden.', 'wrce'), '0.4.0');
        die();
    }

    /**
     * @param $code
     * @param $message
     * @param $context
     * @return WP_REST_Response
     */
    protected function error($code, $message, $context = null)
    {
        return new WP_REST_Response(
            array_filter([
                'message' => $message,
                'data' => $context
            ]),
            $code
        );
    }

}
