<?php

namespace RTMCustomMails\EmailSections;

use RTMCustomMails\Email\RTMCustomEmail;
use RTMCustomMails\Model\EmailSection;
use Timber\Timber;
use Twig\Environment;
use Twig\Extension\SandboxExtension;
use Twig\Sandbox\SecurityPolicy;
use WC_Email;
use WordpressModels\Traits\SingletonTrait;
use WP_Term;

/**
 *
 */
class TwigSectionRender
{

    use SingletonTrait;

    private ?Environment $twig = null;

    private function __construct()
    {
        add_action('wrce_email_sections', [$this, 'renderEmailSections'], 10, 2);

        add_action('wrce_render_email_section', [$this, 'renderSection'], 10, 3);

        add_filter('wrce_section_matches_conditions', [$this, 'simpleMatchSectionConditions'], 10, 4);

        add_filter('timber/twig', [$this, 'saveCurrentTwigEnv']);
    }

    /**
     * Render an email section term.
     *
     * @param EmailSection $section
     * @param WC_Email $email
     * @param array $args
     * @return void
     */
    public function renderSection(EmailSection $section, WC_Email $email, array $args)
    {
        if ($section->isShowTitle()) {
            echo "<h4>{$section->getTitle()}</h4>";
        }

        // get the current extension list, to reset later
        $currentExtensions = $this->twig->getExtensions();

        // todo: define policy based on allowed variables
        $policy = new SecurityPolicy();
        $this->twig->addExtension(new SandboxExtension($policy, true));

        $text = html_entity_decode($section->getText());
        try {
            $compiledTemplate = Timber::compile_string($text, $args);
        } catch (\Throwable $throwable) {
            $compiledTemplate = $this->twig?->isDebug()
                ? "<div style='background: red; color: white'><strong>[DEBUG] Error in section \"{$section->getId()}\":</strong> {$throwable->getMessage()}</div>"
                : '';
        }

        // reset twig extensions
        $this->twig->setExtensions($currentExtensions);

        echo wptexturize(wpautop($compiledTemplate));
    }

    /**
     * Calculate email sections for the current email type and arguments.
     *
     * Sections can be either EmailSection objects, which are then passed to the `wrce_render_email_section` action,
     * or regular old hooks (e.g. `woocommerce_email_header`).
     *
     * @param WC_Email $email
     * @param array $args
     * @return void
     * @throws \ReflectionException
     */
    public function renderEmailSections(WC_Email $email, array $args)
    {
        $email = RTMCustomEmail::asCustomEmail($email);
        $sections = $email->getSections();

        // render sections
        foreach ($sections as $section) {
            // skip if section is disabled or conditional
            if (!$section->isEnable() || !apply_filters('wrce_section_matches_conditions', true, $section, $email, $args)) {
                continue;
            }

            [$action, $action_arguments] = apply_filters('wrce_email_section_action', $section->getId(), $args);

            // set default args for default hook
            if ($action === 'wrce_render_email_section') {
                $action_arguments = [$section, $email, $args];
            }

            do_action($action, ...$action_arguments);
        }
    }

    /**
     * Match conditions by comparing intersections between terms contained in order and condition set in $section.
     *
     * @param bool $value
     * @param EmailSection $section
     * @param WC_Email $email
     * @param $args
     * @return bool true if $value is true and:
     *                  -   $section['condition'] does not exist or is falsy
     *                  -   some match between term ids in $section['condition'] and `wrce_email_sections_get_conditional_terms` filter results.
     */
    public function simpleMatchSectionConditions(bool $value, EmailSection $section, WC_Email $email, $args)
    {
        // early exit if no condition set
        if (!$section->getCondition()) {
            return $value;
        }

        $condition_terms = apply_filters('wrce_email_sections_get_conditional_terms', [], $args, $email);
        if (empty(array_filter($condition_terms, fn($a) => !($a instanceof WP_Term)))) {
            $condition_terms = array_map(fn($t) => $t->term_id, $condition_terms);
        }

        return $value && array_intersect($section->getCondition(), $condition_terms);
    }

    public function saveCurrentTwigEnv($twig)
    {
        return $this->twig = $twig;
    }

}