<?php

namespace RTMCustomMails\EmailSections;

use RTMCustomMails\Email\RTMCustomEmail;
use RTMCustomMails\Util;
use WC_Abstract_Order;
use WC_Email;
use WC_Meta_Data;
use WordpressModels\Traits\SingletonTrait;
use WP_User;

/**
 * Contains functions for injecting context into specific email types.
 *
 * Functions are defined static, such that other classes can statically use them.
 */
class TwigContextInjection
{

    use SingletonTrait;

    private function __construct()
    {
        add_filter('timber/context', [self::class, 'injectEmailContext'], 50);

        add_filter('wrce_inject_context/default', [self::class, 'injectDefaultContext']);

        add_filter('wrce_inject_context/admin=1', [self::class, 'injectAdminContext']);
        add_filter('wrce_inject_context/recipient_type=admin', [self::class, 'injectAdminContext']);
        add_filter('wrce_inject_context/admin=0', [self::class, 'injectNonAdminContext']);
        add_filter('wrce_inject_context/recipient_type=customer', [self::class, 'injectNonAdminContext']);

        add_filter('wrce_inject_context/object_type=order', [self::class, 'injectOrderContext']);
    }

    /**
     * @param array $context
     * @return array|mixed|void
     */
    public static function injectEmailContext(array $context)
    {
        $context = apply_filters("wrce_inject_context/default", $context);

        if (($email = $context['email'] ?? false) instanceof WC_Email) {

            if ($email instanceof RTMCustomEmail) {
                $context = apply_filters("wrce_inject_context/object_type={$email->getObjectType()}", $context, $email->object, $email);
                $recipientType = $email->getRecipientType();
                $context = apply_filters("wrce_inject_context/recipient_type={$recipientType}", $context, $email);
            }

            if (!isset($recipientType)) {
                $admin_customer = (int)!$email->is_customer_email();
                $context = apply_filters("wrce_inject_context/admin=$admin_customer", $context, $email);
            }

            $context = apply_filters("wrce_inject_context/email_id=$email->id", $context, $email);
        }
        return $context;
    }

    /**
     * @param array $context
     * @return array
     */
    public static function injectDefaultContext(array $context)
    {
        return $context;
    }

    /**
     * @param array $context
     * @return array
     */
    public static function injectAdminContext(array $context)
    {
        $user = $context['user'] ?? get_user_by('email', get_option('admin_email'));
        $context['recipient_name'] = $user->first_name . ' ' . $user->last_name;

        return $context;
    }

    /**
     * @param array $context
     * @return array
     */
    public static function injectNonAdminContext(array $context)
    {
        if (isset($context['order'])) {
            $context['recipient_name'] = $context['order']->get_billing_first_name();
        } else {
            $context['recipient_name'] ??= $context['user']?->first_name
                ?: $context['billing_first_name']
                    ?: $context['user_login']
                        ?: (get_bloginfo('name') . ' Customer');
        }
        return $context;
    }

    /**
     * @param array $context
     * @return array
     * @throws \ReflectionException
     */
    public static function injectOrderContext(array $context): array
    {
        /** @var WC_Abstract_Order|null $order */
        $order = $context['order'] ?? null;

        if ($order) {
            if ($user = new WP_User($order->get_customer_id())) {
                $context['customer'] = $user;
            }

            $context['recipient_name'] = $user->get_billing_first_name();

            $meta = array_reduce($order->get_meta_data(), function (array $c, WC_Meta_Data $m) {
                $meta_key = $m->get_data()['key'];
                $key = str_starts_with($meta_key, '_') ? substr($meta_key, 1) : $meta_key;
                $c[$key] = $m->get_data()['value'];
                return $c;
            }, []);

            $billing_meta = array_filter($meta, fn($key) => str_starts_with($key, 'billing_'), ARRAY_FILTER_USE_KEY);
            $meta = array_filter(array_diff_assoc($meta, $billing_meta), fn($k) => !str_starts_with($k, 'buckaroo_'), ARRAY_FILTER_USE_KEY);

            $order_getters = Util::getClassMethods($order, 'get_order_', ['currency']);
            $billing_getters = Util::getClassMethods($order, 'get_billing_', []);
            $formatted_getters = Util::getClassMethods($order, 'get_formatted_', []);
            $customer_getters = Util::getClassMethods($order, 'get_customer_', ['id', 'ip_address', 'user_agent']);
            $ctx = Util::invokeClassMethods(
                [...$order_getters, ...$billing_getters, ...$formatted_getters, ...$customer_getters],
                'get_',
                $order
            );
            $context = array_merge($context, $ctx, $billing_meta, ['order_meta' => $meta]);
        }
        return $context;
    }

}