<?php

namespace RTMCustomMails\Email;

use ReflectionClass;
use RTMCustomMails\Model\EmailSection;
use Timber\Timber;
use WC_Email;

/**
 * @template T
 * @class RTMCustomEmail<T>
 *
 * @property T $object
 */
abstract class RTMCustomEmail extends WC_Email
{

    protected ?string $recipient_type;

    protected array $renderContext = [];

    protected bool $lastResult = false;

    protected string $object_type = 'order';

    protected array $triggerHooks = [];

    public function __construct()
    {
        parent::__construct();

        $this->initTriggerHooks();

        add_filter("wrce_email_render_context_$this->id", [$this, 'prepareContext']);
    }

    public static function asCustomEmail(WC_Email $email): RTMCustomEmail
    {
        if ($email instanceof RTMCustomEmail) {
            return $email;
        }

        return new class($email) extends RTMCustomEmail {
            private WC_Email $proxy;

            public function __construct(WC_Email $proxy)
            {
                $this->proxy = $proxy;

                foreach (get_object_vars($proxy) as $key => $value) {
                    $this->{$key} = $value;
                }
                parent::__construct();
            }

            public function prepareContext($baseContext): array
            {
                foreach ($this->proxy->find as $key => $template) {
                    $this->placeholders[$template] = $this->proxy->replace[$key];
                }

                return parent::prepareContext($baseContext);
            }

            public function trigger(...$args)
            {
                $this->proxy->trigger(...$args);
            }

            public function getTriggerHooks(): array
            {
                return [];
            }
        };
    }

    /**
     * @return mixed
     */
    abstract public function trigger();

    public function prepareContext($baseContext)
    {
        $placeHoldersContext = [];
        foreach ($this->find ?? [] as $key => $template) {
            $placeHoldersContext[substr($template, 1, strlen($template) - 2)] = $this->replace[$key];
        }

        foreach ($this->placeholders ?? [] as $template => $value) {
            $placeHoldersContext[substr($template, 1, strlen($template) - 2)] = $value;
        }

        return $baseContext + $placeHoldersContext + [
            $this->object_type => $this->object,
            'email_heading' => $this->get_heading(),
            'additional_content' => $this->get_additional_content(),
            'sent_to_admin' => true,
            'plain_text' => false,
            'email' => $this,
        ];
    }

    /**
     * @return string[]|array[]
     */
    abstract public function getTriggerHooks(): array;

    public function send($to, $subject, $message, $headers, $attachments)
    {
        do_action('wrce_before_send', $this, $to, $subject, $message, $headers, $attachments);
        $this->lastResult = parent::send($to, $subject, $message, $headers, $attachments);
        Timber::$context_cache = []; // clear Timber context
        return $this->lastResult;
    }

    /**
     * @return bool
     */
    public function getLastResult(): bool
    {
        return $this->lastResult;
    }

    /**
     * @return bool|string
     */
    public function get_content_html()
    {
        global $is_rendering_email;
        $is_rendering_email = true;

        // temporarily reset context
        $currentContext = Timber::$context_cache;
        Timber::$context_cache = [];

        $this->buildRenderContext();

        $renderedEmail = Timber::compile('base-email.twig', $this->renderContext);

        Timber::$context_cache = $currentContext;

        $is_rendering_email = false;

        return $renderedEmail;
    }

    /**
     * Extension for recipient types;
     * @return string
     */
    public function getRecipientType(): string
    {
        return strtolower($this->recipient_type ?? ($this->customer_email ? 'customer' : 'admin'));
    }

    /**
     * Initialize hook on which this email is triggered.
     *
     * This method applies filter `wrce_email_trigger_hooks` with arguments of $this->getTriggerHooks() and the email
     * instance.
     * For each defined hook, if it is a valid string or array with function definitions, call `add_action`.
     *
     * @return void
     * @throws \ReflectionException
     */
    public function initTriggerHooks(): void
    {
        // trigger hooks are filterable and expandable with hook 'wrce_trigger_hooks_{email_id}'
        $hooks = apply_filters('wrce_email_trigger_hooks', $this->getTriggerHooks(), $this);

        foreach ($hooks as $hook) {
            // set default values
            $triggerFunction = 'trigger';
            $priority = 10;

            // when a hook is an array, the array may contain the following keys, or be in order
            // with default values if not set:
            // - 1 or 'function'    => 'trigger'
            // - 2 or 'priority'    => 10
            // - 3 or 'args'        => false
            if (is_array($hook)) {
                if ($f = ($hook[1] ?? $hook['function'] ?? false)) {
                    $triggerFunction = $f;
                }

                if ($p = ($hook[2] ?? $hook['priority'] ?? false)) {
                    $priority = $p;
                }

                if ($a = ($hook[3] ?? $hook['args'] ?? false)) {
                    $args = $a;
                }

                $hook = $hook[0];
            } elseif (!is_string($hook)) {
                // anything other than string or array is skipped
                continue;
            }

            // use reflection to get parameter amount if not set
            $args ??= (new ReflectionClass($this))
                ->getMethod($triggerFunction)
                ->getNumberOfParameters();

            // add the trigger action
            add_action($hook, [$this, $triggerFunction], $priority, $args);

            $this->triggerHooks[] = $hook;
        }
    }

    /**
     * Set the render context
     * @param array $context
     * @return void
     */
    public function setRenderContext(array $context): void
    {
        $this->renderContext = $context;
    }

    /**
     * Get the set context and Timber context;
     * @param array $context
     * @return array
     */
    public function getRenderContext(): array
    {
        return $this->renderContext;
    }

    /**
     * @return string
     */
    public function getObjectType(): string
    {
        return $this->object_type;
    }

    /**
     * @return EmailSection[]
     * @throws \ReflectionException
     */
    public function getSections(): array
    {
        $option = maybe_unserialize($this->get_option('sections', []));
        if (!is_array($option)) {
            return [];
        }

        // ensure correct array indexing
        $option = array_values($option);
        return array_map(fn(array $section) => EmailSection::hydrate(data: $section), $option);
    }

    public function setSections(array $sections): bool
    {
        $sections = array_map(
            fn($section) => $section instanceof EmailSection ? $section->dehydrate() : $section,
            $sections);

        return $this->update_option('sections', $sections);
    }

    public function is_manual()
    {
        return (bool) $this->triggerHooks;
    }

    public function getAttachedDocuments()
    {
        return apply_filters('wrce_attached_documents', [], $this);
    }

    private function buildRenderContext()
    {
        $this->renderContext = [];
        $this->renderContext = apply_filters('timber/context', apply_filters("wrce_email_render_context_{$this->id}", $this->renderContext, $this));
    }

}

