<?php
/*
Plugin Name: User Specific Content
Plugin URI: https://rtmbusiness.nl
Description: This Plugin allows you to select specific users by user name, or by role name who can view a  specific post content or page content.
Version: 2.0
Author: Bainternet, Beau Fiechter (RTM Business)
Author URI: https://rtmbusiness.nl
*/
/*
		* 	Copyright (C) 2014  Ohad Raz
		*	http://en.bainternet.info
		*	admin@bainternet.info

		This program is free software; you can redistribute it and/or modify
		it under the terms of the GNU General Public License as published by
		the Free Software Foundation; either version 2 of the License, or
		(at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program; if not, write to the Free Software
		Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/* Disallow direct access to the plugin file */

if (basename($_SERVER['PHP_SELF']) == basename(__FILE__)) {
    die('Sorry, but you cannot access this page directly.');
}

require __DIR__ . '/vendor/autoload.php';

use RTMBusiness\UserSpecificContent\UserSpecificContentPanel;

class UserSpecificContent
{

    // Class Variables
    /**
     * used as localiztion domain name
     * @var string
     */
    var $localization_domain = "rtm-usc";

    public $options = false;

    /**
     * Class constructor
     */
    public function __construct()
    {
        load_plugin_textdomain($this->localization_domain, false, dirname(plugin_basename(__FILE__)) . '/lang/');
        $this->hooks();
    }

    /**
     * hooks
     * function used to add action and filter hooks
     * Used with `admin_hooks`, `client_hooks`, `and common_hooks`
     * @return void
     */
    public function hooks()
    {
        if (is_admin()) {
            $this->admin_hooks();
        } else {
            $this->client_hooks();
        }
        $this->common_hooks();
    }

    /**
     * common_hooks
     * hooks for both admin and client sides
     * @return void
     */
    public function common_hooks()
    {
        /* add_filter hooks */
        add_action('init', [$this, 'init']);
        /* Save Meta Box */
        add_action('save_post', [$this, 'save_post']);
    }

    /**
     * admin_hooks
     * Admin side hooks should go here
     * @return void
     */
    public function admin_hooks()
    {
        add_action('init', function () {
            $panel = new UserSpecificContentPanel(
                [
                    'title' => __('User Specific Content', $this->localization_domain),
                    'name' => __('User Specific Content', $this->localization_domain),
                    'capability' => 'manage_options',
                    'option' => 'U_S_C',
                    'localization_domain' => $this->localization_domain
                ]
            );

            $panel->register_options_sections();

            $panel->add_help_tab(
                [
                    'id' => 'user_specific_content',
                    'title' => __('User Specific Content', $this->localization_domain),
                    'content' => '<div style="min-height: 350px">
                <h2 style="text-align: center;">' . __('User Specific Content', $this->localization_domain) . '</h2>
                <div>
                		<panel>' . __('If you have any questions or problems head over to', $this->localization_domain) . ' <a href="http://wordpress.org/support/plugin/user-specific-content">' . __('Plugin Support', $this->localization_domain) . '</a></panel>
                        <panel>' . __('If you like my wrok then please ', $this->localization_domain) . '<a class="button button-primary" href="https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=K4MMGF5X3TM5L" target="_blank">' . __('Donate', $this->localization_domain) . '</a>
                </div>
        </div>
        '
                ]
            );
            $panel->add_help_tab(
                [
                    'id' => 'option_panel',
                    'title' => __('Option panel', $this->localization_domain),
                    'content' => '<div style="min-height: 350px">
                <panel>' . __('All of the options here are pretty much self explanatory', $this->localization_domain) . '</panel>
                <ul>
                	<li><strong>Global Blocked message</strong> - ' . __('(If set in a metabox the it overwrites this message for that secific post/page)', $this->localization_domain) . '</li>
					<li><strong>Use with "the_content" hook?</strong> - ' . __('Block content using `the_content` filter hook', $this->localization_domain) . '</li>
					<li><strong>Use with "the_excerpt" hook?</strong> - ' . __('Block content using `the_excerpt` filter hook', $this->localization_domain) . '</li>
					<li><strong>list user names?</strong> - ' . __('If unchecked then the metabox will not show an option to limit by user names.', $this->localization_domain) . '</li>
					<li><strong>User List Type</strong> - ' . __('This option lets you set the field type of the user list in the metabox', $this->localization_domain) . '</li>
					<li><strong>list user roles?</strong> - ' . __('If unchecked then the metabox will not show an option to limit by user roles.', $this->localization_domain) . '</li>
					<li><strong>User Roles List Type</strong> - ' . __('This option lets you set the field type of the user role list in the metabox', $this->localization_domain) . '</li>
					<li><strong>Capability</strong> - ' . __('The capability needed by the user to see the metabox', $this->localization_domain) . '</li>
					<li><strong>POST TYPES</strong> - ' . __('Lets you enable or the metabox on any (public) post type', $this->localization_domain) . '</li>
				</ul>
        </div>
        '
                ]
            );
            $panel->add_help_tab(
                [
                    'id' => 'shortcode',
                    'title' => __('Shortcode', $this->localization_domain),
                    'content' => '<div style="min-height: 350px">
                <panel>' . __('Since version 0.7 you can use a shortcode <pre>[U_O]</pre> which accepts the following parameters:', $this->localization_domain) . '</panel>
                <ul>
                	<li><strong>user_id</strong> - ' . __('specific user ids form more then one separate by comma', $this->localization_domain) . '</li>
					<li><strong>user_name</strong> - ' . __('specific user names form more then one separate by comma', $this->localization_domain) . '</li>
					<li><strong>user_role</strong> - ' . __('specific user role form more then one separate by comma', $this->localization_domain) . '</li>
					<li><strong>blocked_message</strong> - ' . __('specific Content Blocked message', $this->localization_domain) . '</li>
				</ul>
				<panel>eg:</panel><pre>[O_U user_role="Administrator" blocked_message="admins only!"]admin content goes here[/O_U]</pre>
        </div>
        '
                ]
            );
        });

        // add meta box
        add_action('add_meta_boxes', [$this, 'content_box']);

        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
    }

    public function enqueue_scripts()
    {
        $screen = get_current_screen();
        $opts = $this->get_options();
        $enabled_post_types = array_keys(array_filter($opts['posttypes']));

        if ($screen->base == 'post' && in_array($screen->post_type, $enabled_post_types)) {
            wp_enqueue_style('rtm-usc-admin', plugin_dir_url(__FILE__) . 'assets/css/style.css');
            wp_enqueue_script('rtm-usc-admin-select', plugin_dir_url(__FILE__) . 'assets/js/usc-select.js');
        }
    }

    /**
     * client_hooks
     * client side hooks should go here
     * @return void
     */
    public function client_hooks()
    {
    }

    public function init()
    {
        do_action('User_specific_content_filter_add', $this);
    }

    /**
     * Load or initialize the options for the plugin.
     *
     * Use $force_reload = true to force a reload of the settings, even with the `options` property has been set.
     *
     * @param bool $force_reload
     * @return array|bool
     */
    public function get_options(bool $force_reload = false)
    {
        if ($this->options && !$force_reload) {
            return $this->options;
        }

        $post_types = array_reduce(get_post_types(), function ($carry, $type) {
            $carry[$type] = false;
            return $carry;
        }, []);

        $default = array(
            'b_message' => '',
            'posttypes' => $post_types,
            'capability' => 'manage_options',
        );

        $opts = get_option('U_S_C');
        if (!empty($opts)) {
            foreach ($opts as $key => $value) {
                $default[$key] = $value;
            }
        }

        update_option('U_S_C', $default);
        $this->options = $default;
        return $default;
    }


    /**
     * Define the metabox content in edit pages.
     * @param $post_type
     */
    public function content_box($post_type)
    {
        $options = $this->get_options();
        if (!current_user_can($options['capability'])) {
            return;
        }

        $allowed_types = array();
        foreach ((array)$options['posttypes'] as $key => $value) {
            if ($value) {
                $allowed_types[] = $key;
            }
        }

        //added a filter to enable controlling the allowed post types by filter hook
        $allowed_types = apply_filters('USC_allowed_post_types', $allowed_types);

        if (in_array($post_type, (array)$allowed_types)) {
            add_meta_box(
                'User_specific_content',
                __('User specific content box'),
                [$this, 'render_metabox_content'],
                $post_type
            );
        }

        //allow custom types by action hook
        do_action('USC_add_meta_box', $this);
    }

    /**
     * Render metabox content in edit pages.
     */
    public function render_metabox_content()
    {
        global $post;

        $options = $this->get_options();
        $savedusers = get_post_meta($post->ID, 'U_S_C_users', true);
        $savedoptions = get_post_meta($post->ID, 'U_S_C_options', true);

        // Use nonce for verification
        wp_nonce_field(plugin_basename(__FILE__), 'User_specific_content_box_inner');

        echo __('Select users to show this content to: ', $this->localization_domain);

        //by user
        if ($options['list_users']) {
            echo '<h4>' . __('By User Name:', $this->localization_domain) . '</h4><p>';
            $site_id = 1;
            if (is_multisite()) {
                $site_id = get_current_blog_id();
            }

            $blogusers = get_users([
                'blog_id' => $site_id,
                'orderby' => 'nicename'
            ]);

            if (empty($savedusers)) {
                $savedusers = [];
            }

            echo '
                    <label for="usc-select-users" class="usc-clear-selection">
                     <a href="#" class="button">Clear selection</a>
                    </label>
                    <select id="usc-select-users" class="usc-select" name="U_S_C_users[]" multiple="multiple">';
            foreach ($blogusers as $user) {
                echo '<option ';
                if (in_array($user->ID, (array)$savedusers)) {
                    echo ' selected="selected" ';
                }
                echo 'value="' . $user->ID . '">' . $user->display_name . '</option>';
            }

            echo '</select></p>';
        }
        echo '<h4>' . __('Members and Guests', $this->localization_domain) . '</h4>';
        echo '<p><label><input type="checkbox" name="U_S_C_options[logged]" value="1"';
        if (isset($savedoptions['logged']) && $savedoptions['logged'] == 1) {
            echo ' checked';
        }
        echo '>' . __('Logged in users only', $this->localization_domain) . '</label><br/><span class="description">'
            . __('If this box is checked then content will show only to logged-in users and everyone else will get the blocked message', $this->localization_domain);
        echo '</span></p>';

        //none logged-in
        echo '<hp><label>
		<input type="checkbox" name="U_S_C_options[non_logged]" value="1"';
        if (isset($savedoptions['non_logged']) && $savedoptions['non_logged'] == 1) {
            echo ' checked';
        }
        echo '>' . __('Guest users only', $this->localization_domain) . '</label><br/><span class="description">' .
            __('If this box is checked then content will show only to non-logged in visitors and everyone else will get the blocked message', $this->localization_domain);
    }

    /**
     * Save metabox data with save_post hook.
     *
     * @param $post_id
     * @return mixed
     */
    public function save_post($post_id)
    {
        global $post;

        // check if out form key is present and nonce is valid
        if (!isset($_POST['User_specific_content_box_inner']) || !wp_verify_nonce($_POST['User_specific_content_box_inner'], plugin_basename(__FILE__))) {
            return $post_id;
        }

        // dont autosave
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return $post_id;
        }

        // OK, we're authenticated: we need to find and save the data
        $saved_users = get_post_meta($post_id, 'U_S_C_users', true);
        $saved_options = get_post_meta($post->ID, 'U_S_C_options', true);

        $this->update_or_delete_meta('U_S_C_options', $post_id, $saved_options);
        $this->update_or_delete_meta('U_S_C_users', $post_id, $saved_users);

        if (isset($_POST['U_S_C_message'])) {
            update_post_meta($post_id, 'U_S_C_message', $_POST['U_S_C_message']);
        }

        return $post_id;
    }

    /**
     * Update post meta if the $_POST contains the given option name, or delete the old options if it was not set.
     *
     * @param $option_name
     * @param $post_id
     * @param $old_options
     */
    public function update_or_delete_meta($option_name, $post_id, $old_options)
    {
        if (isset($_POST[$option_name]) && !empty($_POST[$option_name])) {
            $new_options = [];
            foreach ($_POST[$option_name] as $key => $value) {
                $new_options[$key] = $value;
            }
            update_post_meta($post_id, $option_name, $new_options);
        } else {
            if (!empty($old_options)) {
                delete_post_meta($post_id, $option_name);
            }
        }
    }


    public function User_specific_content_filter($content)
    {
        global $post;
        if (isset($post->ID)) {
            $post_options = get_post_meta($post->ID, 'U_S_C_options', true);
            $post_message = get_post_meta($post->ID, 'U_S_C_message', true);
            $post_users = get_post_meta($post->ID, 'U_S_C_users', true);
        } else {
            $post_options = [];
            $post_message = '';
            $post_users = [];
        }

        if (isset($post_options) && !empty($post_options)) {
            // none logged only
            if (isset($post_options['non_logged']) && $post_options['non_logged'] == 1) {
                if (is_user_logged_in()) {
                    return $this->displayMessage($post_message);
                }
            }
            //logged in users only
            if (isset($post_options['logged']) && $post_options['logged'] == 1) {
                if (!is_user_logged_in()) {
                    return $this->displayMessage($post_message);
                }
            }
        }

        $run_check = 0;
        if (empty($post_users)) {
            return $content;
        }
        $current_user = wp_get_current_user();

        if (in_array($current_user->ID, $post_users)) {
            return $content;
        }

        return $this->displayMessage($post_message);
    }

    public function displayMessage($m)
    {
        global $post;
        if (isset($m) && $m != '') {
            return apply_filters('user_specific_content_blocked', $m, $post);
        } else {
            $options = $this->get_options();
            return apply_filters('user_specific_content_blocked', $options['b_massage'], $post);
        }
    }
}//end class

add_action('init', 'init_user_specific_content_plugin', 0);
function init_user_specific_content_plugin()
{
    global $user_specific_content;
    $user_specific_content = new UserSpecificContent();
}

/**
 * Deny access to a page or post if the currently logged in user is not in USC_users array.
 */
function rtm_deny_access_if_post_is_protected()
{
    // only for public pages
    if (!is_admin()) {
        // get the current post/page
        $id = get_the_ID();

        // check whether the U_S_C_users meta key is available
        $users = get_post_meta($id, 'U_S_C_users', true);

        if (empty($users) // post is not protected for specific users
            || in_array(get_current_user_id(), $users) // user is in allowed user array
            || current_user_can('administrator') // user is admin
        ) {
            // exit if the user is valid
            return;
        }

        // throw 404 if user has no access to the post
        global $wp_query;
        $wp_query->set_404();
        status_header(404);
    }
}

add_action('wp', 'rtm_deny_access_if_post_is_protected');

/**
 * Filter protected posts from 'Latest Posts' widget if the current user does not have access
 * @param $args
 * @return mixed
 */
function rtm_filter_latest_posts_widget($args)
{
    $uid = get_current_user_id();
    $strlen = strlen(strval($uid));

    if (!is_admin() && !current_user_can('administrator')) {
        $args['meta_query'] = [
            'relation' => 'AND',
            [
                'relation' => 'OR',
                [
                    'key' => 'U_S_C_users',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key' => 'U_S_C_users',
                    'compare' => 'LIKE',
                    'value' => 's:' . $strlen . ':"' . $uid . '"'
                ],
            ]
        ];
    }

    return $args;
}

add_action('widget_posts_args', 'rtm_filter_latest_posts_widget');

/**
 * Filter protected products from the product catalog.
 * @param $mq
 * @return array
 */
function rtm_wc_product_meta_query($mq)
{
    if (!is_admin() && !current_user_can('administrator')) {
        $uid = get_current_user_id();
        $strlen = strlen(strval($uid));

        if (empty($mq)) {
            return [
                'relation' => 'AND',
                [
                    'relation' => 'OR',
                    [
                        'key' => 'U_S_C_users',
                        'compare' => 'NOT EXISTS',
                    ],
                    [
                        'key' => 'U_S_C_users',
                        'compare' => 'LIKE',
                        'value' => 's:' . $strlen . ':"' . $uid . '"'
                    ],
                ]
            ];
        }

        return array_merge($mq, [
            'relation' => 'AND',
            [
                'relation' => 'OR',
                [
                    'key' => 'U_S_C_users',
                    'compare' => 'NOT EXISTS',
                ],
                [
                    'key' => 'U_S_C_users',
                    'compare' => 'LIKE',
                    'value' => 's:' . $strlen . ':"' . $uid . '"'
                ],
            ]
        ]);
    }
    return $mq;
}

add_filter('woocommerce_product_query_meta_query', 'rtm_wc_product_meta_query');

/**
 * Add 'Users' column to the products admin table.
 *
 * @param $columns
 * @return mixed
 */
function woocommerce_add_products_table_users_column($columns)
{
    $columns['U_S_C_users'] = __('Users');
    return $columns;
}

add_filter('manage_product_posts_columns', 'woocommerce_add_products_table_users_column');

/**
 *
 *
 * @param $column
 * @param $post_id
 */
function woocommerce_fill_products_table_users_column($column, $post_id)
{
    if ($column === 'U_S_C_users') {
        $user_ids = get_post_meta($post_id, 'U_S_C_users', true);
        if (empty($user_ids)) {
            echo __('Everyone');
            return;
        }

        /** @var WP_User[] $users */
        $users = get_users([
            'include' => $user_ids
        ]);

        foreach ($users as $user) {
            echo '<p><a href="' . $user->user_url . '">' . $user->display_name . '</a></p>';
        }
    }
}

add_filter('manage_product_posts_custom_column', 'woocommerce_fill_products_table_users_column', 10, 2);
