<?php

namespace RtmMail;

/**
 * Core - handles all plugin initialization
 */
class Core
{
    private static $pages;

    private $catcher;
    private $listener;

    /**
     * __construct - methods that need to be executed on object creation
     */
    public function __construct()
    {
        $settings = get_option('rtm_mail_settings');
        $mail_capability = $settings['mail_capability'] ?? 'manage_options';
        // Set dashboard variable to prevent 2 dashboard pages
        $dashboard = new Dashboard();

        // Set plugin pages
        self::$pages = [
            'main' => [ // Main page (Dashboard)
                'page' => $dashboard,
                'page_title' => __('WP Mail Logger - Dashboard', 'rtm-mail'),
                'menu_title' => 'WP Mail Logger',
                'capability' => $mail_capability,
                'menu_slug' => 'rtm-mail-dashboard',
                'icon_url' => 'dashicons-email-alt',
            ],
            'dashboard' => [ // Sub menu page (Dashboard)
                'page' => $dashboard,
                'parent_slug' => 'rtm-mail-dashboard',
                'page_title' => __('WP Mail Logger - Dashboard', 'rtm-mail'),
                'menu_title' => __('Dashboard', 'rtm-mail'),
                'capability' => $mail_capability,
                'menu_slug' => 'rtm-mail-dashboard',
            ],
            'overview' => [ // Sub menu page (Overview)
                'page' => new Overview(),
                'parent_slug' => 'rtm-mail-dashboard',
                'page_title' => __('WP Mail Logger - Overview', 'rtm-mail'),
                'menu_title' => __('Overview', 'rtm-mail'),
                'capability' => $mail_capability,
                'menu_slug' => 'rtm-mail-overview',
            ],
            'settings' => [ // Sub menu page (Settings)
                'page' => new Settings(),
                'parent_slug' => 'rtm-mail-dashboard',
                'page_title' => __('WP Mail Logger - Settings', 'rtm-mail'),
                'menu_title' => __('Settings', 'rtm-mail'),
                'capability' => 'manage_options',
                'menu_slug' => 'rtm-mail-settings',
            ],
            'mailsmtp' => [ // Sub menu page (Settings)
                'page' => new MailSMTP(),
                'parent_slug' => 'rtm-mail-dashboard',
                'page_title' => __('WP Mail Logger - Mail SMTP', 'rtm-mail'),
                'menu_title' => __('Mail SMTP', 'rtm-mail'),
                'capability' => 'manage_options',
                'menu_slug' => 'rtm-mail-mailsmtp',
            ],
            'events' => [ // Sub menu page (Debug Events)
                'page' => new Events(),
                'parent_slug' => 'rtm-mail-dashboard',
                'page_title' => __('WP Mail Logger - Debug Events', 'rtm-mail'),
                'menu_title' => __('Debug Events', 'rtm-mail'),
                'capability' => 'manage_options',
                'menu_slug' => 'rtm-mail-events',
                'is_pro' => true,
            ],
            'details' => [ // Hidden sub menu page (Log details)
                'page' => new Details(),
                'parent_slug' => '',
                'page_title' => __('WP Mail Logger - Details', 'rtm-mail'),
                'menu_title' => __('Details', 'rtm-mail'),
                'capability' => $mail_capability,
                'menu_slug' => 'rtm-mail-details',
            ],
        ];

        // Instantiate the Catcher
        $this->catcher = new Catcher();

        // Instantiate error listener
        $this->listener = new EventListener();

        // Set custom Wonolog channel
        add_filter('wonolog.channels', function ($channels) {
            $channels[] = EventListener::CHANNEL_NAME;
            return $channels;
        });
        // Change wonolog folder
        add_filter('wonolog.default-handler-folder', function ($folder) {
            $upload = wp_upload_dir();
            return $upload['basedir'] . '/rtm-mail/logs';
        });

        // Apply all filters
        $this->add_filters();
        // Add all actions for initialization
        $this->add_actions();
    }

    /**
     * add_filters - all filters that need to be executed
     */
    private function add_filters()
    {
        // add the highest priority, so it's the last executed method
        add_filter('wp_mail', [$this->catcher, 'catch_mail'], 9999999);
        // Admin footer text
        add_filter('admin_footer_text', [$this, 'display_footer_text']);

        add_filter( 'plugin_action_links_' . plugin_basename(RTM_MAIL_PLUGIN_FILE), [$this, 'plugin_actions'], 10, 1);
    }

    /**
     * add_actions - all actions that need to be executed
     */
    private function add_actions()
    {
        add_action('admin_init', [$this, 'init']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_menu', [$this, 'add_pages']);
        add_action('plugins_loaded', [$this, 'check_migrations']);
        add_action('admin_footer', [$this, 'display_footer']);
        add_action('wp_mail_failed', [$this->catcher, 'mail_error'], 9999999);
        // Event listener hooks
        add_action('rtmmail_migration_failed', [$this->listener, 'migration_failed'], 10, 2);
        add_action('rtmmail_migration_success', [$this->listener, 'migration_success']);
        add_action('rtmmail_send_failed', [$this->listener, 'send_failed'], 10, 2);
        add_action('rtmmail_send_success', [$this->listener, 'send_success']);
        add_action('rtmmail_log_deleted', [$this->listener, 'log_deleted'], 10, 2);
        add_action('rtmmail_mail_caught', [$this->listener, 'mail_caught']);
    }

    /**** Hook Methods ****/

    /**
     * init - initialize plugin
     */
    public function init()
    {
        load_plugin_textdomain('rtm-mail', false, RTM_MAIL_PLUGIN_PATH . '/assets/languages');
    }

    /**
     * activate - callback for activation hook
     */
    public function activate()
    {
        if (class_exists('WPMailSMTP\Core')) {
            add_action('admin_notices', function () {
                echo '<div class="notice notice-warning"> ';
                echo '<p><strong>WP Mail Logger: </strong>';
                echo __('You are currently using <code>WP-Mail-SMTP</code> which may conflict with the sender options in this plugin.', 'rtm-mail');
                echo '</p>';
                echo '</div>';
            });
        }
        update_option('rtm_mail_settings', [
            'mail_capability' => 'manage_options',
        ]);

        update_option('rtm_mail_smtp_settings', [
            'smtp_enabled' => false,
            'host' => '',
            'encryption' => 'none',
            'port' => 0,
            'authentication' => true,
            'username' => '',
            'password' => '',
        ]);
    }

    /**
     * deactivate - callback for deactivation hook
     */
    public function deactivate()
    {
        delete_option('rtm_mail_settings');
        wp_clear_scheduled_hook('rtm_mail_send_caught_logs');
        wp_clear_scheduled_hook('rtm_mail_send_log');
    }

    /**
     * check_migrations - Check for update changes
     */
    public function check_migrations()
    {
        (new Migration(RTM_MAIL_VERSION))->check();
    }

    /**
     * add_pages - add the pages to the administrator menu
     */
    public function add_pages()
    {
        foreach (self::$pages as $page => $page_data) {
            if ($page === 'main') {
                add_menu_page($page_data['page_title'], $page_data['menu_title'], $page_data['capability'], $page_data['menu_slug'], [$page_data['page'], 'display'], $page_data['icon_url']);
            } else {
                add_submenu_page($page_data['parent_slug'], $page_data['page_title'], $page_data['menu_title'], $page_data['capability'], $page_data['menu_slug'], [$page_data['page'], 'display']);
            }
        }
    }

    /**
     * display_footer - add RTM Business logo to footer
     */
    public function display_footer()
    {
        // Check if user is on one of the rtm-mail pages
        if ($this->page_check()) {
            echo '<div class="rtm-mail-footer">';
            echo '<div class="rtm-mail-logo"><a href="https://www.rtmbusiness.nl/?utm_source=rtm-mail-plugin&utm_medium=footer&utm_campaign=mail-plugin" target="_blank"><img src="' . RTM_MAIL_PLUGIN_PATH . 'assets/images/rtm-logo.png" alt="rtm-logo" /></a></div>';
            echo '</div>';
        }
    }

    /**
     * display_footer_text - change the admin footer text
     */
    public function display_footer_text($text)
    {
        // Check if user is on one of the rtm-mail pages
        if ($this->page_check()) {
            $text = sprintf(__('Thanks for using the <span style="font-weight: 600;">WP Mail Logger Plugin</span>. Consider upgrading to <a href="%s" style="font-weight: 600">PRO</a> to get more features!', 'rtm-mail'), RTM_MAIL_PRO_SITE);
        }
        return $text;
    }

    /**
     * plugin_actions - add action list
     */
    public function plugin_actions($links)
    {
        $links['pro'] = sprintf(__('<a href="%s" style="font-weight: bold">Upgrade to PRO</a> ', 'rtm-mail'), RTM_MAIL_PRO_SITE);
        return $links;
    }

    /**
     * enqueue_scripts - enqueues the scripts needed for the plugin
     */
    public function enqueue_scripts()
    {
        // Only load scripts on WP Mail Logger pages
        if ($this->page_check()) {
            // Load fontawesome
            wp_enqueue_style('fontawesome', 'https://use.fontawesome.com/releases/v5.7.0/css/all.css');
            // Load CodeMirror
            wp_enqueue_script('rtm-mail-codemirror-js', 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.63.3/codemirror.min.js', [], false, false);
            wp_enqueue_style('rtm-mail-codemirror-css', 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.63.3/codemirror.min.css');
            wp_enqueue_style('rtm-mail-codemirror-theme-css', 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.63.3/theme/monokai.min.css');
            wp_enqueue_script('rtm-mail-codemirror-html-js', 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.63.3/mode/xml/xml.min.js', [], false, false);
            wp_enqueue_script('rtm-mail-codemirror-closetags-js', 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.63.3/addon/edit/closetag.min.js', [], false, false);

            // Load custom styling css
            wp_enqueue_style('rtm-mail', RTM_MAIL_PLUGIN_PATH . 'assets/css/style.css');
            wp_register_script('rtm-mail', RTM_MAIL_PLUGIN_PATH . 'assets/js/script.js');
            wp_localize_script('rtm-mail', 'rtm_mail_translation', [
                'type_address' => __('Type an address', 'rtm-mail'),
                'status_sent' => __('Sent', 'rtm-mail'),
            ]);
            wp_enqueue_script('rtm-mail');
            // Load flatpickr datetime picker scripts
            wp_enqueue_style('rtm-mail-flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css');
            wp_enqueue_script('rtm-mail-flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr', ['jquery'], false, true);
        }
    }

    /**
     * page_check - Check if the user is on one of the plugins pages
     * @return bool
     */
    private function page_check()
    {
        // Loop through every available page
        foreach (self::$pages as $page_data) {
            // Return true if user is on one of the pages
            if ($page_data['page']->page_check()) {
                return true;
            }
        }
        // Return false when user is on none of the pages
        return false;
    }

    /**
     * render_page_header - renders the page header for every page
     */
    public static function render_page_header($current_page)
    {
?>
        <div class="navbar">
            <div class="navbar-item">
                <div class="nav-logo">
                    <img src="<?php echo RTM_MAIL_PLUGIN_PATH; ?>assets/images/logo-icon.png" alt="wp-mail-logger logo"
                         class="logo-icon"/>
                    <img src="<?php echo RTM_MAIL_PLUGIN_PATH; ?>assets/images/logo.png" alt="wp-mail-logger logo"
                         class="logo"/>
                    <span class="nav-version">v<?php echo RTM_MAIL_VERSION; ?></span>
                </div>
                <a href="#" class="nav-link mobile-link-toggle"><i class="fas fa-bars"></i></a>
                <div class="nav-menu-links">
                    <?php foreach (self::$pages as $page => $page_data) { ?>
                        <?php if (isset($page_data['parent_slug']) && !empty($page_data['parent_slug'])) { ?>
                            <?php if (current_user_can($page_data['capability'])) { ?>
                                <?php if (!isset($page_data['is_pro']) || !$page_data['is_pro']) { ?>
                                    <a href="<?php echo get_admin_url(); ?>admin.php?page=<?php echo $page_data['menu_slug']; ?>" class="nav-link <?php echo ($page === $current_page) ? 'nav-selected' : ''; ?>"><?php echo $page_data['menu_title']; ?></a>
                                <?php } else { ?>
                                    <a href="<?php echo get_admin_url(); ?>admin.php?page=<?php echo $page_data['menu_slug']; ?>" class="nav-link <?php echo ($page === $current_page) ? 'nav-selected' : ''; ?>"><?php echo $page_data['menu_title']; ?> <span class="badge badge-pro" style="margin-left: 5px;">PRO</span></a>
                                <?php } ?>
                            <?php } ?>
                        <?php } ?>
                    <?php } ?>
                </div>
            </div>
            <div class="navbar-item navbar-button" style="margin-right: 20px;">
                <a href="<?php echo RTM_MAIL_PRO_SITE; ?>" target="_blank" class="button button-pro"><?php echo __('Upgrade to PRO'); ?></a>
            </div>
        </div>
        <div class="navbar-mobile-container">
            <div class="navbar-mobile">
                <?php foreach (self::$pages as $page => $page_data) { ?>
                    <?php if (isset($page_data['parent_slug']) && !empty($page_data['parent_slug'])) { ?>
                        <?php if (current_user_can($page_data['capability'])) { ?>
                            <?php if (!isset($page_data['is_pro']) || !$page_data['is_pro']) { ?>
                                <a href="<?php echo get_admin_url(); ?>admin.php?page=<?php echo $page_data['menu_slug']; ?>"
                                   class="nav-link-mobile <?php echo ($page === $current_page) ? 'nav-mobile-selected' : ''; ?>"><?php echo $page_data['menu_title']; ?></a>
                            <?php } else { ?>
                                <a href="<?php echo get_admin_url(); ?>admin.php?page=<?php echo $page_data['menu_slug']; ?>" class="nav-link-mobile <?php echo ($page === $current_page) ? 'nav-mobile-selected' : ''; ?>"><?php echo $page_data['menu_title']; ?>
                                    <span class="badge badge-pro" style="margin-left: 5px;">PRO</span></a>
                            <?php } ?>
                        <?php } ?>
                    <?php } ?>
                <?php } ?>
            </div>
        </div>
<?php
    }
}
