<?php

namespace RtmMail;

use PHPMailer\PHPMailer\PHPMailer;
use RtmMail\Helpers\LogHelper;

/**
 * Catcher - handles every incoming mail from the wp_mail hook
 */
class Catcher
{
    /**
     * catch_email - wp_mail hook callback to log all emails
     * @param $args
     * @return array (phpmailer)
     */
    public function catch_mail($args)
    {
        // if there is no Log_id in the headers it has to be logged
        if (empty($args['headers']) || substr($args["headers"][0], 0, 2) !== 'Id') {
            // Convert "to" to an array when it's a string
            if (is_string($args['to'])) {
                // Replace commas in the strings with semicolons
                $args['to'] = str_replace(',', ';', $args['to']);
                // Set every email in a new recipients array
                $recipients = [];
                foreach (explode(';', $args['to']) as $recipient) {
                    $recipients[] = str_replace(' ', '', $recipient);
                }
                // Set the recipients array to the new "to" arg
                $args['to'] = $recipients;
            }
            $args['headers'] = $args['headers'] ?? [];
            $args['headers'] = is_string($args['headers']) ? explode("\n", str_replace("\\r\\n", "\n", $args['headers'])) : $args['headers'];
            $args['backtrace'] = $this->get_backtrace();

            // format the mail data
            $mail_data = $this->format($args);

            // Save the mail data as new log item
            $log_id = LogHelper::save($mail_data);

            // Log caught mail
            do_action('rtmmail_mail_caught', $log_id);

            $smtp_settings = get_option('rtm_mail_smtp_settings');
            $smtp_enabled = isset($smtp_settings['smtp_enabled']) && filter_var($smtp_settings['smtp_enabled'], FILTER_VALIDATE_BOOLEAN);

            if ($smtp_enabled) {
                self::send_smtp($log_id, $mail_data);
            } else {
                LogHelper::update($log_id, ['status' => 'sent', 'sent' => date('Y-m-d H:i:s', time())]);
                // Set the Id of the mail log in header
                array_unshift($args['headers'], "Id: " . $log_id);
            }
        }

        return $args;
    }

    /**
     * mail_error - handles error on mail failure
     * @param $error
     */
    public function mail_error($error)
    {
        // Check if there are headers available, so you can retrieve the Id
        if (!empty($error->error_data['wp_mail_failed']['headers'])) {
            $log_id = (int)$error->error_data['wp_mail_failed']['headers']['Id'];
            LogHelper::update($log_id, ['status' => 'failed', 'sent' => null]);
            do_action('rtmmail_send_failed', $log_id, $error->errors['wp_mail_failed'][0]);
        }
    }

    /**
     * send_mail - sends mail based on saved log
     * @param $log_id
     * @return string|void error message or 'success'
     */
    public static function send_mail($log_id)
    {
        $mail_log = LogHelper::get([
            'post_per_page' => null,
            'where' => [
                'id' => [
                    'type' => '=',
                    'value' => $log_id,
                ]
            ]
        ]);

        $mail_log = $mail_log[0] ?? null;

        // Return if the log doesn't exist anymore
        if ($mail_log == null) {
            return;
        }

        $smtp_settings = get_option('rtm_mail_smtp_settings');
        $smtp_enabled = isset($smtp_settings['smtp_enabled']) && filter_var($smtp_settings['smtp_enabled'], FILTER_VALIDATE_BOOLEAN);
        if ($smtp_enabled) {
            return self::send_smtp($log_id, $mail_log);
        } else {
            $mail_data = [];
            // set the subject
            $mail_data['subject'] = $mail_log['subject'];
            // set the receivers
            $mail_data['to'] = $mail_log['receiver'];
            // set the body
            $mail_data['message'] = $mail_log['body'];
            // set the headers
            $mail_data['headers'] = [];

            // Set the cc
            if (!empty($mail_log['cc'])) {
                foreach ($mail_log['cc'] as $cc_address) {
                    if (!empty($cc_address)) {
                        $mail_data['headers'][] = 'Cc: ' . $cc_address;
                    }
                }
            }
            // Set the bcc
            if (!empty($mail_log['bcc'])) {
                foreach ($mail_log['bcc'] as $bcc_address) {
                    if (!empty($bcc_address)) {
                        $mail_data['headers'][] = 'Bcc: ' . $bcc_address;
                    }
                }
            }

            // Set the remaining headers
            if (!empty($mail_log['headers'])) {
                foreach ($mail_log['headers'] as $header_key => $header_value) {
                    if (!empty($header_value)) {
                        $mail_data['headers'][] = ucfirst($header_key) . ': ' . $header_value;
                    }
                }
            }

            // Set the id in headers
            array_unshift($mail_data['headers'], "Id: " . $log_id);
            // set the attachments
            $mail_data['attachments'] = [];
            if (!empty($mail_log['attachments'])) {
                foreach ($mail_log['attachments'] as $attachment) {
                    if (!empty($attachment)) {
                        $mail_data['attachments'][] = $attachment;
                    }
                }
            }
            $success = wp_mail($mail_data['to'], $mail_data['subject'], $mail_data['message'], $mail_data['headers'], $mail_data['attachments']);
            // Check if sending was successful
            if ($success) {
                // Set status on sent and redirect to the log details page
                LogHelper::update($log_id, ['status' => 'sent', 'sent' => date('Y-m-d H:i:s', time())]);
                do_action('rtmmail_send_success', $log_id);
                return 'success';

            } else {
                // Set the status to failed and redirect to the log details page
                LogHelper::update($log_id, ['status' => 'failed', 'sent' => null]);
                return __('Failed sending email, check the debug logs for more info!', 'rtm-mail');
            }
        }
    }

    private static function send_smtp($log_id, $mail_data)
    {
        $settings = get_option('rtm_mail_settings');
        $sender_options = $settings['sender_options'] ?? null;

        $sender_mail = ($sender_options != null && !empty($sender_options['address'])) ? $sender_options['address'] : get_option('admin_email');
        $sender_title = ($sender_options != null && !empty($sender_options['title'])) ? $sender_options['title'] : get_bloginfo('name');

        $encryption_key = get_option('rtm_mail_smtp_key', false);
        $smtp_settings = get_option('rtm_mail_smtp_settings');
        $smtp_enabled = $smtp_settings['smtp_enabled'] ?? false;

        if (!$smtp_enabled) {
            return __('SMTP is not enabled');
        }

        $host = $smtp_settings['host'] ?? '';
        $encryption = $smtp_settings['encryption'] ?? 'none';
        $port = $smtp_settings['port'] ?? 0;
        $authentication = $smtp_settings['authentication'] ?? true;
        $username = $smtp_settings['username'] ?? '';
        $password = isset($smtp_settings['password']) ? Cryptor::Decrypt($smtp_settings['password'], $encryption_key) : '';

        // Include the PHPMailer class
        require_once ABSPATH . WPINC . '/PHPMailer/PHPMailer.php';
        require_once ABSPATH . WPINC . '/PHPMailer/SMTP.php';
        require_once ABSPATH . WPINC . '/PHPMailer/Exception.php';
        $mail = new PHPMailer(true);
        try {
            $mail->CharSet = get_bloginfo('charset');
            $mail->IsSMTP();

            // Set content type
            $mail->ContentType = $mail_data['headers']['content-type'];
            $mail->IsHTML($mail_data['headers']['content-type'] === 'text/html');

            // Set SMTP Host and Port
            $mail->Host = $host;
            $mail->Port = $port;

            // Set encryption type
            if ($encryption !== 'none') {
                $mail->SMTPSecure = $encryption;
            }

            // Check and set authentication
            if ($authentication) {
                $mail->SMTPAuth = true;
                $mail->Username = $username;
                $mail->Password = $password;
            }

            // Set from address
            $mail->SetFrom($sender_mail, $sender_title);
            // Set all to addresses
            foreach ($mail_data['receiver'] as $to_address) {
                if (!empty($to_address)) {
                    $mail->AddAddress($to_address);
                }
            }
            // Set all cc addresses
            foreach($mail_data['cc'] as $cc_address) {
                if (!empty($cc_address)) {
                    $mail->AddCC($cc_address);
                }
            }
            // Set all bcc addresses
            foreach($mail_data['bcc'] as $bcc_address) {
                if (!empty($bcc_address)) {
                    $mail->AddBcc($bcc_address);
                }
            }

            // Set body and subject
            $mail->Subject = $mail_data['subject'];
            $mail->Body = $mail_data['body'];

            // Set attachments
            foreach ($mail_data['attachments'] as $attachment) {
                $mail->addAttachment($attachment, basename($attachment));
            }

            // Set timeout
            $mail->Timeout = 10;

            $mail->Send();

            LogHelper::update($log_id, ['status' => 'sent', 'sent' => date('Y-m-d H:i:s', time())]);
            do_action('rtmmail_send_success', $log_id);

        } catch(\Exception $ex) {
            // Set the status to failed and redirect to the log details page
            LogHelper::update($log_id, ['status' => 'failed', 'sent' => null]);
            do_action('rtmmail_send_failed', $log_id, $mail->ErrorInfo);
            return $mail->ErrorInfo;
        }

        return 'success';
    }

    /**
     * format - formats the mail data to save as a log item
     * @param $args
     * @return array (formatted mail data)
     */
    private function format($args)
    {
        // Return value array
        $formatted_data = [];
        // Set default sender options if it exists otherwise set the wordpress admin mail
        $formatted_data['sender'] = get_option('admin_email');
        $sender_title = get_bloginfo('name');
        $formatted_data['receiver'] = [];
        $formatted_data['cc'] = [];
        $formatted_data['bcc'] = [];
        $formatted_data['subject'] = $args['subject'];
        $formatted_data['body'] = stripslashes(htmlspecialchars_decode($args['message']));
        $formatted_data['attachments'] = [];
        $formatted_data['headers'] = [];
        $formatted_data['backtrace'] = $args['backtrace'] ?? [];
        $formatted_data['created'] = date('Y-m-d H:i:s', time());

        $args['to'] = $args['to'] ?? [];
        $args['headers'] = $args['headers'] ?? [];
        $args['attachments'] = $args['attachments'] ?? [];

        // Check for every 'to' if it's not empty
        foreach ($args['to'] as $receiver_mail) {
            if (!empty($receiver_mail)) {
                $formatted_data['receiver'][] = $receiver_mail;
            }
        }

        // Check for every header if the email is not empty and add it to the formatted data
        foreach ($args['headers'] as $header) {
            if (strpos(strtolower($header), 'from:') !== false) {
                // Remove header key and trim quotes
                $from_email = trim(str_replace('from: ', '', strtolower($header)), '\'"');
                if (!empty($from_email)) {
                    $formatted_data['sender'] = $from_email;
                }
            } else if (strpos(strtolower($header), 'bcc:') !== false) {
                // Remove header key and trim quotes
                $bcc_email = trim(str_replace('bcc: ', '', strtolower($header)), '\'"');
                if (!empty($bcc_email)) {
                    $formatted_data['bcc'][] = $bcc_email;
                }
            } else if (strpos(strtolower($header), 'cc:') !== false) {
                // Remove header key and trim quotes
                $cc_email = trim(str_replace('cc: ', '', strtolower($header)), '\'"');
                if (!empty($cc_email)) {
                    $formatted_data['cc'][] = $cc_email;
                }
            } else {
                $header_data = explode(':', str_replace(' ', '', strtolower($header)));
                $formatted_data['headers'][$header_data[0]] = $header_data[1] ?? '';
            }
        }
        // Set the default from header
        $formatted_data['headers']['from'] = $sender_title . ' <' . $formatted_data['sender'] . '>';

        // Set attachments
        $upload = wp_upload_dir();
        $upload_dir = $upload['basedir'] . '/rtm-mail/attachments';
        // Create new dir directory if it doesn't exist
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755);
        }
        // Loop through every attachment and add it to the formatted data
        foreach ($args['attachments'] as $attachment) {
            $date = date('Y_m_d H_i_s', time());
            $file_name = substr($attachment, strrpos($attachment, '/'));
            $extension = explode('.', $file_name)[1];
            // new file directory and name based on date so its unique (rtm-mail/attachments/filename-YYYY_mm_dd_hh_mm_ss.ext)
            $new_file = $upload_dir . str_replace('.' . $extension, '', $file_name) . '-' . str_replace(' ', '', $date) . '.' . $extension;
            if (copy($attachment, $new_file)) {
                $formatted_data['attachments'][] = $new_file;
            } else {
                printf(__('WP Mail Logger FATAL ERROR: Couldn\'t copy file %s to directory. Attachment is not added to logged mail', 'rtm-mail'), $new_file);
            }
        }

        return $formatted_data;
    }

    /**
     * get_backtrace - trace back function calls until the catch_mail call
     * @return array
     */
    private function get_backtrace()
    {
        $result = [];
        $trace = array_reverse(debug_backtrace());
        array_pop($trace); // remove the last call from the trace
        // Loop through every line and add it to the result
        foreach ($trace as $trace_line) {
            if (!isset($trace_line['class']) || $trace_line['class'] !== 'WP_Hook') {
                $call = isset($trace_line['class']) ? $trace_line['class'] . $trace_line['type'] . $trace_line['function'] : $trace_line['function'];
                if (strpos($call, 'require') === false && strpos($call, 'include') === false) {
                    $result[] = [
                        'call' => $call,
                        'line' => $trace_line['line'],
                        'file' => $trace_line['file']
                    ];
                }
            }
        }
        return $result;
    }
}
