<?php

namespace RtmMail;

use Monolog\Logger;

/**
 * EventListener - Custom listener for error/debug logging
 */
class EventListener
{
    /**
     * CHANNEL_NAME - Custom name for Wonolog log channels
     */
    const CHANNEL_NAME = 'RTMMAIL';

    /**** Hook Callback Methods ****/

    /**
     * migration_success - hook callback for when a migration class succeeded
     * @param $migration_class
     */
    public function migration_success($migration_class)
    {
        do_action('wonolog.log', [
            'message' => "Migration of class <code>{$migration_class}</code> was successful",
            'channel' => self::CHANNEL_NAME,
            'level' => Logger::INFO,
            'context' => compact('migration_class'),
        ]);
    }

    /**
     * migration_failed - hook callback for when a migration class fails
     * @param $migration_class
     * @param $error
     */
    public function migration_failed($migration_class, $error)
    {
        do_action('wonolog.log', [
            'message' => "Migration of class <code>{$migration_class}</code> failed because of an error: {$error}",
            'channel' => self::CHANNEL_NAME,
            'level' => Logger::ERROR,
            'context' => compact('migration_class'),
        ]);
    }

    /**
     * log_deleted - hook callback when a mail log gets deleted by a user
     * @param $log_id
     * @param $user_id
     */
    public function log_deleted($log_id, $user_id)
    {
        $user_data = get_user_by('id', $user_id);
        do_action('wonolog.log', [
            'message' => "Email Log (#{$log_id}) has been deleted by {$user_data->display_name}",
            'channel' => self::CHANNEL_NAME,
            'level' => Logger::DEBUG,
            'context' => compact('log_id', 'user_id'),
        ]);
    }

    /**
     * log_edited - hook callback when a mail log gets edited by a user
     * @param $log_id
     * @param $user_id
     */
    public function log_edited($log_id, $user_id)
    {
        $user_data = get_user_by('id', $user_id);
        do_action('wonolog.log', [
            'message' => "{$user_data->display_name} edited Email Log: #{$log_id}",
            'channel' => self::CHANNEL_NAME,
            'level' => Logger::DEBUG,
            'context' => compact('log_id', 'user_id'),
        ]);
    }

    /**
     * send_failed - hook callback for when an email fails to be sent
     * @param $log_id
     * @param $error
     */
    public function send_failed($log_id, $error)
    {
        do_action('wonolog.log', [
            'message' => "Failed to send email (#{$log_id}): {$error}",
            'channel' => self::CHANNEL_NAME,
            'level' => Logger::ERROR,
            'context' => compact('log_id', 'error'),
        ]);
    }

    /**
     * send_success - hook callback for when an email has been sent
     * @param $log_id
     */
    public function send_success($log_id)
    {
        do_action('wonolog.log', [
            'message' => "Email with log id {$log_id} has been sent",
            'channel' => self::CHANNEL_NAME,
            'level' => Logger::INFO,
            'context' => compact('log_id'),
        ]);
    }
}