<?php

namespace RtmMail;

/**
 * Core - handles all plugin initialization
 */
class Core
{
    private $overview;
    private $settings;
    private $details;
    private $events;
    private $catcher;
    private $listener;

    /**
     * __construct - methods that need to be executed on object creation
     */
    public function __construct()
    {
        // Instantiate all pages
        $this->overview = new Overview();
        $this->settings = new Settings();
        $this->details = new Details();
        $this->events = new Events();

        // Instantiate the Catcher
        $this->catcher = new Catcher();

        // Instatiate error listener
        $this->listener = new EventListener();

        // Set custom Wonolog channel
        add_filter('wonolog.channels', function($channels) {
            $channels[] = EventListener::CHANNEL_NAME;
            return $channels;
        });
        // Change wonolog folder
        add_filter('wonolog.default-handler-folder', function($folder) {
            $upload = wp_upload_dir();
            return $upload['basedir'] . '/rtm-mail/logs';
        });

        // Apply all filters
        $this->add_filters();
        // Add all actions for initialization
        $this->add_actions();
    }

    /**
     * add_filters - all filters that need to be executed
     */
    private function add_filters()
    {
        // add the highest priority, so it's the last executed method
        add_filter('wp_mail', [$this->catcher, 'catch_mail'], 9999999);
        // Admin footer text
        add_filter('admin_footer_text', [$this, 'display_footer_text']);
    }

    /**
     * add_actions - all actions that need to be executed
     */
    private function add_actions()
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_menu', [$this, 'add_pages']);
        add_action('plugins_loaded', [$this, 'check_migrations']);
        add_action('admin_footer', [$this, 'display_footer']);
        add_action('wp_mail_failed', [$this->catcher, 'mail_error'], 9999999);
        // Event listener hooks
        add_action('rtmmail_migration_failed', [$this->listener, 'migration_failed'], 10, 2);
        add_action('rtmmail_migration_success', [$this->listener, 'migration_success']);
        add_action('rtmmail_send_failed', [$this->listener, 'send_failed'], 10, 2);
        add_action('rtmmail_send_success', [$this->listener, 'send_success']);
        add_action('rtmmail_log_deleted', [$this->listener, 'log_deleted'], 10, 2);
        add_action('rtmmail_log_edited', [$this->listener, 'log_edited'], 10, 2);
    }

    /**** Hook Methods ****/

    /**
     * activate - callback for activation hook
     */
    public function activate()
    {
        if (class_exists('WPMailSMTP\Core')) {
            add_action('admin_notices', function () {
                echo '<div class="notice notice-warning"> ';
                echo '<p><strong>RTM Mail: </strong>';
                echo __('You are currently using <code>WP-Mail-SMTP</code> which may conflict with the sender options in this plugin.', 'rtm-mail');
                echo '</p>';
                echo '</div>';
            });
        }
        update_option('rtm_mail_settings', [
            'send_mails' => true,
            'addresses' => [],
            'sender_options' => [
                'title' => '',
                'address' => '',
            ],
        ]);
    }

    /**
     * deactivate - callback for deactivation hook
     */
    public function deactivate()
    {
        delete_option('rtm_mail_settings');
    }

    /**
     * check_migrations - Check for update changes
     */
    public function check_migrations()
    {
        (new Migration(RTM_MAIL_VERSION))->check();
    }

    /**
     * add_pages - add the pages to the administrator menu
     */
    public function add_pages()
    {
        // Main admin page (overview)
        add_menu_page(__('RTM Mail - Overview', 'rtm-mail'), 'RTM Mail', 'manage_options', 'rtm-mail-overview', [$this->overview, 'display'], 'dashicons-email-alt');
        // Sub menu page (settings)
        add_submenu_page('rtm-mail-overview', __('RTM Mail - Settings', 'rtm-mail'), __('Settings', 'rtm-mail'), 'manage_options', 'rtm-mail-settings', [$this->settings, 'display']);
        // Sub menu page (events)
        add_submenu_page('rtm-mail-overview', __('RTM Mail - Events', 'rtm-mail'), __('Debug Events', 'rtm-mail'), 'activate_plugins', 'rtm-mail-events', [$this->events, 'display']);
        // Hidden submenu (details)
        add_submenu_page('', __('RTM Mail - Details', 'rtm-mail'), __('Details', 'rtm-mail'), 'manage_options', 'rtm-mail-details', [$this->details, 'display']);
    }

    /**
     * display_footer - add RTM Business logo to footer
     */
    public function display_footer()
    {
        // Check if user is on one of the rtm-mail pages
        if ($this->overview->page_check() || $this->settings->page_check() || $this->details->page_check() || $this->events->page_check()) {
            echo '<div class="rtm-mail-footer">';
            echo '<div class="rtm-mail-logo"><a href="https://www.rtmbusiness.nl/?utm_source=rtm-mail-plugin&utm_medium=footer&utm_campaign=mail-plugin" target="_blank"><img src="' . RTM_MAIL_PLUGIN_PATH . 'assets/images/rtm-logo.png" alt="rtm-logo" /></a></div>';
            echo '</div>';
        }
    }

    /**
     * display_footer_text - change the admin footer text
     */
    public function display_footer_text($text)
    {
        // Check if user is on one of the rtm-mail pages
        if ($this->overview->page_check() || $this->settings->page_check() || $this->details->page_check() || $this->events->page_check()) {
            $text = __('Thanks for using the RTM Mail Plugin!', 'rtm-mail');
        }
        return $text;
    }

    /**
     * enqueue_scripts - enqueues the scripts needed for the plugin
     */
    public function enqueue_scripts()
    {
        // Only load scripts on RTM mail pages
        if ($this->overview->page_check() || $this->settings->page_check() || $this->details->page_check() || $this->events->page_check()) {
            // Load custom styling css
            wp_enqueue_style('rtm-mail', RTM_MAIL_PLUGIN_PATH . 'assets/css/style.css');
            wp_register_script('rtm-mail', RTM_MAIL_PLUGIN_PATH . 'assets/js/script.js');
            wp_localize_script('rtm-mail', 'rtm_mail_translation', [
                'type_address' => __('Type an address', 'rtm-mail'),
                'status_sent' => __('Sent', 'rtm-mail'),
            ]);
            wp_enqueue_script('rtm-mail');
            // Load flatpickr datetime picker scripts
            wp_enqueue_style('rtm-mail-flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css');
            wp_enqueue_script('rtm-mail-flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr', ['jquery'], false, true);
        }
    }
}
