<?php

namespace RtmMail;

use DateTime;
use Exception;
use RtmMail\Helpers\LogHelper;
use WP_List_Table;

/**
 * LogTable - Table construct for the overview page based on WP_List_Table
 */
class LogsTable extends WP_List_Table
{
    public function __construct()
    {
        parent::__construct([
            'singular' => 'log',
            'plural' => 'logs',
            'ajax' => false
        ]);
    }

    /**
     * get_columns - all the columns we want in our overview
     * @return array
     */
    public function get_columns()
    {
        $columns = [
            'cb' => '<input type="checkbox" />',
            'subject' => __('Subject (#id)', 'rtm-mail'),
            'sender' => __('Sender', 'rtm-mail'),
            'receiver' => __('Receiver', 'rtm-mail'),
            'status' => __('Status', 'rtm-mail'),
            'created' => __('Created on', 'rtm-mail'),
            'sent' => __('Sent on', 'rtm-mail'),
            'action' => __('Action', 'rtm-mail')
        ];

        return $columns;
    }

    /**
     * get_sortable_columns - Sets column sortable based on 'column_name' => 'db_column_name'
     * @return array
     */
    public function get_sortable_columns()
    {
        $sortable_columns = [
            'subject' => ['id', false],
            'created' => ['created', false],
            'sent' => ['sent', false],
        ];

        return $sortable_columns;
    }

    /**
     * column_default - default way to show column data
     * @param array|object $item
     * @param string $column_name
     * @return string|void
     */
    function column_default($item, $column_name)
    {
        return esc_html($item[$column_name]);
    }

    /**
     * column_cb - column checkbox for selecting rows
     * @param array|object $item
     * @return string|void
     */
    public function column_cb($item)
    {
        return sprintf('<input type="checkbox" name="%1$s[]" value="%2$s" />', 'id', $item['id']);
    }

    /**
     * column_subject - column subject row data
     * @param $item
     * @return string
     */
    public function column_subject($item)
    {
        // Add the mail id after the subject
        return $item['subject'] . ' (#' . $item['id'] . ")";
    }

    /**
     * column_receiver - column subject row data
     * @param $item
     * @return string|void
     */
    public function column_receiver($item)
    {
        $label = !empty($item['receiver']) && !empty($item['receiver'][0]) ? $item['receiver'][0] : __('No receivers..');
        $receiver_count = count($item['receiver']);
        if ($receiver_count > 1) {
            $label .= ' (' . ($receiver_count - 1) . ' ' . __('more', 'rtm-mail') . ')';
        }

        return $label;
    }

    /**
     * column_status - column status row data
     * @param $item
     * @return string
     */
    public function column_status($item)
    {
        // Create a badge based on the status type
        $label = __('Caught', 'rtm-mail');
        switch ($item['status']) {
            case 'caught':
                $label = __('Caught', 'rtm-mail');
                break;
            case 'failed':
                $label = __('Failed', 'rtm-mail');
                break;
            case 'sent':
                $label = __('Sent', 'rtm-mail');
                break;
        }
        return '<span class="badge badge-' . $item['status'] . '">' . $label . '</span>';
    }

    /**
     * column_created - column created row data
     * @param $item
     * @return string
     * @throws Exception
     */
    public function column_created($item)
    {
        // Format date to dd-mm-yyyy
        $date = new DateTime($item['created']);
        return $date->format('d-m-Y H:i');
    }

    /**
     * column_sent - column sent row data
     * @param $item
     * @return string|void
     * @throws Exception
     */
    public function column_sent($item)
    {
        // Quick check if sent is not empty, if not format date to dd-mm-yyyy
        return !empty($item['sent']) ? (new DateTime($item['sent']))->format('d-m-Y H:i') : __('Not sent yet...', 'rtm-mail');
    }

    /**
     * column_action - add action button to row item
     * @param $item
     * @return string
     */
    public function column_action($item)
    {
        // Change label based on if the email was sent
        $label = __('Edit', 'rtm-mail');
        if ($item['status'] == 'sent') {
            $label = __('Details', 'rtm-mail');
        }
        return '<a href="' . get_admin_url() . 'admin.php?page=rtm-mail-details&log_id=' . $item['id'] . '"><input type="button" name="submit" id="submit" class="button button-action" value="' . $label . '" /></a>';
    }

    /**
     * search_box - displays the search box
     * @param string $text
     * @param string $input_id
     */
    public function search_box($text, $input_id)
    {
        if (empty($_REQUEST['s']) && !$this->has_items()) {
            return;
        }

        $input_id = $input_id . '-search-input';

        if (!empty($_REQUEST['orderby'])) {
            echo '<input type="hidden" name="orderby" value="' . esc_attr($_REQUEST['orderby']) . '" />';
        }
        if (!empty($_REQUEST['order'])) {
            echo '<input type="hidden" name="order" value="' . esc_attr($_REQUEST['order']) . '" />';
        }
        if (!empty($_REQUEST['post_mime_type'])) {
            echo '<input type="hidden" name="post_mime_type" value="' . esc_attr($_REQUEST['post_mime_type']) . '" />';
        }
        if (!empty($_REQUEST['detached'])) {
            echo '<input type="hidden" name="detached" value="' . esc_attr($_REQUEST['detached']) . '" />';
        }
        ?>
        <div class="search-filter">
            <div class="search-box">
                <label class="screen-reader-text" for="<?php echo esc_attr($input_id); ?>"><?php echo $text; ?>:</label>
                <input type="search" id="<?php echo esc_attr($input_id); ?>" name="s"
                       placeholder="<?php echo __('Search on content', 'rtm-mail'); ?>"
                       value="<?php _admin_search_query(); ?>"/>
                <?php submit_button($text, '', '', false, array('id' => 'search-submit')); ?>
            </div>
            <div class="filter-box">
                <div class="filter-label-box">
                    <p class="filter-label"><?php echo __('Filter by', 'rtm-mail'); ?></p>
                </div>
                <div class="filter-input">
                    <input type="checkbox" id="mail_subject" name="search_filter[subject]" checked>
                    <label for="mail_subject"><?php echo __('Subject', 'rtm-mail'); ?></label>
                    <input type="checkbox" id="mail_sender" name="search_filter[sender]" checked>
                    <label for="mail_sender"><?php echo __('Sender', 'rtm-mail'); ?></label>
                    <input type="checkbox" id="mail_receiver" name="search_filter[receiver]" checked>
                    <label for="mail_receiver"><?php echo __('Receiver', 'rtm-mail'); ?></label>
                    <input type="checkbox" id="mail_cc" name="search_filter[cc]" checked>
                    <label for="mail_cc"><?php echo __('CC', 'rtm-mail'); ?></label>
                    <input type="checkbox" id="mail_bcc" name="search_filter[bcc]" checked>
                    <label for="mail_bcc"><?php echo __('BCC', 'rtm-mail'); ?></label>
                    <input type="checkbox" id="mail_status" name="search_filter[status]" checked>
                    <label for="mail_status"><?php echo __('Status', 'rtm-mail'); ?></label>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * display_tablenav - displays top or bottom row for the table
     * @param string $which
     */
    public function display_tablenav($which)
    {
        // Only want things at the bottom so return when top
        if ('top' === $which) {
            return;
        }
        ?>
        <div class="tablenav <?php echo esc_attr($which); ?>">

            <?php if ($this->has_items()) : ?>
                <div class="alignleft actions bulkactions">
                    <input type="button" name="delete_logs" id="delete-submit" class="button button-delete"
                           value="Delete selected items">
                </div>
            <?php
            endif;
            $this->extra_tablenav($which);
            $this->pagination($which);
            ?>

            <br class="clear"/>
        </div>
        <?php
    }

    /**
     * prepare_items - Sets table items and different pagination settings
     */
    public function prepare_items()
    {
        $columns = $this->get_columns();
        $sortable = $this->get_sortable_columns();

        $this->_column_headers = [$columns, [], $sortable];

        // Set the default settings
        $defaults = LogHelper::DEFAULT_ARGS;
        // Get total amount of items from the logs
        $total_items = LogHelper::get_total_rows();
        $post_per_page = $defaults['post_per_page'];

        // Calculate total pages
        $total_pages = ceil($total_items / $post_per_page);

        // Set pagination values
        $this->set_pagination_args([
            "total_items" => $total_items,
            "total_pages" => $total_pages,
            "per_page" => $post_per_page,
        ]);
        // Set the items
        $this->items = LogHelper::get($_GET);
    }
}