<?php

namespace RtmMail;

/**
 * Core - handles all plugin initialization
 */
class Core
{
    private $overview;
    private $settings;
    private $details;
    private $catcher;

    /**
     * __construct - methods that need to be executed on object creation
     */
    public function __construct()
    {
        // Instantiate all pages
        $this->overview = new Overview();
        $this->settings = new Settings();
        $this->details = new Details();

        // Instantiate the Catcher
        $this->catcher = new Catcher();

        // Apply all filters
        $this->add_filters();
        // Add all actions for initialization
        $this->add_actions();
    }

    /**
     * add_filters - all filters that need to be executed
     */
    private function add_filters()
    {
        // add the highest priority, so it's the last executed method
        add_filter('wp_mail', [$this->catcher, 'catch_mail'], 9999999);
        // Admin footer text
        add_filter('admin_footer_text', [$this, 'display_footer_text']);
    }

    /**
     * add_actions - all actions that need to be executed
     */
    private function add_actions()
    {
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_menu', [$this, 'add_pages']);
        add_action('plugins_loaded', [$this, 'check_migrations']);
        add_action('admin_footer', [$this, 'display_footer']);
        add_action('wp_mail_failed', [$this->catcher, 'mail_error'], 9999999);
    }

    /**** Hook Methods ****/

    /**
     * check_migrations - Check for update changes
     */
    public function check_migrations()
    {
        (new Migration(RTM_MAIL_VERSION))->check();
    }

    /**
     * add_pages - add the pages to the administrator menu
     */
    public function add_pages()
    {
        // Main admin page (overview)
        add_menu_page(__('RTM Mail - Overview', 'rtm-mail'), 'RTM Mail', 'manage_options', 'rtm-mail-overview', [$this->overview, 'display'], 'dashicons-email-alt');
        // Sub menu page (settings)
        add_submenu_page('rtm-mail-overview', __('RTM Mail - Settings', 'rtm-mail'), __('Settings', 'rtm-mail'), 'manage_options', 'rtm-mail-settings', [$this->settings, 'display']);
        add_submenu_page('', __('RTM Mail - Details', 'rtm-mail'), __('Details', 'rtm-mail'), 'manage_options', 'rtm-mail-details', [$this->details, 'display']);
    }

    /**
     * display_footer - add RTM Business logo to footer
     */
    public function display_footer()
    {
        // Check if user is on one of the rtm-mail pages
        if ($this->overview->page_check() || $this->settings->page_check() || $this->details->page_check()) {
            echo '<div class="rtm-mail-footer">';
            echo '<div class="rtm-mail-logo"><a href="https://www.rtmbusiness.nl/?utm_source=rtm-mail-plugin&utm_medium=footer&utm_campaign=mail-plugin" target="_blank"><img src="' . RTM_MAIL_PLUGIN_PATH . 'assets/images/rtm-logo.png" alt="rtm-logo" /></a></div>';
            echo '</div>';
        }
    }

    /**
     * display_footer_text - change the admin footer text
     */
    public function display_footer_text($text)
    {
        // Check if user is on one of the rtm-mail pages
        if ($this->overview->page_check() || $this->settings->page_check() || $this->details->page_check()) {
            $text = __('Thanks for using the RTM Mail Plugin!', 'rtm-mail');
        }
        return $text;
    }

    /**
     * enqueue_scripts - enqueues the scripts needed for the plugin
     */
    public function enqueue_scripts()
    {
        // Load custom styling css
        wp_enqueue_style('rtm-mail', RTM_MAIL_PLUGIN_PATH . 'assets/css/style.css');
        wp_register_script('rtm-mail', RTM_MAIL_PLUGIN_PATH . 'assets/js/script.js');
        wp_localize_script('rtm-mail', 'rtm_mail_translation', [
            'type_address' => __('Type an address', 'rtm-mail')
        ]);
        wp_enqueue_script('rtm-mail');
        // Load flatpickr datetime picker scripts
        wp_enqueue_style('rtm-mail-flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css');
        wp_enqueue_script('rtm-mail-flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr', ['jquery'], false, true);
    }
}