<?php

namespace RtmMail;

use RtmMail\Helpers\LogHelper;

/**
 * Catcher - handles every incoming mail from the wp_mail hook
 */
class Catcher
{
    /**
     * catch_email - wp_mail hook callback to log all emails
     * @param $args
     * @return array (phpmailer)
     */
    public function catch_mail($args)
    {
        // if there is no Log_id in the headers it has to be logged
        if (empty($args['headers']) || substr($args["headers"][0], 0, 2) != 'Id') {
            $settings = get_option('rtm_mail_settings');
            // Convert "to" to an array when it's a string
            $args['to'] = gettype($args['to']) == 'string' ? $args['to'] = [$args['to']] : $args['to'];

            // Add every email from setting to receiver/cc/bcc
            if (isset($settings['addresses'])) {
                foreach ($settings['addresses'] as $address) {
                    $email_address = $address['address'];
                    $email_type = $address['type'];
                    // If the type is a recipient, add it to "to" otherwise add it to the "headers"
                    if ($email_type == 'recipient') {
                        $args['to'][] = $email_address;
                    } else {
                        $args['headers'][] = ucfirst($email_type) . ': ' . $email_address;
                    }
                }
            }
            // format the mail data
            $mail_data = $this->format($args);

            // Save the mail data as new log item
            $log_id = LogHelper::save($mail_data);

            // Check if email has to be sent or not
            if (!$settings['send_mails']) {
                // Set empty receivers so the email won't be sent
                $args['to'] = [];
                $args['headers'] = [];
            } else {
                LogHelper::update($log_id, ['status' => 'sent', 'sent' => date('Y-m-d H:i:s', time())]);
                array_unshift($args['headers'], "Id: " . $log_id);
            }
        }

        return $args;
    }

    /**
     * mail_error - handles error on mail failure
     * @param $error
     */
    public function mail_error($error)
    {
        // Check if there are headers available, so you can retrieve the Id
        if (!empty($error->error_data['wp_mail_failed']['headers'])) {
            $log_id = (int)$error->error_data['wp_mail_failed']['headers']['Id'];
            LogHelper::update($log_id, ['status' => 'failed', 'sent' => null]);
        }
        // TODO: Error log
    }

    /**
     * send_mail - sends mail based on saved log
     * @param $log_id
     */
    public static function send_mail($log_id)
    {
        $mail_log = LogHelper::get([
            'post_per_page' => null,
            'where' => [
                'id' => [
                    'type' => '=',
                    'value' => $log_id,
                ]
            ]
        ]);

        $mail_log = $mail_log[0] ?? null;
        if ($mail_log != null && $mail_log['status'] != 'sent') {
            $mail_data = [];
            // set the subject
            $mail_data['subject'] = $mail_log['subject'];
            // set the receivers
            $mail_data['to'] = $mail_log['receiver'];
            // set the body
            $mail_data['message'] = $mail_log['body'];
            // set the headers
            $mail_data['headers'] = [];
            // Set the sender
            $mail_data['headers'][] = 'From: ' . $mail_log['sender'];
            // Set the cc
            if (!empty($mail_log['cc'])) {
                foreach ($mail_log['cc'] as $cc_address) {

                    if (!empty($cc_address)) {
                        $mail_data['headers'][] = 'Cc: ' . $cc_address;
                    }
                }
            }
            // Set the bcc
            if (!empty($mail_log['bcc'])) {
                foreach ($mail_log['bcc'] as $bcc_address) {
                    if (!empty($bcc_address)) {
                        $mail_data['headers'][] = 'Bcc: ' . $bcc_address;
                    }
                }
            }
            // Set the id in headers
            array_unshift($mail_data['headers'], "Id: " . $log_id);
            // set the attachments
            $mail_data['attachments'] = [];
            if (!empty($mail_log['attachments'])) {
                foreach ($mail_log['attachments'] as $attachment) {
                    if (!empty($attachment)) {
                        $mail_data['attachments'][] = $attachment;
                    }
                }
            }
            $success = wp_mail($mail_data['to'], $mail_data['subject'], $mail_data['message'], $mail_data['headers'], $mail_data['attachments']);
            // Check if sending was successful
            if ($success) {
                // Set status on sent and redirect to the log details page
                LogHelper::update($log_id, ['status' => 'sent', 'sent' => date('Y-m-d H:i:s', time())]);
                wp_redirect(get_admin_url() . 'admin.php?page=rtm-mail-details&log_id=' . $log_id . '&sent_successful=true');
            } else {
                // Set the status to failed and redirect to the log details page
                LogHelper::update($log_id, ['status' => 'failed', 'sent' => null]);
                wp_redirect(get_admin_url() . 'admin.php?page=rtm-mail-details&log_id=' . $log_id . '&sent_failed=true');
            }
            exit();
        }
    }

    /**
     * format - formats the mail data to save as a log item
     * @param $args
     * @return array (formatted mail data)
     */
    private function format($args)
    {
        // Return value array
        $formatted_data = [];
        // Default from mail is the WordPress admin mail
        $formatted_data['sender'] = get_option('admin_email');
        $formatted_data['receiver'] = [];
        $formatted_data['cc'] = [];
        $formatted_data['bcc'] = [];
        $formatted_data['subject'] = $args['subject'];
        $formatted_data['body'] = stripslashes(htmlspecialchars_decode($args['message']));
        $formatted_data['attachments'] = [];
        $formatted_data['created'] = date('Y-m-d H:i:s', time());

        // Check for every 'to' if it's not empty
        foreach ($args['to'] as $receiver_mail) {
            if (!empty($receiver_mail)) {
                $formatted_data['receiver'][] = $receiver_mail;
            }
        }

        // Check for every header if the email is not empty and add it to the formatted data
        foreach ($args['headers'] as $header) {
            if (strpos(strtolower($header), 'from:') !== false) {
                $from_email = str_replace('from: ', '', strtolower($header));
                if (!empty($from_email)) {
                    $formatted_data['sender'] = $from_email;
                }
            } else if (strpos(strtolower($header), 'bcc:') !== false) {
                $bcc_email = str_replace('bcc: ', '', strtolower($header));
                if (!empty($bcc_email)) {
                    $formatted_data['bcc'][] = $bcc_email;
                }
            } else if (strpos(strtolower($header), 'cc:') !== false) {
                $cc_email = str_replace('cc: ', '', strtolower($header));
                if (!empty($cc_email)) {
                    $formatted_data['cc'][] = $cc_email;
                }
            }
        }

        // Set attachments
        if (isset($args['attachments'])) {
            $upload = wp_upload_dir();
            $upload_dir = $upload['basedir'] . '/rtm-mail-attachments';
            // Create new dir directory if it doesn't exist
            if (!is_dir($upload_dir)) {
                mkdir($upload_dir, 0700);
            }
            $date = date('Y_m_d H_i_s', time());
            // Loop through every attachment and add it to the formatted data
            foreach ($args['attachments'] as $attachment) {
                $file_name = substr($attachment, strrpos($attachment, '/'));
                $extension = explode('.', $file_name)[1];
                // new file directory and name based on date so its unique (rtm-mail-attachments/filename-YYYY_mm_dd_hh_mm_ss.ext)
                $new_file = $upload_dir . str_replace('.' . $extension, '', $file_name) . '-' . str_replace(' ', '', $date) . '.' . $extension;
                if (copy($attachment, $new_file)) {
                    $formatted_data['attachments'][] = $new_file;
                } else {
                    printf(__('RTM MAIL FATAL ERROR: Couldn\'t copy file %s to directory. Attachment is not added to logged mail', 'rtm-mail'), $new_file);
                }
            }
        }

        return $formatted_data;
    }
}