<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Cloner;

use RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\Caster;
use RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Exception\ThrowingCasterException;

/**
 * AbstractCloner implements a generic caster mechanism for objects and resources.
 *
 * @author Nicolas Grekas <p@tchwork.com>
 */
abstract class AbstractCloner implements ClonerInterface
{
    public static $defaultCasters = [
        '__PHP_Incomplete_Class' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\Caster', 'castPhpIncompleteClass'],

        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\CutStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'castStub'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\CutArrayStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'castCutArray'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ConstStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'castStub'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\EnumStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'castEnum'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ScalarStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'castScalar'],

        'Fiber' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\FiberCaster', 'castFiber'],

        'Closure' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castClosure'],
        'Generator' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castGenerator'],
        'ReflectionType' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castType'],
        'ReflectionAttribute' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castAttribute'],
        'ReflectionGenerator' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castReflectionGenerator'],
        'ReflectionClass' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castClass'],
        'ReflectionClassConstant' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castClassConstant'],
        'ReflectionFunctionAbstract' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castFunctionAbstract'],
        'ReflectionMethod' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castMethod'],
        'ReflectionParameter' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castParameter'],
        'ReflectionProperty' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castProperty'],
        'ReflectionReference' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castReference'],
        'ReflectionExtension' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castExtension'],
        'ReflectionZendExtension' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ReflectionCaster', 'castZendExtension'],

        'RtmMailVendor\Dependencies\Doctrine\Common\Persistence\ObjectManager' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'RtmMailVendor\Dependencies\Doctrine\Common\Proxy\Proxy' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DoctrineCaster', 'castCommonProxy'],
        'RtmMailVendor\Dependencies\Doctrine\ORM\Proxy\Proxy' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DoctrineCaster', 'castOrmProxy'],
        'RtmMailVendor\Dependencies\Doctrine\ORM\PersistentCollection' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DoctrineCaster', 'castPersistentCollection'],
        'RtmMailVendor\Dependencies\Doctrine\Persistence\ObjectManager' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],

        'DOMException' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castException'],
        'DOMStringList' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castLength'],
        'DOMNameList' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castLength'],
        'DOMImplementation' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castImplementation'],
        'DOMImplementationList' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castLength'],
        'DOMNode' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castNode'],
        'DOMNameSpaceNode' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castNameSpaceNode'],
        'DOMDocument' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castDocument'],
        'DOMNodeList' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castLength'],
        'DOMNamedNodeMap' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castLength'],
        'DOMCharacterData' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castCharacterData'],
        'DOMAttr' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castAttr'],
        'DOMElement' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castElement'],
        'DOMText' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castText'],
        'DOMDocumentType' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castDocumentType'],
        'DOMNotation' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castNotation'],
        'DOMEntity' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castEntity'],
        'DOMProcessingInstruction' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castProcessingInstruction'],
        'DOMXPath' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DOMCaster', 'castXPath'],

        'XMLReader' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\XmlReaderCaster', 'castXmlReader'],

        'ErrorException' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castErrorException'],
        'Exception' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castException'],
        'Error' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castError'],
        'Symfony\Bridge\Monolog\Logger' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'RtmMailVendor\Dependencies\Symfony\Component\DependencyInjection\ContainerInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'RtmMailVendor\Dependencies\Symfony\Component\EventDispatcher\EventDispatcherInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'Symfony\Component\HttpClient\AmpHttpClient' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castHttpClient'],
        'Symfony\Component\HttpClient\CurlHttpClient' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castHttpClient'],
        'Symfony\Component\HttpClient\NativeHttpClient' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castHttpClient'],
        'Symfony\Component\HttpClient\Response\AmpResponse' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castHttpClientResponse'],
        'Symfony\Component\HttpClient\Response\CurlResponse' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castHttpClientResponse'],
        'Symfony\Component\HttpClient\Response\NativeResponse' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castHttpClientResponse'],
        'RtmMailVendor\Dependencies\Symfony\Component\HttpFoundation\Request' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castRequest'],
        'Symfony\Component\Uid\Ulid' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castUlid'],
        'Symfony\Component\Uid\Uuid' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castUuid'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarExporter\Internal\LazyObjectState' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SymfonyCaster', 'castLazyObjectState'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Exception\ThrowingCasterException' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castThrowingCasterException'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\TraceStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castTraceStub'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\FrameStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castFrameStub'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Cloner\AbstractCloner' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'RtmMailVendor\Dependencies\Symfony\Component\ErrorHandler\Exception\FlattenException' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castFlattenException'],
        'RtmMailVendor\Dependencies\Symfony\Component\ErrorHandler\Exception\SilencedErrorContext' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ExceptionCaster', 'castSilencedErrorContext'],

        'Imagine\Image\ImageInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ImagineCaster', 'castImage'],

        'Ramsey\Uuid\UuidInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\UuidCaster', 'castRamseyUuid'],

        'ProxyManager\Proxy\ProxyInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ProxyManagerCaster', 'castProxy'],
        'PHPUnit_Framework_MockObject_MockObject' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'PHPUnit\Framework\MockObject\MockObject' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'PHPUnit\Framework\MockObject\Stub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'Prophecy\Prophecy\ProphecySubjectInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],
        'Mockery\MockInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\StubCaster', 'cutInternals'],

        'PDO' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\PdoCaster', 'castPdo'],
        'PDOStatement' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\PdoCaster', 'castPdoStatement'],

        'AMQPConnection' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\AmqpCaster', 'castConnection'],
        'AMQPChannel' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\AmqpCaster', 'castChannel'],
        'AMQPQueue' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\AmqpCaster', 'castQueue'],
        'AMQPExchange' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\AmqpCaster', 'castExchange'],
        'AMQPEnvelope' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\AmqpCaster', 'castEnvelope'],

        'ArrayObject' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castArrayObject'],
        'ArrayIterator' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castArrayIterator'],
        'SplDoublyLinkedList' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castDoublyLinkedList'],
        'SplFileInfo' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castFileInfo'],
        'SplFileObject' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castFileObject'],
        'SplHeap' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castHeap'],
        'SplObjectStorage' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castObjectStorage'],
        'SplPriorityQueue' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castHeap'],
        'OuterIterator' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castOuterIterator'],
        'WeakMap' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castWeakMap'],
        'WeakReference' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\SplCaster', 'castWeakReference'],

        'Redis' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RedisCaster', 'castRedis'],
        'Relay\Relay' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RedisCaster', 'castRedis'],
        'RedisArray' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RedisCaster', 'castRedisArray'],
        'RedisCluster' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RedisCaster', 'castRedisCluster'],

        'DateTimeInterface' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DateCaster', 'castDateTime'],
        'DateInterval' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DateCaster', 'castInterval'],
        'DateTimeZone' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DateCaster', 'castTimeZone'],
        'DatePeriod' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DateCaster', 'castPeriod'],

        'GMP' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\GmpCaster', 'castGmp'],

        'MessageFormatter' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\IntlCaster', 'castMessageFormatter'],
        'NumberFormatter' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\IntlCaster', 'castNumberFormatter'],
        'IntlTimeZone' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\IntlCaster', 'castIntlTimeZone'],
        'IntlCalendar' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\IntlCaster', 'castIntlCalendar'],
        'IntlDateFormatter' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\IntlCaster', 'castIntlDateFormatter'],

        'Memcached' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\MemcachedCaster', 'castMemcached'],

        'Ds\Collection' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DsCaster', 'castCollection'],
        'Ds\Map' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DsCaster', 'castMap'],
        'Ds\Pair' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DsCaster', 'castPair'],
        'RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DsPairStub' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\DsCaster', 'castPairStub'],

        'mysqli_driver' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\MysqliCaster', 'castMysqliDriver'],

        'CurlHandle' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castCurl'],

        ':dba' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castDba'],
        ':dba persistent' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castDba'],

        'GdImage' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castGd'],
        ':gd' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castGd'],

        ':pgsql large object' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\PgSqlCaster', 'castLargeObject'],
        ':pgsql link' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\PgSqlCaster', 'castLink'],
        ':pgsql link persistent' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\PgSqlCaster', 'castLink'],
        ':pgsql result' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\PgSqlCaster', 'castResult'],
        ':process' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castProcess'],
        ':stream' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castStream'],

        'OpenSSLCertificate' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castOpensslX509'],
        ':OpenSSL X.509' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castOpensslX509'],

        ':persistent stream' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castStream'],
        ':stream-context' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\ResourceCaster', 'castStreamContext'],

        'XmlParser' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\XmlResourceCaster', 'castXml'],
        ':xml' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\XmlResourceCaster', 'castXml'],

        'RdKafka' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castRdKafka'],
        'RdKafka\Conf' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castConf'],
        'RdKafka\KafkaConsumer' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castKafkaConsumer'],
        'RdKafka\Metadata\Broker' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castBrokerMetadata'],
        'RdKafka\Metadata\Collection' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castCollectionMetadata'],
        'RdKafka\Metadata\Partition' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castPartitionMetadata'],
        'RdKafka\Metadata\Topic' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castTopicMetadata'],
        'RdKafka\Message' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castMessage'],
        'RdKafka\Topic' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castTopic'],
        'RdKafka\TopicPartition' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castTopicPartition'],
        'RdKafka\TopicConf' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\RdKafkaCaster', 'castTopicConf'],

        'FFI\CData' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\FFICaster', 'castCTypeOrCData'],
        'FFI\CType' => ['RtmMailVendor\Dependencies\Symfony\Component\VarDumper\Caster\FFICaster', 'castCTypeOrCData'],
    ];

    protected $maxItems = 2500;
    protected $maxString = -1;
    protected $minDepth = 1;

    /**
     * @var array<string, list<callable>>
     */
    private array $casters = [];

    /**
     * @var callable|null
     */
    private $prevErrorHandler;

    private array $classInfo = [];
    private int $filter = 0;

    /**
     * @param callable[]|null $casters A map of casters
     *
     * @see addCasters
     */
    public function __construct(?array $casters = null)
    {
        $this->addCasters($casters ?? static::$defaultCasters);
    }

    /**
     * Adds casters for resources and objects.
     *
     * Maps resources or objects types to a callback.
     * Types are in the key, with a callable caster for value.
     * Resource types are to be prefixed with a `:`,
     * see e.g. static::$defaultCasters.
     *
     * @param callable[] $casters A map of casters
     *
     * @return void
     */
    public function addCasters(array $casters)
    {
        foreach ($casters as $type => $callback) {
            $this->casters[$type][] = $callback;
        }
    }

    /**
     * Sets the maximum number of items to clone past the minimum depth in nested structures.
     *
     * @return void
     */
    public function setMaxItems(int $maxItems)
    {
        $this->maxItems = $maxItems;
    }

    /**
     * Sets the maximum cloned length for strings.
     *
     * @return void
     */
    public function setMaxString(int $maxString)
    {
        $this->maxString = $maxString;
    }

    /**
     * Sets the minimum tree depth where we are guaranteed to clone all the items.  After this
     * depth is reached, only setMaxItems items will be cloned.
     *
     * @return void
     */
    public function setMinDepth(int $minDepth)
    {
        $this->minDepth = $minDepth;
    }

    /**
     * Clones a PHP variable.
     *
     * @param int $filter A bit field of Caster::EXCLUDE_* constants
     */
    public function cloneVar(mixed $var, int $filter = 0): Data
    {
        $this->prevErrorHandler = set_error_handler(function ($type, $msg, $file, $line, $context = []) {
            if (\E_RECOVERABLE_ERROR === $type || \E_USER_ERROR === $type) {
                // Cloner never dies
                throw new \ErrorException($msg, 0, $type, $file, $line);
            }

            if ($this->prevErrorHandler) {
                return ($this->prevErrorHandler)($type, $msg, $file, $line, $context);
            }

            return false;
        });
        $this->filter = $filter;

        if ($gc = gc_enabled()) {
            gc_disable();
        }
        try {
            return new Data($this->doClone($var));
        } finally {
            if ($gc) {
                gc_enable();
            }
            restore_error_handler();
            $this->prevErrorHandler = null;
        }
    }

    /**
     * Effectively clones the PHP variable.
     */
    abstract protected function doClone(mixed $var): array;

    /**
     * Casts an object to an array representation.
     *
     * @param bool $isNested True if the object is nested in the dumped structure
     */
    protected function castObject(Stub $stub, bool $isNested): array
    {
        $obj = $stub->value;
        $class = $stub->class;

        if (str_contains($class, "@anonymous\0")) {
            $stub->class = get_debug_type($obj);
        }
        if (isset($this->classInfo[$class])) {
            [$i, $parents, $hasDebugInfo, $fileInfo] = $this->classInfo[$class];
        } else {
            $i = 2;
            $parents = [$class];
            $hasDebugInfo = method_exists($class, '__debugInfo');

            foreach (class_parents($class) as $p) {
                $parents[] = $p;
                ++$i;
            }
            foreach (class_implements($class) as $p) {
                $parents[] = $p;
                ++$i;
            }
            $parents[] = '*';

            $r = new \ReflectionClass($class);
            $fileInfo = $r->isInternal() || $r->isSubclassOf(Stub::class) ? [] : [
                'file' => $r->getFileName(),
                'line' => $r->getStartLine(),
            ];

            $this->classInfo[$class] = [$i, $parents, $hasDebugInfo, $fileInfo];
        }

        $stub->attr += $fileInfo;
        $a = Caster::castObject($obj, $class, $hasDebugInfo, $stub->class);

        try {
            while ($i--) {
                if (!empty($this->casters[$p = $parents[$i]])) {
                    foreach ($this->casters[$p] as $callback) {
                        $a = $callback($obj, $a, $stub, $isNested, $this->filter);
                    }
                }
            }
        } catch (\Exception $e) {
            $a = [(Stub::TYPE_OBJECT === $stub->type ? Caster::PREFIX_VIRTUAL : '').'⚠' => new ThrowingCasterException($e)] + $a;
        }

        return $a;
    }

    /**
     * Casts a resource to an array representation.
     *
     * @param bool $isNested True if the object is nested in the dumped structure
     */
    protected function castResource(Stub $stub, bool $isNested): array
    {
        $a = [];
        $res = $stub->value;
        $type = $stub->class;

        try {
            if (!empty($this->casters[':'.$type])) {
                foreach ($this->casters[':'.$type] as $callback) {
                    $a = $callback($res, $a, $stub, $isNested, $this->filter);
                }
            }
        } catch (\Exception $e) {
            $a = [(Stub::TYPE_OBJECT === $stub->type ? Caster::PREFIX_VIRTUAL : '').'⚠' => new ThrowingCasterException($e)] + $a;
        }

        return $a;
    }
}
