<?php
/**
 * @license proprietary
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace RtmMailVendor\Dependencies\WordpressModels\DependencyInjection\Metabox;

use RtmMailVendor\Dependencies\Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Metabox service registry.
 */
class MetaboxRegistry
{

    /**
     * @var array{
     *     id: string,
     *     title: string,
     *     callback: callable,
     *     screen: string,
     *     context: string,
     *     priority: string,
     *     callback_args: array
     * }
     */
    private array $metaboxes = [];

    public function __construct(private ContainerInterface $container)
    {
    }

    /**
     * Registers the metaboxes for the current action to WordPress.
     *
     * @param \WP_Post|null $post
     * @return void
     */
    public function doAddMetabox(?\WP_Post $post)
    {
        $metaboxes = $this->metaboxes[current_action()];

        foreach ($metaboxes as $metabox) {
            if (apply_filters("wpm_should_render_metabox_{$metabox['id']}", true, $post)) {
                add_meta_box($metabox['id'],
                    $metabox['title'],
                    $metabox['callback'],
                    $metabox['screen'],
                    $metabox['context'],
                    $metabox['priority'],
                    $metabox['callback_args']
                );
            }
        }
    }

    /**
     * Schedule actions for metaboxes.
     *
     * Iterates over the metaboxes and schedules the actions for each.
     *
     * @return void
     */
    public function registerMetaboxes()
    {
        foreach ($this->metaboxes as $hook => $metaboxes) {
            // add the action to add the metabox
            add_action($hook, [$this, 'doAddMetabox']);

            // add additional hooks for each metabox
            foreach ($metaboxes as $metabox) {
                $id = $metabox['id'];
                $postType = $metabox['screen'];
                $serviceId = self::getMetaboxServiceId($id, $postType);
                $service = $this->container->get($serviceId);

                if (is_a($service, AbstractMetabox::class)) {
                    // add shouldRender hook
                    add_filter("wpm_should_render_metabox_$id", [$service, 'shouldRender'], 10, 2);

                    $enqueueSuffix = is_null($postType) ? '' : "/$postType";
                    // add script enqueue hooks
                    add_action("admin_enqueue_scripts-post.php$enqueueSuffix", fn() => $this->scheduleScriptEnqueue($service, $metabox));
                    add_action("admin_enqueue_scripts-post-new.php$enqueueSuffix", fn() => $this->scheduleScriptEnqueue($service, $metabox));
                }
            }
        }
    }

    /**
     * Runs the script enqueue for a metabox.
     *
     * @param AbstractMetabox $service
     * @param array $metabox
     * @return void
     */
    public function scheduleScriptEnqueue(AbstractMetabox $service, array $metabox): void
    {
        global $post;

        // dont enqueue if the metabox should not render
        if (apply_filters("wpm_should_render_metabox_{$metabox['id']}", true, $post)) {
            $service->enqueueScript();
        }
    }

    /**
     * Add a metabox to the registry.
     * Runs for all tagged 'metabox' services at container initialization.
     *
     * @param string $id
     * @param string $title
     * @param callable $callback
     * @param string $context
     * @param string $priority
     * @param string|array|null $postType
     * @param array $args
     * @return void
     */
    public function addMetabox(string            $id,
                               string            $title,
                               callable          $callback,
                               string            $context = 'advanced',
                               string            $priority = 'default',
                               string|array|null $postType = null,
                               array             $args = [])
    {
        if (is_string($postType)) {
            $postType = [$postType];
        } elseif (is_null($postType)) {
            $postType = [null];
        }

        foreach ($postType as $type) {
            $hook = !$type ? 'add_meta_boxes' : "add_meta_boxes_$type";

            $this->metaboxes[$hook][] = [
                'id' => $id,
                'title' => $title,
                'callback' => $callback,
                'screen' => $type,
                'context' => $context,
                'priority' => $priority,
                'callback_args' => $args
            ];
        }
    }

    /**
     * @return array
     */
    public function getMetaboxes(): array
    {
        return $this->metaboxes;
    }

    public static function getMetaboxServiceId(string $id, ?string $postType = null)
    {
        return !$postType ? "metabox.$id" : "metabox.$postType.$id";
    }

}