<?php
/**
 * @license MIT
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace RtmMailVendor\Dependencies\SPFLib\Term\Modifier;

use RtmMailVendor\Dependencies\SPFLib\Macro\MacroString;
use RtmMailVendor\Dependencies\SPFLib\Term\Modifier;

/**
 * Class that represents the "unknown" modifier.
 *
 * @see https://tools.ietf.org/html/rfc7208#section-12
 */
class UnknownModifier extends Modifier
{
    /**
     * @var string
     */
    private $name;

    /**
     * @var \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString
     */
    private $value;

    /**
     * Initialize the instance.
     *
     * @param \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString|string|null $value
     *
     * @throws \RtmMailVendor\Dependencies\SPFLib\Exception\InvalidMacroStringException if $value is a non empty string which does not represent a valid MacroString
     */
    public function __construct(string $name, $value)
    {
        $this->name = $name;
        if (!$value instanceof MacroString) {
            $value = MacroString\Decoder::getInstance()->decode($value === null ? '' : $value, MacroString\Decoder::FLAG_ALLOWEMPTY);
        }
        $this->value = $value;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term::__toString()
     */
    public function __toString(): string
    {
        return $this->getName() . '=' . (string) $this->getValue();
    }

    public function __clone()
    {
        $this->value = clone $this->getValue();
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\Modifier::getName()
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * Get the name of the modifier (the part after '=').
     */
    public function getValue(): MacroString
    {
        return $this->value;
    }
}
