<?php
/**
 * @license MIT
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace RtmMailVendor\Dependencies\SPFLib\Term\Modifier;

use RtmMailVendor\Dependencies\SPFLib\Macro\MacroString;
use RtmMailVendor\Dependencies\SPFLib\Term\Modifier;
use RtmMailVendor\Dependencies\SPFLib\Term\TermWithDomainSpec;

/**
 * Class that represents the "redirect" modifier.
 *
 * @see https://tools.ietf.org/html/rfc7208#section-6.1
 */
class RedirectModifier extends Modifier implements TermWithDomainSpec
{
    /**
     * The handle that identifies this modifier.
     *
     * @var string
     */
    public const HANDLE = 'redirect';

    /**
     * @var \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString
     */
    private $domainSpec;

    /**
     * Initialize the instance.
     *
     * @param \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString|string $domainSpec
     *
     * @throws \RtmMailVendor\Dependencies\SPFLib\Exception\InvalidMacroStringException if $domainSpec is a string which does not represent a valid MacroString, or if it's an empty MacroString instance
     */
    public function __construct($domainSpec)
    {
        if (!$domainSpec instanceof MacroString || $domainSpec->isEmpty()) {
            $domainSpec = MacroString\Decoder::getInstance()->decode($domainSpec);
        }
        $this->domainSpec = $domainSpec;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term::__toString()
     */
    public function __toString(): string
    {
        return static::HANDLE . '=' . $this->getDomainSpec();
    }

    public function __clone()
    {
        $this->domainSpec = clone $this->getDomainSpec();
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\Modifier::getName()
     */
    public function getName(): string
    {
        return static::HANDLE;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\TermWithDomainSpec::getDomainSpec()
     */
    public function getDomainSpec(): MacroString
    {
        return $this->domainSpec;
    }
}
