<?php
/**
 * @license MIT
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace RtmMailVendor\Dependencies\SPFLib\Term\Mechanism;

use RtmMailVendor\Dependencies\SPFLib\Macro\MacroString;
use RtmMailVendor\Dependencies\SPFLib\Term\Mechanism;
use RtmMailVendor\Dependencies\SPFLib\Term\TermWithDomainSpec;

/**
 * Class that represents the "ptr" mechanism.
 *
 * @see https://tools.ietf.org/html/rfc7208#section-5.5
 */
class PtrMechanism extends Mechanism implements TermWithDomainSpec
{
    /**
     * The handle that identifies this mechanism.
     *
     * @var string
     */
    public const HANDLE = 'ptr';

    /**
     * @var \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString
     */
    private $domainSpec;

    /**
     * Initialize the instance.
     *
     * @param string $qualifier the qualifier of this mechanism (the value of one of the Mechanism::QUALIFIER_... constants)
     * @param \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString|string|null $domainSpec
     */
    public function __construct(string $qualifier, $domainSpec = null)
    {
        parent::__construct($qualifier);
        if (!$domainSpec instanceof MacroString) {
            $domainSpec = MacroString\Decoder::getInstance()->decode($domainSpec === null ? '' : $domainSpec, MacroString\Decoder::FLAG_ALLOWEMPTY);
        }
        $this->domainSpec = $domainSpec;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term::__toString()
     */
    public function __toString(): string
    {
        $result = $this->getQualifier(true) . static::HANDLE;
        $domainSpec = $this->getDomainSpec();
        if (!$domainSpec->isEmpty()) {
            $result .= ':' . (string) $domainSpec;
        }

        return $result;
    }

    public function __clone()
    {
        $this->domainSpec = clone $this->getDomainSpec();
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\Mechanism::getName()
     */
    public function getName(): string
    {
        return static::HANDLE;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\TermWithDomainSpec::getDomainSpec()
     */
    public function getDomainSpec(): MacroString
    {
        return $this->domainSpec;
    }
}
