<?php
/**
 * @license MIT
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

declare(strict_types=1);

namespace RtmMailVendor\Dependencies\SPFLib\Term\Mechanism;

use RtmMailVendor\Dependencies\SPFLib\Macro\MacroString;
use RtmMailVendor\Dependencies\SPFLib\Term\Mechanism;
use RtmMailVendor\Dependencies\SPFLib\Term\TermWithDomainSpec;

/**
 * Class that represents the "include" mechanism.
 *
 * @see https://tools.ietf.org/html/rfc7208#section-5.2
 */
class IncludeMechanism extends Mechanism implements TermWithDomainSpec
{
    /**
     * The handle that identifies this mechanism.
     *
     * @var string
     */
    public const HANDLE = 'include';

    /**
     * @var \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString
     */
    private $domainSpec;

    /**
     * Initialize the instance.
     *
     * @param string $qualifier the qualifier of this mechanism (the value of one of the Mechanism::QUALIFIER_... constants)
     * @param \RtmMailVendor\Dependencies\SPFLib\Macro\MacroString|string $domainSpec
     *
     * @throws \RtmMailVendor\Dependencies\SPFLib\Exception\InvalidMacroStringException if $domainSpec is a string which does not represent a valid MacroString, or if it's an empty MacroString instance
     */
    public function __construct(string $qualifier, $domainSpec)
    {
        parent::__construct($qualifier);
        if (!$domainSpec instanceof MacroString || $domainSpec->isEmpty()) {
            $domainSpec = MacroString\Decoder::getInstance()->decode($domainSpec);
        }
        $this->domainSpec = $domainSpec;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term::__toString()
     */
    public function __toString(): string
    {
        return $this->getQualifier(true) . static::HANDLE . ':' . (string) $this->getDomainSpec();
    }

    public function __clone()
    {
        $this->domainSpec = clone $this->getDomainSpec();
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\Mechanism::getName()
     */
    public function getName(): string
    {
        return static::HANDLE;
    }

    /**
     * {@inheritdoc}
     *
     * @see \RtmMailVendor\Dependencies\SPFLib\Term\TermWithDomainSpec::getDomainSpec()
     */
    public function getDomainSpec(): MacroString
    {
        return $this->domainSpec;
    }
}
