<?php
/**
 * @license MIT
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace RtmMailVendor\Dependencies\Doctrine\Bundle\DoctrineBundle;

use RtmMailVendor\Dependencies\Doctrine\ORM\EntityManagerInterface;
use RtmMailVendor\Dependencies\Doctrine\ORM\ORMException;
use RtmMailVendor\Dependencies\Doctrine\Persistence\Proxy;
use ProxyManager\Proxy\LazyLoadingInterface;
use RtmMailVendor\Dependencies\Psr\Container\ContainerInterface;
use RtmMailVendor\Dependencies\Symfony\Bridge\Doctrine\ManagerRegistry;
use RtmMailVendor\Dependencies\Symfony\Component\VarExporter\LazyObjectInterface;
use RtmMailVendor\Dependencies\Symfony\Contracts\Service\ResetInterface;

use function array_keys;
use function assert;

/**
 * References all Doctrine connections and entity managers in a given Container.
 */
class Registry extends ManagerRegistry implements ResetInterface
{
    /**
     * @param string[] $connections
     * @param string[] $entityManagers
     */
    public function __construct(ContainerInterface $container, array $connections, array $entityManagers, string $defaultConnection, string $defaultEntityManager)
    {
        $this->container = $container;

        parent::__construct('ORM', $connections, $entityManagers, $defaultConnection, $defaultEntityManager, Proxy::class);
    }

    /**
     * Resolves a registered namespace alias to the full namespace.
     *
     * This method looks for the alias in all registered entity managers.
     *
     * @see Configuration::getEntityNamespace
     *
     * @param string $alias The alias
     *
     * @return string The full namespace
     */
    public function getAliasNamespace($alias)
    {
        foreach (array_keys($this->getManagers()) as $name) {
            $objectManager = $this->getManager($name);

            if (! $objectManager instanceof EntityManagerInterface) {
                continue;
            }

            try {
                /** @psalm-suppress UndefinedMethod ORM < 3 specific */
                return $objectManager->getConfiguration()->getEntityNamespace($alias);
            } catch (ORMException $e) {
            }
        }

        throw ORMException::unknownEntityNamespace($alias);
    }

    public function reset(): void
    {
        foreach ($this->getManagerNames() as $managerName => $serviceId) {
            $this->resetOrClearManager($managerName, $serviceId);
        }
    }

    private function resetOrClearManager(string $managerName, string $serviceId): void
    {
        if (! $this->container->initialized($serviceId)) {
            return;
        }

        $manager = $this->container->get($serviceId);

        assert($manager instanceof EntityManagerInterface);

        if ((! $manager instanceof LazyLoadingInterface && ! $manager instanceof LazyObjectInterface) || $manager->isOpen()) {
            $manager->clear();

            return;
        }

        $this->resetManager($managerName);
    }
}
