<?php
/**
 * @license MIT
 *
 * Modified by RTM Business on 16-December-2024 using {@see https://github.com/BrianHenryIE/strauss}.
 */

namespace RtmMailVendor\Dependencies\Doctrine\DBAL\Driver;

use RtmMailVendor\Dependencies\Doctrine\DBAL\Connection;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Driver\API\ExceptionConverter;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Driver\API\PostgreSQL;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Exception;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Platforms\AbstractPlatform;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Platforms\PostgreSQL100Platform;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Platforms\PostgreSQL94Platform;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Platforms\PostgreSQLPlatform;
use RtmMailVendor\Dependencies\Doctrine\DBAL\Schema\PostgreSQLSchemaManager;
use RtmMailVendor\Dependencies\Doctrine\DBAL\VersionAwarePlatformDriver;
use RtmMailVendor\Dependencies\Doctrine\Deprecations\Deprecation;

use function assert;
use function preg_match;
use function version_compare;

/**
 * Abstract base implementation of the {@see Driver} interface for PostgreSQL based drivers.
 */
abstract class AbstractPostgreSQLDriver implements VersionAwarePlatformDriver
{
    /**
     * {@inheritDoc}
     */
    public function createDatabasePlatformForVersion($version)
    {
        if (preg_match('/^(?P<major>\d+)(?:\.(?P<minor>\d+)(?:\.(?P<patch>\d+))?)?/', $version, $versionParts) === 0) {
            throw Exception::invalidPlatformVersionSpecified(
                $version,
                '<major_version>.<minor_version>.<patch_version>',
            );
        }

        $majorVersion = $versionParts['major'];
        $minorVersion = $versionParts['minor'] ?? 0;
        $patchVersion = $versionParts['patch'] ?? 0;
        $version      = $majorVersion . '.' . $minorVersion . '.' . $patchVersion;

        if (version_compare($version, '10.0', '>=')) {
            return new PostgreSQL100Platform();
        }

        Deprecation::trigger(
            'doctrine/dbal',
            'https://github.com/doctrine/dbal/pull/5060',
            'PostgreSQL 9 support is deprecated and will be removed in DBAL 4.'
                . ' Consider upgrading to Postgres 10 or later.',
        );

        return new PostgreSQL94Platform();
    }

    /**
     * {@inheritDoc}
     */
    public function getDatabasePlatform()
    {
        return new PostgreSQL94Platform();
    }

    /**
     * {@inheritDoc}
     *
     * @deprecated Use {@link PostgreSQLPlatform::createSchemaManager()} instead.
     */
    public function getSchemaManager(Connection $conn, AbstractPlatform $platform)
    {
        Deprecation::triggerIfCalledFromOutside(
            'doctrine/dbal',
            'https://github.com/doctrine/dbal/pull/5458',
            'AbstractPostgreSQLDriver::getSchemaManager() is deprecated.'
                . ' Use PostgreSQLPlatform::createSchemaManager() instead.',
        );

        assert($platform instanceof PostgreSQLPlatform);

        return new PostgreSQLSchemaManager($conn, $platform);
    }

    public function getExceptionConverter(): ExceptionConverter
    {
        return new PostgreSQL\ExceptionConverter();
    }
}
