<?php

namespace RtmMail\Actions\Handlers;

use RtmMail\Helpers\LogHelper;
use RtmMail\Logger;
use RtmMail\Utils\LoggerUtils;

class RemoveQueueAction extends AbstractActionHandler
{
    public function __construct($context = 'overview')
    {
        parent::__construct('dequeue_logs', 'rtm_mail_remove_queue', $context);
    }

    public function handleAction($log_ids = [], $queue_timestamp = null): array
    {
        if (! $this->verifyNonce()) {
            return [ 'success' => false, 'message' => 'Nonce verification failed.' ];
        }

        // Validating the log_ids
        if (empty($log_ids)) {
            return [ 'success' => false, 'message' => 'Log ID is required.' ];
        }

        if ($this->context !== "overview" && ( ! isset($queue_timestamp) || ! is_numeric($queue_timestamp) )) {
            return [ 'success' => false, 'message' => 'Valid queue timestamp is required.' ];
        }

        // Unscheduling the event and updating the log status
        foreach ($log_ids as $log_id) {
            if ($this->context !== 'overview' && $queue_timestamp) {
                wp_unschedule_event((int) $queue_timestamp, 'rtm_mail_send_log', [ $log_id ]);
            } else {
                // If in 'Overview', unschedule all hooks for the log ID without a specific timestamp
                wp_clear_scheduled_hook('rtm_mail_send_log', [ $log_id ]);
            }

            LogHelper::update($log_id, [ 'status' => 'caught' ]);

            $user = LoggerUtils::getUser();
            Logger::warning(
                "Email log (#$log_id) has been dequeued",
                "$user has removed mail log (#$log_id) from the queue",
                $this->context . ':handle_requests',
                'Log',
                [
                    "mail_log" => $log_id,
                    "user"     => $user,
                ]
            );
        }

        if ($this->context === "overview") {
            return [
                'success'  => true,
                'message'  => 'Log successfully removed from the queue.',
                'redirect' => get_admin_url() . 'admin.php?page=rtm-mail-overview'
            ];
        } else {
            return [
                'success'  => true,
                'message'  => 'Log successfully removed from the queue.',
                'redirect' => get_admin_url() . 'admin.php?page=rtm-mail-details&log_id=' . implode(',', $log_ids)
            ];
        }
    }
}
