<?php

namespace RtmMail\Actions\Handlers;

use RtmMail\Helpers\LogHelper;
use RtmMail\Logger;
use RtmMail\Utils\LoggerUtils;

class QueueLogAction extends AbstractActionHandler
{
    public function __construct($context = 'Details')
    {
        parent::__construct('queue_logs', 'rtm_mail_queue_mails', $context);
    }

    public function handleAction($log_ids = [], $queue_date = ''): array
    {
        if (! $this->verifyNonce()) {
            return [ 'success' => false, 'message' => 'Nonce verification failed.' ];
        }

        // Ensure log_ids is an array
        $log_ids = is_array($log_ids) ? $log_ids : [ $log_ids ];

        // Filter out invalid IDs (empty, non-numeric values)
        $log_ids = array_filter($log_ids, function ($id) {
            return ! empty($id) && is_numeric($id);
        });

        // Make the array unique
        $log_ids = array_unique($log_ids);

        if (empty($log_ids)) {
            return [ 'success' => false, 'message' => 'Valid Log ID is required.' ];
        }

        if (empty($queue_date)) {
            return [ 'success' => false, 'message' => 'Queue date is required.' ];
        }

        $queue_timestamp = strtotime($queue_date . ' ' . wp_timezone_string());

        if ($queue_timestamp <= time()) {
            return [
                'success' => false,
                'message' => 'Selected date and time are earlier than the current time, please select a valid date and time!'
            ];
        }

        $user = LoggerUtils::getUser();
        foreach ($log_ids as $log_id) {
            wp_schedule_single_event($queue_timestamp, 'rtm_mail_send_log', [ $log_id ]);
            LogHelper::update($log_id, [ 'status' => 'queued' ]);
            Logger::info(
                "Email log (#{$log_id}) has been queued",
                "Log (#{$log_id}) has been queued for {$queue_date}",
                $this->context . ':handle_requests',
                'Log',
                [
                    "mail_log"   => $log_id,
                    "queue_date" => $queue_date,
                    "user"       => $user,
                ]
            );
        }

        return [
            'success' => true,
            'message' => 'Logs have been successfully queued.'
        ];
    }
}
