<?php

namespace RtmMail;
use RtmMail\Utils\LoggerUtils;

/**
 * Settings - handles every user interaction for the settings page
 */
class Settings extends AbstractPage
{
    protected $slug = 'settings';

    public function __construct()
    {
        parent::__construct();
    }

    /**
     * handle_requests - every page can handle their requests (post/get)
     */
    public function handle_requests()
    {
        // Check if the user is on the settings page
        if ($this->page_check()) {
            // Check for save settings post request
            if (isset($_POST['save_settings'])) {
                // Verify if the post is from the right source
                if (wp_verify_nonce($_POST['_wpnonce'], 'rtm_mail_save_settings')) {
                    // Get previous settings
                    $prev_settings = get_option('rtm_mail_settings');

                    $addresses = [];
                    $outgoing_mails = $_POST['option_email'];
                    $outgoing_types = $_POST['outgoing_type'];
                    $sender_options = $_POST['sender_options'];
                    $mail_capability = $_POST['mail_capability'];
                    $cron_schedule = $_POST['mail_cron_schedule'];

                    // Check if send mails and edit mails is enabled
                    $block_mails = isset($_POST['block_mails']) ? filter_var($_POST['block_mails'], FILTER_VALIDATE_BOOLEAN) : false;
                    $edit_mails = isset($_POST['edit_mails']) ? filter_var($_POST['edit_mails'], FILTER_VALIDATE_BOOLEAN) : false;
                    $cron_enabled = isset($_POST['mail_cron_enabled']) ? filter_var($_POST['mail_cron_enabled'], FILTER_VALIDATE_BOOLEAN) : false;
                    $clear_deleted_events = isset($_POST['clear_deleted_events']) ? filter_var($_POST['clear_deleted_events'], FILTER_VALIDATE_BOOLEAN) : false;

                    // Loop through every outgoing mail and check if an address is filled in
                    foreach ($outgoing_mails as $index => $address) {
                        if (!empty($address)) {
                            // Add to addresses array
                            $addresses[] = [
                                'address' => $address,
                                'type' => $outgoing_types[$index]
                            ];
                        }
                    }

                    $prev_cron_enabled = $prev_settings['mail_cron_enabled'] ?? false;
                    $prev_cron_schedule = $prev_settings['mail_cron_schedule'] ?? 'hourly';
                    // Check if cron settings is changed
                    if ($cron_enabled !== $prev_cron_enabled || $cron_schedule !== $prev_cron_schedule) {
                        // Clear current schedule
                        wp_clear_scheduled_hook('rtm_mail_send_caught_logs');
                        // Set new schedule
                        if ($cron_enabled) {
                            wp_schedule_event((time() + wp_get_schedules()[$cron_schedule]['interval']), $cron_schedule, 'rtm_mail_send_caught_logs');
                        }
                    }

                    $updated_settings = [
                        'block_mails' => $block_mails,
                        'edit_mails' => $edit_mails,
                        'addresses' => $addresses,
                        'sender_options' => $sender_options,
                        'mail_capability' => $mail_capability,
                        'clear_deleted_events' => $clear_deleted_events,
                        'mail_cron_enabled' => $cron_enabled,
                        'mail_cron_schedule' => $cron_schedule,
                    ];
                    // Save to option table
                    update_option('rtm_mail_settings', $updated_settings);

                    $user = LoggerUtils::getUser();
                    $changed_settings = array_filter($updated_settings, function ($setting) use ($prev_settings, $updated_settings) {
                        return isset($prev_settings[$setting]) && $prev_settings[$setting] !== $updated_settings[$setting];
                    }, ARRAY_FILTER_USE_KEY);
                    $changed_settings = array_combine(
                        array_keys($changed_settings),
                        array_map(function($updated_setting) use ($prev_settings, $updated_settings) {
                        return [
                            "from" => $prev_settings[$updated_setting],
                            "to" => $updated_settings[$updated_setting],
                        ];
                    }, array_keys($changed_settings)));

                    Logger::info(
                        "Settings saved",
                        "{$user} has changed the settings of the plugin",
                        'Settings::handle_requests',
                        'Settings',
                        [
                            "user" => $user,
                            "changed_settings" => $changed_settings
                        ]
                    );

                    // Send notification
                    echo '<div class="notice notice-success"> ';
                    echo '<p><strong>WP Mail Logger: </strong>';
                    echo __('Settings successfully saved!', 'rtm-mail');
                    echo '</p>';
                    echo '</div>';
                } else {
                    // Quit when invalid nonce
                    echo '<div class="notice notice-error"> ';
                    echo '<p><strong>WP Mail Logger: </strong>';
                    echo __('Invalide nonce!', 'rtm-mail');
                    echo '</p>';
                    echo '</div>';
                }
            }
        }
    }
}
