<?php

namespace RtmMail;

use RtmMail\Migrations\LoggerMigration;

class Logger
{
    const CHANNEL = 'RTM-MAIL-LOGGER';

    /**
     * Adds new log item to the event logger
     * @param string $severity severity level based on monolog
     * @param string $title title of the log
     * @param string $message message of the log
     * @param string $source source where the log is from
     * @param string $category category of the log
     * @param array $data given data for log
     * @return void
     */
    private static function log(string $severity, string $title, string $message, string $source, string $category, array $data): void
    {
        // Check for logger migration before logging
        if (!(new LoggerMigration())->is_migrated()) {
            return;
        }
        do_action('rtm-mail.log', [
            'message' => $message,
            'channel' => self::CHANNEL,
            'level' => $severity,
            'context' => [
                'title' => $title,
                'category' => $category,
                'source' => $source,
                'metadata' => self::getMetadata(),
                'data' => $data
            ],
            'datetime' => new \DateTime()
        ]);
    }

    /**
     * @param string $title title of the log
     * @param string $message message of the log
     * @param string $source source where the log is from
     * @param string $category category of the log
     * @param array $data given data for log
     * @return void
     */
    public static function error(string $title, string $message, string $source, string $category = 'plugin', array $data = []): void
    {
        self::log('error', $title, $message, $source, $category, $data);
    }

    /**
     * @param string $title title of the log
     * @param string $message message of the log
     * @param string $source source where the log is from
     * @param string $category category of the log
     * @param array $data given data for log
     * @return void
     */
    public static function warning(string $title, string $message, string $source, string $category = 'plugin', array $data = []): void
    {
        self::log('warning', $title, $message, $source, $category, $data);
    }

    /**
     * @param string $title title of the log
     * @param string $message message of the log
     * @param string $source source where the log is from
     * @param string $category category of the log
     * @param array $data given data for log
     * @return void
     */
    public static function info(string $title, string $message, string $source, string $category = 'plugin', array $data = []): void
    {
        self::log('info', $title, $message, $source, $category, $data);
    }

    /**
     * @param string $title title of the log
     * @param string $message message of the log
     * @param string $source source where the log is from
     * @param string $category category of the log
     * @param array $data given data for log
     * @return void
     */
    public static function debug(string $title, string $message, string $source, string $category = 'plugin', array $data = []): void
    {
        self::log('debug', $title, $message, $source, $category, $data);
    }

    /**
     * Gets the metadata that contains WordPress information
     * @return array
     */
    private static function getMetadata(): array
    {
        // TODO: Generate metadata (as much data as possible, such as WP data or Plugin data)
        return [];
    }
}
