<?php

namespace RtmMail\Helpers;

use Devdot\Monolog\Reader\LogReader;
use RtmMail\Models\Entities\Log;

/**
 * EventHelper - Helper for loading log data to a table
 */
class EventHelper
{
    /**
     * DEFAULT_ARGS - default settings to use for getting all logs
     * @type array
     */
    const DEFAULT_ARGS = [
        'orderby' => 'date',
        'order' => 'DESC',
        'post_per_page' => 10,
        'paged' => 1,
        'date' => null,
        's' => null,
    ];

    /**
     * get - Get all the logs from WP_PLUGIN_DIR/rtm-mail/logs
     * @param array $args
     * @return array|int
     */
    public static function get($args = [], $isCount = false)
    {
        // Merge the defaults with given args
        $args = array_merge(self::DEFAULT_ARGS, $args);
        $log = Log::query()->limit($args['post_per_page'])->offset(($args['paged'] - 1) * $args['post_per_page'])->order('desc');
        if (isset($args['filter_severity']) && isset($args['severity_type']) && !empty($args['severity_type'])) {
            $log->where('severity', esc_sql($args['severity_type']));
        }
        if (isset($args['s']) && !empty($args['s']) && isset($args['search_filter'] )) {
            foreach ($args['search_filter'] as $filter_column => $is_filter) {
                $filter_column = esc_sql($filter_column);
                // Search by the selected filter column
                if (filter_var($is_filter, FILTER_VALIDATE_BOOLEAN)) {
                    if ($filter_column === "log_id") {
                        if (is_numeric($args['s'])) {
                            $log->whereLike('data', '"mail_log";i:'.esc_sql($args['s']), 'OR');
                        } else {
                            $log->whereLike('data', '"mail_log";s:'.strlen($args['s']).':"'.esc_sql($args['s']).'"', 'OR');
                        }
                    } else if ($filter_column === "id") {
                        $log->where($filter_column, esc_sql($args['s']));
                    } else {
                        $log->whereLike($filter_column, esc_sql($args['s']), 'OR');
                    }
                }
            }
        }

        return !$isCount ? $log->execute() : $log->count();
    }

    /**
     * delete - delete all log files
     * @param false $deleted_only - only delete lines from deleted mail logs
     */
    public static function delete($deleted_only = false)
    {
        $upload = wp_upload_dir();
        // Get all directories in logs path
        $log_directories = self::get_directories($upload['basedir'] . '/rtm-mail/logs');
        // Array to save every log file path
        $log_paths = [];
        foreach ($log_directories as $directory) {
            // Scan for files in each directory
            $directory_files = array_diff(scandir($directory), ['.', '..']);
            foreach ($directory_files as $log_file) {
                // For every file check if it has a log extension
                $ext = explode('.', $log_file);
                if (!empty($log_file) && isset($ext[1]) && strtolower($ext[1]) === 'log') {
                    // Add to paths array
                    $log_paths[] = $directory . '/' . $log_file;
                }
            }
        }
        // Check if only to remove deleted mail logs
        if (!$deleted_only) {
            foreach ($log_paths as $file_path) {
                unlink($file_path);
            }
        } else {
            // Get all deleted mail logs
            $deleted_logs = self::get(['post_per_page' => null, 'search_filter' => ['message' => 'on'], 's' => 'deleted']);
            // Loop through all logs and paths
            foreach ($deleted_logs as $mail_log) {
                foreach ($log_paths as $file_path) {
                    // Get content of log file
                    $file = file_get_contents($file_path);
                    // Get every line
                    $lines = explode("\n", str_replace("\\r\\n", "\n", $file));
                    $new_lines = [];
                    // Loop over every line
                    foreach ($lines as $line) {
                        // Check if the current log_id contains in this line
                        if (strpos($line, '"log_id":"' . $mail_log['context']['log_id'] . '"')) {
                            // Skip this line
                            continue;
                        }
                        // Add line to the new lines if its not empty
                        if (!empty($line)) {
                            $new_lines[] = $line;
                        }
                    }
                    // Create new file content
                    $new_file = implode("\n", $new_lines);
                    // Set new file content
                    file_put_contents($file_path, $new_file);
                }
            }
        }
    }

    /**
     * get_total_rows - Get total amount of rows
     * @return int|void
     */
    public static function get_total_rows($args = [])
    {
        $args = array_merge($args, ['post_per_page' => -1]);
        return self::get($args, true);
    }
}
