<?php

namespace RtmMail;

use RtmMail\Helpers\EventHelper;
use RtmMail\Helpers\LogHelper;

/**
 * EventDetails - handles every user interaction for the event details page
 */
class EventDetails extends AbstractPage
{
    protected $slug = 'event_details';

    /**
     * event_logs - all events by log_id
     * @var object
     */
    private $event_logs;
    /**
     * mail_log - single log object from the database
     * @var object
     */
    private $mail_log;

    public function __construct()
    {
        parent::__construct();
    }

    /**
     * handle_requests - every page can handle their requests (post/get)
     */
    public function handle_requests()
    {
        // Check if user is on the details page
        if ($this->page_check()) {
            // If someone goes to the details page with a log_id
            if (isset($_GET['log_id'])) {
                // sanitize the log_id
                $log_id = sanitize_key($_GET['log_id']);
                // Get the log by the log_id and set it to the current log item
                $this->mail_log = LogHelper::get([
                    'post_per_page' => null,
                    'where' => [
                        'id' => [
                            'type' => '=',
                            'value' => $log_id,
                        ]
                    ]
                ]);
                // Get all events by log_id
                $this->event_logs = EventHelper::get([
                    'post_per_page' => null,
                    'search_filter' => [
                        'log_id' => true,
                    ],
                    's' => $log_id,
                ]);
            }
        }
    }

    /**
     * display - Displays the page view for the user
     */
    public function display()
    {
        // Check if user is on the details page
        if ($this->page_check()) {
            global $events;
            global $log;
            // set global $log to the current loaded log item
            $log = $this->mail_log;
            // set global $events to the current debug events logs
            $events = $this->event_logs;
            // load in the details view for the user
            require __DIR__ . '/Views/' . $this->slug . '.php';
        }
    }

    /**
     * build_dump - Builds the request debug block
     * @return string - HTML code block
     */
    public function build_dump($request)
    {
        $html = '<div class="request-dump-block">';
        foreach ($request as $request_type => $request_data) {
            if (!empty($request_data)) {
                $html .= '<div class="request-dump-item">';
                $html .= '<p class="request-item-label">' . $request_type . '</p>';
                $html .= '<pre class="request-item-data">';
                $html .= $this->build_request_data($request_data);
                $html .= '</pre>';
                $html .= '</div>';
            }
        }
        $html .= '</div>';
        return $html;
    }

    /**
     * build_request_data - Attach all request values to the request debug block
     * @param $data - Incoming request data
     * @param string $html - Current html block (for recursion)
     * @param int $level - Current level of the request value (dents the values to the right)
     * @return string - HTML code block
     */
    private function build_request_data($data, $html = '', $level = 1)
    {
        if (empty($html)) {
            $html = '<p><span class="request-dump-type">' . gettype($data) . ' (' . count($data) . ')</span> ';
            $html .= is_array($data) ? '[</p>' : '{</p>';
        }
        $html .= '<div class="request-dump-data" style="margin-left: calc(7px * ' . $level . ');">';
        foreach ($data as $key => $value) {
            $html .= '<p>"<span class="request-dump-key">' . $key . '</span>" => ';
            if (is_array($value) || is_object($value)) {
                $html .= '<span class="request-dump-type">' . gettype($value) . ' (' . count($value) . ')</span> ';
                $html .= is_array($value) ? '[</p>' : '{</p>';
                $html = $this->build_request_data($value, $html, $level + 1);
            } else {
                $value = is_string($value) ? '"<span class="request-dump-value">' . $value . '</span>"' : '<span class="request-dump-value">' . $value . '</span>';
                $html .= $value . '</p>';
            }
        }
        $html .= '</div>';
        $html .= is_array($data) ? '<p>]</p>' : '<p>}</p>';
        return $html;
    }
}
