<?php

require_once dirname(__FILE__).'/migrations/MigrationInterface.php';

class TaalthuisDatabaseMigrator
{

    const DATABASE_VERSION = "1.0";

    public function __construct()
    {
        global $migration_obj, $rtm_ae_migrations;
        $rtm_ae_migrations = [];

        foreach (scandir(dirname(__FILE__).'/migrations/') as $filename) {
            if ($filename === 'MigrationInterface.php') {
                continue;
            }
            $path = dirname(__FILE__) . '/migrations/' . $filename;
            if (is_file($path)) {
                require_once $path;

                $rtm_ae_migrations[$migration_obj->get_name()] = $migration_obj;
            }
        }

        register_activation_hook(__FILE__, [$this, 'check_migrations']);
        register_deactivation_hook(__FILE__, [$this, 'rollback_migrations']);
        add_action('init', [$this, 'convert_records'], PHP_INT_MAX);
    }

    public function rollback_migrations() {
        $opt_val = get_option('rtm-ae-rollback-on-deactivate', false);
        if ($opt_val === true) {
            global $rtm_ae_migrations;
            $migrations_done = maybe_unserialize(get_option('rtm-ae-migrations', []));
            foreach ($rtm_ae_migrations as $migration_name => $migration) {
                if (in_array($migration_name, $migrations_done)) {
                    $migration->rollback();
                    $migrations_done[] = $migration_name;
                }
            }

            delete_option('rtm-ae-migrations');
            delete_option('rtm-ae-version');
            delete_option('rtm-ae-delete-db-on-deactivate');
        }
    }

    public function check_migrations()
    {
        $opt_val = get_option('rtm-ae-version', false);
        if ($opt_val === false || $opt_val !== self::DATABASE_VERSION) {
            $this->migrate();
            update_option('rtm-ae-version', self::DATABASE_VERSION);
            $this->convert_records();
        }
    }

    public function migrate()
    {
        global $rtm_ae_migrations;
        $migrations_done = maybe_unserialize(get_option('rtm-ae-migrations', []));
        foreach ($rtm_ae_migrations as $migration_name => $migration) {
            if (!in_array($migration_name, $migrations_done)) {
                $migration->migrate();
                $migrations_done[] = $migration_name;
            }
        }

        update_option('rtm-ae-migrations', $migrations_done);
    }

    public function convert_records()
    {
        if (is_ajax() || wp_doing_cron()) {
            return;
        }
        if (get_option('rtm-ae-version', false) === false
            || empty(maybe_unserialize(get_option('rtm-ae-migrations', [])))) {
            $this->check_migrations();
        }

        /** @var WC_Product[] $products */
        $lastest = wc_get_products([
            'order' => 'desc',
            'limit' => 1,
            'return' => 'ids'
        ]);
        if (empty($lastest) || tt_get_course_by_product($lastest[0])) {
            return;
        }

        $products = wc_get_products([
            'limit' => -1,
            'order' => 'asc'
        ]);

        foreach ($products as $product) {
            $attr = $product->get_attribute('pa_kind');
            if ($attr === 'Course') {
                $post_id = $product->get_id();
                $start = (new DateTime())->setTimestamp(strtotime(get_post_meta($post_id, 'start', true)));
                $end = (new DateTime())->setTimestamp(strtotime(get_post_meta($post_id, 'end', true)));

                tt_update_course_data(new Course([
                    'product' => $product,
                    'start_date' => $start,
                    'end_date' => $end
                ]));
            }
        }

    }

}

new TaalthuisDatabaseMigrator();