<?php

/**
 * Flush rewrite rules on plugin activation.
 */
function courses_flush_rewrite_rules()
{
    add_rewrite_endpoint('my-courses', EP_ROOT | EP_PAGES);
    flush_rewrite_rules();
}

register_activation_hook(__FILE__, 'courses_flush_rewrite_rules');
register_deactivation_hook(__FILE__, 'courses_flush_rewrite_rules');

/**
 * Register new endpoint to use inside My Account page.
 *
 * @see https://developer.wordpress.org/reference/functions/add_rewrite_endpoint/
 */
function my_courses_endpoints()
{
    add_rewrite_endpoint('my-courses', EP_ROOT | EP_PAGES);
}

add_action('init', 'my_courses_endpoints');

/**
 * Add my-courses query var.
 *
 * @param array $vars
 * @return array
 */
function my_courses_query_vars($vars)
{
    $vars[] = 'my-courses';

    return $vars;
}

add_filter('query_vars', 'my_courses_query_vars', 0);

function add_courses_item($items)
{
    $before = array_splice($items, 0, 1);

    return $before + ['my-courses' => __('My Courses')] + $items;
}

add_filter('woocommerce_account_menu_items', 'add_courses_item');

function render_courses_page()
{
    if (!empty(get_query_var('my-courses'))) {
        wc_get_template('myaccount/course.php');
    } else {
        wc_get_template('myaccount/my-courses.php');
    }
}

add_action('woocommerce_account_my-courses_endpoint', 'render_courses_page');

function taalthuis_get_my_courses(WP_User $wp_user)
{
    if ($wp_user) {
        $orders = wc_get_orders([
            'customer_id' => $wp_user->ID,
            'status' => array_diff(wc_get_order_statuses(), ['wc-cancelled', 'wc-refund'])
        ]);

        $courses = array_reduce($orders, function (array $carry, WC_Order $order) {
            $items = $order->get_items();
            foreach ($items as $item) {
                $prod_id = $item->get_product_id();
                $course = tt_get_course_by_product($prod_id);
                if ($course) {
                    $carry[] = $course;
                }
            }
            return $carry;
        }, []);

        return $courses;
    }
    return [];
}

add_filter('taalthuis_get_my_courses', 'taalthuis_get_my_courses');


function taalthuis_render_courses($courses)
{
    if (empty($courses)) {
        $courses = taalthuis_get_my_courses(wp_get_current_user());
    }
    ?>
    <h2>My Courses</h2>
    <p>
        In this section all of your current and previous courses are listed.
    </p>

    <div class="my-courses-container">
        <?php

        foreach ($courses as $course) {
            do_action('taalthuis_render_course', $course);
        }

        ?>
    </div>
    <?php
}

add_action('taalthuis_render_courses', 'taalthuis_render_courses');


function taalthuis_render_course(Course $course)
{
    $product = $course->get_product();
    $post = get_post($product->get_id());
    ?>
    <div class="my-course">
        <div class="course-title">
            <?= $product->get_name() . ' (' . $course->get_start_date()->format('Y') . ')' ?>
            <?php $now = (new DateTime('now'))->getTimestamp();
            if ($course->get_start_date()->getTimestamp() > $now) {
                echo '<span class="course-upcoming">upcoming</span>';
            } elseif ($course->get_start_date()->getTimestamp() < $now && $course->get_end_date()->getTimestamp() > $now) {
                echo '<span class="course-upcoming">current</span>';
            }
            ?>
            <span class="links">
                <?php
                $links = apply_filters('taalthuis_my_course_course_links', [], $course);

                foreach ($links as $id => $link) { ?>
                    <a href="<?= $link['url'] ?? '' ?>" class="<?= $link['classes'] ?? '' ?> link">
                        <?= $link['label'] ?? $id ?? '' ?>
                    </a>
                <?php } ?>
            </span>
        </div>
        <div class="course-desc">
            <?= do_shortcode("[course_dates product={$product->get_id()}]") ?>
        </div>
    </div>
    <?php
}

add_action('taalthuis_render_course', 'taalthuis_render_course');

function taalthuis_my_course_course_links($links, Course $course)
{
    if (is_account_page()) {
        $links['more'] = [
            'label' => __('More about this course', 'taalthuis'),
            'url' => wc_get_account_endpoint_url('my-courses') . $course->get_id(),
        ];

        global $wp;
        /** @var $wp wp */
        $parts = explode('/', $wp->request);
        if (end($parts) !== 'my-courses') {
            $links['my-courses'] = [
                'label' => __('View all your courses', 'taalthuis'),
                'url' => wc_get_account_endpoint_url('my-courses'),
            ];
        }
    }

    return $links;
}

add_filter('taalthuis_my_course_course_links', 'taalthuis_my_course_course_links', 10, 2);


add_action('woocommerce_account_dashboard', 'add_dashboard_course_section');

function add_dashboard_course_section()
{
    $user = wp_get_current_user();

    $orders = wc_get_orders([
        'customer_id' => $user->ID,
        'status' => array_diff(wc_get_order_statuses(), ['wc-cancelled', 'wc-refund'])
    ]);

    $latest_order = $orders[0];
    foreach ($latest_order->get_items() as $item) {
        $product = wc_get_product($item->get_product_id());
        if ($product->get_attribute('pa_kind') === 'Course') {
            break;
        }
    }

    $course = tt_get_course_by_product($product);

    $upcoming = 'latest';
    $timestamp = (new DateTime('now'))->getTimestamp();
    if ($course->get_start_date()->getTimestamp() > $timestamp) {
        $upcoming = 'upcoming';
    } elseif ($course->get_end_date()->getTimestamp() > $timestamp) {
        $upcoming = 'current';
    }

    ?>
    <h2 style="margin-top: 1rem;">Your <?= $upcoming ?> course:</h2>

    <div class="my-courses-container">
        <?php do_action('taalthuis_render_course', $course); ?>
    </div>
    <?php
}

function my_courses_enqueue_scripts()
{
    $basepath = ACADEMIA_EXTENSIONS_ID . DIRECTORY_SEPARATOR;
    wp_enqueue_style('taalthuis-my-courses', plugins_url($basepath . 'assets/css/my-courses.css'), [], '1.0.0');
}

add_action('wp_enqueue_scripts', 'my_courses_enqueue_scripts');



add_action('taalthuis_account_course', 'taalthuis_account_course');

