/*!
 * Van Ons Gruntfile
 * http://www.van-ons.nl/
 * @author Donny Oexman <donny@van-ons.nl>
 */

'use strict';

module.exports = function (grunt) {
  require('load-grunt-tasks')(grunt);
  require('time-grunt')(grunt);
  const sass = require('node-sass');

  /**
   * Grunt configuration
   */
  grunt.initConfig({
    pkg: grunt.file.readJSON('package.json'),

    /**
     * Set project info
     */
    project: {
      src: {
        dir: './src',
        sass: '<%= project.src.dir %>/scss',
        js: '<%= project.src.dir %>/js',
        svg: '<%= project.src.dir %>/svg'
      },
      app: {
        dir: '.',
        assets: './assets',
        js: '<%= project.app.assets %>/js',
        components: '<%= project.app.assets %>/components',
        images: '<%= project.app.assets %>/images',
        svg: '<%= project.app.assets %>/svg',
        fonts: '<%= project.app.assets %>/fonts'
      },
      banner: {
        general: '/*!\n' +
          ' * <%= pkg.title %>\n' +
          ' * <%= pkg.url %>\n' +
          ' * @author <%= pkg.author.name %> <<%= pkg.author.email %>> (<%= pkg.author.url %>)\n' +
          ' * @version <%= pkg.version %>\n' +
          ' * Copyright <%= pkg.copyright %> <%= pkg.author.name %>.\n' +
          ' */\n',
        theme: '/*!\n' +
          ' * Theme Name: <%= pkg.title %>\n' +
          ' * Theme URI: <%= pkg.url %>\n' +
          ' * Description: <%= pkg.title %> theme created by <%= pkg.author.name %>.\n' +
          ' * Author: <%= pkg.author.name %>\n' +
          ' * Author URI: <%= pkg.author.url %>\n' +
          ' * Version: <%= pkg.version %>\n' +
          ' * Text Domain: <%= pkg.name %>\n' +
          ' */\n'
      }
    },

    /*
     * Set notify hooks options
     */
    notify_hooks: {
      options: {
        enabled: true,
        max_jshint_notifications: 5,
        title: "<%= pkg.title %>"
      }
    },

    /**
     * JSHint
     * https://github.com/gruntjs/grunt-contrib-jshint
     * Validates JS files
     */
    jshint: {
      options: {
        jshintrc: true
      },
      dev: {
        options: {
          "devel": true
        },
        files: {
          src: ['<%= project.src.js %>/*.js']
        }
      },
      dist: {
        options: {
          "devel": false
        },
        files: {
          src: ['<%= project.src.js %>/*.js']
        }
      }
    },

    /**
     * Uglify
     * https://github.com/gruntjs/grunt-contrib-uglify
     * Compresses and minifies all JavaScript files into one
     */
    uglify: {
      dist: {
        options: {
          banner: '<%= project.banner.general %>'
        },
        files: {
          '<%= project.app.js %>/app.min.js': '<%= project.src.js %>/app.js'
        }
      }
    },

    /**
     * ImageMIN
     * https://github.com/gruntjs/grunt-contrib-imagemin
     * Minifies images
     */
    imagemin: {
      svg: {
        options: {
          svgoPlugins: [{
            removeViewBox: false
          }, {
            removeUselessStrokeAndFill: false
          }, {
            sortAttrs: true
          }, {
            removeDimensions: true
          }, {
            removeTitle: true
          }]
        },
        files: [{
          expand: true,
          cwd: '<%= project.src.svg %>/',
          src: '{,*/}*.svg',
          dest: '<%= project.app.svg %>/',
          rename: function (dest, src) {
            if (src.indexOf('inline') >= 0) {
              return dest + src + '.php';
            } else {
              return dest + src;
            }
          }
        }]
      },
      dist: {
        options: {
          optimizationLevel: 7,
          cache: false
        },
        files: [{
          expand: true,
          cwd: '<%= project.src.images %>/',
          src: '{,*/}*.{png,jpg,jpeg,gif}',
          dest: '<%= project.app.images %>/'
        }]
      }
    },

    /**
     * SASS
     * https://github.com/sindresorhus/grunt-sass
     * Compiles SASS/SCSS to CSS
     */
    sass: {
      options: {
        implementation: sass,
        includePaths: [
          '<%= project.app.components %>/compass-mixins/lib',
          '<%= project.app.components %>/foundation/scss',
          '<%= project.app.components %>/fontawesome/scss'
        ],
        imagePath: '<%= project.app.images %>'
      },
      dev: {
        options: {
          outputStyle: 'nested',
          sourceMap: true
        },
        files: {
          '<%= project.app.dir %>/style.css': '<%= project.src.sass %>/style.scss'
        }
      },
      dist: {
        options: {
          outputStyle: 'compressed'
        },
        files: {
          '<%= project.app.dir %>/style.css': '<%= project.src.sass %>/style.scss'
        }
      }
    },

    postcss: {
      options: {
        processors: [
          require('autoprefixer')({
            browsers: ['last 3 versions', 'ie >= 9', 'and_chr >= 2.3']
          })
        ]
      },
      dev: {
        options: {
          map: true
        },
        src: '<%= project.app.dir %>/*.css'
      },
      dist: {
        src: '<%= project.app.dir %>/*.css'
      }
    },

    usebanner: {
      options: {
        position: 'top',
        banner: '<%= project.banner.theme %>',
        linebreak: true
      },
      files: {
        src: ['<%= project.app.dir %>/style.css', '<%= project.app.dir %>/style.css']
      }
    },

    /**
     * Notify
     * https://github.com/dylang/grunt-notify
     * Notify when a task is succesful
     */
    notify: {
      js: {
        options: {
          title: 'JavaScript',
          message: 'Minified and checked successfully.'
        }
      },
      sass: {
        options: {
          title: 'SASS',
          message: 'Compiled and moved successfully.'
        }
      },
      dev: {
        options: {
          title: 'Development mode',
          message: 'Development tasks succesfully executed.'
        }
      },
      dist: {
        options: {
          title: 'Production mode',
          message: 'Production tasks succesfully executed.'
        }
      }
    },

    /**
     * Watch
     * https://github.com/gruntjs/grunt-contrib-watch
     * Watches certain files and runs relevant tasks
     */
    watch: {
      options: {
        spawn: false,
        debounceDelay: 1
      },
      grunt: {
        files: ['Gruntfile.js'],
        tasks: ['default']
      },
      js: {
        files: '<%= project.src.js %>/{,*/}*.js',
        tasks: ['jshint:dev', 'uglify:dist', 'notify:js']
      },
      sass: {
        files: '<%= project.src.sass %>/{,*/}*.scss',
        tasks: ['sass:dev', 'postcss:dev', 'usebanner', 'notify:sass']
      }
    }
  });

  /**
   * 'Default' tasks
   * Run `grunt` in the command line
   */
  grunt.registerTask('default', [
    'jshint:dev',
    'uglify:dist',
    'sass:dev',
    'postcss:dev',
    'usebanner',
    'notify:dev',
    'watch'
  ]);

  /**
   * 'Build' tasks
   * Run `grunt build` in the command line
   */
  grunt.registerTask('build', [
    'preen',
    'jshint:dist',
    'uglify',
    'sass:dist',
    'postcss:dist',
    'usebanner',
    'imagemin',
    'notify:dist'
  ]);

  /**
   * Runs the task 'notify_hooks'
   * Required for custom notify options
   */
  grunt.task.run('notify_hooks');
};