<?php
/*
 Plugin Name: WooCommerce Price Based on Country Pro Add-on
 Description: Supercharge Price Based on Country with professionals features
 Author: Oscar Gare
 Version: 2.1.8
 Author URI: https://www.linkedin.com/in/oscargare
 Text Domain: wc-price-based-country
 License: GPLv2
*/

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

include_once( 'includes/class-wcpbc-dependencies.php' );	

if ( ! class_exists( 'WC_Product_Price_Based_Country_Pro' ) ) :

/**
 *
 * @class WC_Product_Price_Based_Country_Pro
 * @version	2.1.7
 */
class WC_Product_Price_Based_Country_Pro {
	
	/**
	 * @var string
	 */
	public static $version = '2.1.8';	
	
	/**
	 * Get the plugin url.
	 * @return string
	 */
	public static function plugin_url() {		
		return plugin_dir_url( __FILE__ );
	}

	/**
	 * Get the plugin path.
	 * @return string
	 */
	
	public static function plugin_path(){
		return plugin_dir_path( __FILE__ );
	}

	/**
	 * Init plugin, Hook actions and filters
	 *
	 * @since 1.0
	 */
	public static function init() {				

		include_once( self::plugin_path() . 'includes/class-wcpbc-exchange-rates-provider.php' );			
		include_once( self::plugin_path() . 'includes/class-wcpbc-update-exchange-rates.php' );					
		include_once( self::plugin_path() . 'includes/class-wcpbc-frontend-currency.php' ); 
		include_once( self::plugin_path() . 'includes/class-wcpbc-integrations-pro.php' );		
		include_once( self::plugin_path() . 'includes/admin/wc-plugin-api-wrapper.php' );		
		include_once( self::plugin_path() . 'includes/admin/class-wcpbc-plugin-updater.php' );		
		
		register_activation_hook( __FILE__, array( __CLASS__, 'install' ) );		
		register_deactivation_hook( __FILE__, array( __CLASS__, 'uninstall' ) );
		add_action( 'init', array( __CLASS__ , 'create_cron_jobs' ) );
		add_action( 'wc_price_based_country_admin_init', array( __CLASS__ , 'admin_includes' ) );				
		add_action( 'plugins_loaded', array( __CLASS__, 'include_settings'), 10 );		
		add_action( 'plugins_loaded', array( __CLASS__, 'updater'), 20 );		
		add_action( 'admin_init', array( __CLASS__ , 'check_license_key' ) );				
		add_action( 'widgets_init', array( __CLASS__, 'register_widgets'), 20 );							
		add_action( 'wc_price_based_country_pro_cron', array( __CLASS__, 'cron') );		

		add_shortcode( 'wcpbc_currency_switcher', array( __CLASS__, 'currency_switcher'), 20 );						
	}		

	/**
	 * Update exchange rates
	 *
	 * @since 1.1
	 * @return void
	 */
	public static function install() {
		
		// Update currency format option
		
		$currency_format = get_option( 'wc_price_based_currency_format', false );
		if ( ! $currency_format ) {
			$currency_pos = get_option( 'woocommerce_currency_pos' );
			$format = '[symbol][price]';

			switch ( $currency_pos ) {
				case 'left' :
					$format = '[symbol][price]';
				break;
				case 'right' :
					$format = '[price][symbol]';
				break;
				case 'left_space' :
					$format = '[symbol]&nbsp;[price]';
				break;
				case 'right_space' :
					$format = '[price]&nbsp;[symbol]';
				break;
			}

			update_option( 'wc_price_based_currency_format', $format );
		}
		
		// WooCommerce subscriptions supports
		delete_transient('wc_report_subscription_events_by_date');
		delete_transient('wc_report_upcoming_recurring_revenue');		
		
		// Trigger action
		do_action( 'wc_price_based_country_pro_installed' );	
	}

	/**
	 * Create cron jobs (clear them first).
	 */
	public static function create_cron_jobs() {				
		if (! wp_next_scheduled( 'wc_price_based_country_pro_cron' ) ) {			
			$ve = get_option( 'gmt_offset' ) > 0 ? '-' : '+';
			wp_schedule_event( strtotime( '00:00 tomorrow ' . $ve . get_option( 'gmt_offset' ) . ' HOURS' ), 'daily', 'wc_price_based_country_pro_cron' );			
		}
	}

	/**
	 * Clear cron on plugin deactivation
	 */
	public static function uninstall() {
		wp_clear_scheduled_hook( 'wc_price_based_country_pro_cron' );		
	}

	/**
	 * Admin includes	 	
	 */
	public static function admin_includes() {		
		include_once( self::plugin_path() . 'includes/admin/class-wcpbc-admin-pro.php' ); 				
	}

	/**
	 * Include settings
	 */
	public static function include_settings() {		
		include_once( self::plugin_path() . 'includes/admin/class-wcpbc-license-settings.php' ); 
	}

	/**
	 * Init plugin updater
	 */
	public static function updater() {		
		if( ! is_admin() || ! class_exists( 'WCPBC_License_Settings' ) ) {
			return;
		}		

		$license_key = WCPBC_License_Settings::instance()->get_option( 'license_key', false );
		$api_key 	 = WCPBC_License_Settings::instance()->get_option( 'api_key',false );		

		if ( ! $license_key || ! $api_key ) {
			return;			
		}

		$updater = new WCPBC_Plugin_Updater( __FILE__, self::$version, array(
			'item_id'     => 1450,			
			'license_key' => $license_key,
			'api_key' 	  => $api_key
		) );
		
	}

	/**
	 * Check license is active
	 */
	public static function check_license_key() {
		$license_settings = get_option( 'wc_price_based_country_license_settings', false );			
		if ( empty( $license_settings['license_key'] ) ) {
			add_action( 'admin_notices', array( __CLASS__, 'no_license_key' ) );
		}
	}		

	/**
	 * Dialy cron tasks
	 */
	public static function cron() {
		if( ! class_exists( 'WCPBC_License_Settings' ) ) {
			return;
		}	
		
		if ( ! WCPBC_License_Settings::instance()->is_status_active() && WCPBC_License_Settings::instance()->get_license_key() ) {						
			WCPBC_License_Settings::instance()->clear_admin_options();
		}		
	}		
	
	/**
	 * Register Widgets
	 *	 
	 */
	public static function register_widgets(){	 	
	 	if ( class_exists( 'WC_Widget' ) ) {
	 		include_once( 'includes/class-wcpbc-widget-currency-switcher.php' );	
	 		register_widget( 'WCPBC_Widget_Currency_Switcher' );
	 	}
	}

	/**
	 * Output Currency Switcher
	 *
	 * @param array $atts
	 * @return string
	 */
	public static function currency_switcher( $atts ) {

		$atts = shortcode_atts( array(			
			'currency_display_style'  	=> '[name] ([symbol])',
			'title' 					=> ''
		), $atts, 'wcpbc_currency_switcher' );

		$atts['currency_display_style'] = str_replace( array( '{', '}'), array('[',']'), $atts['currency_display_style'] );
		
		ob_start();

		the_widget( 'WCPBC_Widget_Currency_Switcher', $atts );

		return ob_get_clean();
	}
	
	/**
	 * WooCommerce Price Based Country inactive notice. 
	 *
	 * @since  2.0.0
	 */
	public static function dependency_inactive_notice() {
		if ( current_user_can( 'activate_plugins' ) ) {
			echo '<div id="message" class="error"><p>';
			printf( __( '%1$sWooCommerce Price Based Country Pro Add-on is inactive%2$s. %3$sWooCommerce Price Based Country plugin %4$s must be active for work. Please %5$sinstall and activate WooCommerce Price Based Country &raquo;%6$s', 'wc-price-based-country' ), '<strong>', '</strong>', '<a href="https://wordpress.org/plugins/woocommerce-product-price-based-on-countries/">', '</a>', '<a href="' . esc_url( admin_url( 'plugins.php' ) ) . '">', '</a>' );
			echo '</p></div>';
		}
	}

	/**
	 * No license key notice
	 *
	 * @since  2.0.0
	 */
	public static function no_license_key() {
		if ( current_user_can( 'activate_plugins' ) && ! ( isset( $_GET[ 'tab' ] ) && $_GET[ 'tab' ] === 'price-based-country' && isset( $_GET[ 'section' ] ) &&  $_GET[ 'section' ] === 'license' ) ) {
			echo '<div id="message" class="error"><p>';
			printf( __( '%1$sWarning!%2$s You didn\'t set your %1$sWooCommerce Price Based Country Pro Add-on%2$s license key yet, which means you\'re missing out on updates and support! Enter your %3$slicense key%4$s.', 'wc-price-based-country' ), '<strong>', '</strong>', '<a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=price-based-country&section=license' ) ) . '">', '</a>' );
			echo '</p></div>';
		}
	}
}

endif;

// Init plugin
if ( is_woocommerce_price_based_country_active() ) {	
	
	WC_Product_Price_Based_Country_Pro::init();
	
} else {
	add_action( 'admin_notices', array( 'WC_Product_Price_Based_Country_Pro', 'dependency_inactive_notice' ) );
}

