<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if( ! class_exists( 'WCPBC_Plugin_Updater' ) ) :

/**
 * Allows plugins to use their own update API.
 *
 * @author oscargare
 * @version 2.1.7
 */

class WCPBC_Plugin_Updater {
    	    
	private $name      = '';
	private $slug      = '';
	private $version   = '';
	private $api_data  = array();
	private $cache_key = '';

	/**
 	 * Class constructor.
 	 *
	 * @param string  $_plugin_file Path to the plugin file.
 	 * @param array   $_api_data    Optional data to send with API calls.
 	 */
	public function __construct( $_plugin_file, $_version, $_api_data = array() ) {
		    		
		$this->name 	= plugin_basename( $_plugin_file );
		$this->slug 	= basename( $_plugin_file, '.php' );
		$this->version 	= $_version;
		$this->api_data = wp_parse_args( $_api_data, array(
			'item_id'     => 0,    			
			'license_key' => '',
			'api_key' 	  => ''
		) );

		$this->cache_key = 'update_check_' . md5( $this->slug . serialize( $this->api_data ) );

		// Set up hooks.
		$this->init();

	}

	/**
	 * Set up WordPress filters to hook into WP's update process.
	 *
	 * @return void
	 */
	public function init() {
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_update' ) );
		add_filter( 'plugins_api', array( $this, 'plugin_api_call' ), 10, 3 );
	}

	/**
	 * Check for Updates at the defined API endpoint and modify the update array.
	 *
	 * This function dives into the update API just when WordPress creates its update array,
	 * then adds a custom API call and injects the custom plugin data retrieved from the API.
	 * It is reassembled from parts of the native WordPress plugin update code.
	 * See wp-includes/update.php line 121 for the original wp_update_plugins() function.
	 *
	 * @uses WC_Plugin_API_Wrapper::update_check()
	 *
	 * @param array  $_transient_data Update array build by WordPress.
	 * @return array Modified update array with custom plugin data.
	 */
	public function check_update( $_transient_data ) {

		if ( ! is_object( $_transient_data ) ) {
			$_transient_data = new stdClass;
		}

		if ( ! empty( $_transient_data->response ) && ! empty( $_transient_data->response[ $this->name ] ) ) {
			return $_transient_data;
		}
		
		if ( false === ( $data = get_transient( $this->cache_key ) ) ) {

			$data = WC_Plugin_API_Wrapper::update_check(
				$this->api_data['item_id'],
				$this->api_data['api_key'],
				$this->api_data['license_key']
			);

			set_transient( $this->cache_key, $data, 3 * HOUR_IN_SECONDS );				
		}

		if ( ! is_wp_error( $data ) ) {

			if ( version_compare( $this->version, $data->new_version, '<' ) ) {

				$data->package = add_query_arg( 'token', $this->api_data[ 'api_key' ], $data->package );

				$_transient_data->response[ $this->name ] = $data;
			}

			$_transient_data->last_checked           = current_time( 'timestamp' );
			$_transient_data->checked[ $this->name ] = $this->version;				
		}

		return $_transient_data;			
	}		
	
	/**
	 * Updates information on the "View version x.x details" page with custom data.
	 *
	 * @uses api_request()
	 *
	 * @param mixed   $def
	 * @param string  $action
	 * @param object  $args
	 * @return object
	 */
	public function plugin_api_call( $def, $action, $args ) {
		if ( $action != 'plugin_information' ) {
			return $def;
		}

		if( ! isset( $args->slug ) || $args->slug != $this->slug ) {
            return $def;    		            
        }

		$response = WC_Plugin_API_Wrapper::plugin_information(
            $this->api_data[ 'item_id' ],
			$this->api_data[ 'api_key' ],
			$this->api_data[ 'license_key' ]
        );
		    		
		return $response;
	}                            
	
	function debug_result( $res, $action, $args ) {
		echo '<pre>'.print_r($res,true).'</pre>';
		return $res;
	}
}

endif;
