<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'WCPBC_Admin_Pro' ) ) :

/**
 *
 * @class WCPBC_Admin_Pro
 * @version	2.1.6
 */
class WCPBC_Admin_Pro {
	
	/**
	 * Hook actions and filters
	 *
	 * @since 1.0
	 */
	public static function init() {
		add_filter( 'woocommerce_general_settings', array( __CLASS__ , 'currency_format_settings' ) );	
		add_filter( 'wc_price_based_country_default_region_data', array( __CLASS__ , 'default_region_data' )  );
		add_filter( 'wc_price_based_country_settings', array( __CLASS__ , 'exchange_provider_settings' ) );		
		add_filter( 'wc_price_based_country_table_region_column_currency', array( __CLASS__ , 'exchange_rate_column_description' ), 10, 3 );
		add_action( 'wc_price_based_country_before_region_data_save', array( __CLASS__ , 'update_exchange_rate_from_api' ) );
		add_action( 'wc_price_based_country_admin_region_fields', array( __CLASS__ , 'region_currency_options' ) );
		add_action( 'wc_price_based_country_get_sections', array( __CLASS__ , 'settings_sections' ) );
		add_action( 'woocommerce_variable_product_bulk_edit_actions', array( __CLASS__, 'variable_product_bulk_edit_actions' ) );
		add_action( 'woocommerce_bulk_edit_variations_default', array( __CLASS__, 'bulk_edit_variations_default' ), 10, 4 );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'load_admin_script' ) );	
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'load_admin_styles' ) );	
	}
	
	/**
	 * Add currency format setting.
	 *
	 * @return array
	 */
	public static function currency_format_settings( $settings ) {

		$general_settings = array();
		
		foreach ( $settings as $setting ) {
			if ( $setting['id'] === 'woocommerce_currency_pos' ) {
				
				$general_settings[] = array(			
					'title' 	=> __( 'Currency Format', 'wc-price-based-country' ),										
					'id'      	=> 'wc_price_based_currency_format',
					'desc'		=> sprintf( __( 'Preview: %s', 'wc-price-based-country'), '<code id="wc_price_based_currency_format_preview"></code>'),
					'desc_tip'	=> __( 'Enter the currency format. Supports the following placeholders: [code] = currency code, [symbol] = currency symbol, [price] = product price.', 'wc-price-based-country'),					
					'css'      	=> 'min-width:350px;',
					'default'  	=> '[symbol][price]',
					'type' 		=> 'text'
				);
			}

			$general_settings[] = $setting;
		}

		return $general_settings;
	}	 
	
	/**
	 * Add currency options to default region data
	 *
	 * @param  string $key
	 * @return array
	 */
	public static function default_region_data( $data ) {

		$data['auto_exchange_rate'] = 'no';		
		$data['currency_format'] 	= '';
		$data['price_thousand_sep'] = get_option( 'woocommerce_price_thousand_sep' );
		$data['price_decimal_sep'] 	= get_option( 'woocommerce_price_decimal_sep' );
		$data['price_num_decimals'] = get_option( 'woocommerce_price_num_decimals' );		
		
		return $data;
	}


	/**
	 * Add exchange rates api providers settings.
	 *
	 * @return array
	 */
	public static function exchange_provider_settings( $settings ) {
		 
		$options = array();
		foreach ( WCPBC_Update_Exchange_Rates::get_exchange_rates_providers() as $id => $provider ) {
			$options[$id] = $provider->get_name();
		}
		
		$pbc_settings = array();
		foreach ($settings as $setting ) {
			
			if ( $setting['type'] ==='sectionend' && $setting['id'] ==='general_options' ) {
				$pbc_settings[] = array(
					'title'    => __( 'Exchange rate API', 'wc-price-based-country' ),
					'desc'     => __( 'This controls which API provider will be used to exchange rates auto-updates.', 'wc-price-based-country'),
					'id'       => 'wc_price_based_country_exchange_rate_api',
					'default'  => current(array_keys($options)),
					'type'     => 'select',
					'class'    => 'wc-enhanced-select',				
					'desc_tip' =>  true,
					'options'  => $options			
				);
			}

			$pbc_settings[] = $setting;
		}
		
		return $pbc_settings;
	}
	
	/**
	 * Add auto update exchange rate description to region table currency column
	 *
	 * @return string
	 */
	public static function exchange_rate_column_description( $output, $region, $default_region_key ) {
		$description = '';

		if ( isset( $region['auto_exchange_rate'] ) &&  $region['auto_exchange_rate'] == 'yes') {
			$description = ' (auto)';
		} elseif( $region['key'] !== $default_region_key ) {
			$description = ' (manual)';
		}
		
		return substr($output, 0, strlen($output)- 7 ) . $description . '</span>';		

	}

	/**
	 * Update exchange rate from api before save if user select auto updates
	 *
	 * @return void
	 */
	public static function update_exchange_rate_from_api( ) {		
		
		if ( isset( $_POST['auto_exchange_rate'] ) &&  isset( $_POST['exchange_rate'] ) && isset( $_POST['currency'] ) && $_POST['auto_exchange_rate'] === 'yes' ) {			
			if ( $rate = WCPBC_Update_Exchange_Rates::get_exchange_rate_from_api( $_POST['currency'] ) ) {			
				$_POST['exchange_rate'] = $rate;
			} else {				
				WC_Admin_Settings::add_error( __( 'Errors on update exchange rate from API.', 'wc-price-based-country' ) );
			}
		}		
	}	

	/**
	 * Add avanced currency options to admin region
	 *
	 * @return void
	 */
	public static function region_currency_options( $region ) {		
		include('views/html-region-currency-options.php');
	}

	/**
	 * Add license settings section
	 * 
	 * @param array $sections
	 * @return array
	 */
	public static function settings_sections( $sections ) {
		$sections['license'] = __( 'License', 'wc-price-based-country' );
		return $sections;
	}

	/**
	 * Add variable bulk actions options	  	
	 */
	public static function variable_product_bulk_edit_actions() {

		$variable_actions = array(
			'regular_price' 		 => __( 'Set regular prices', 'woocommerce' ),
			'regular_price_increase' => __( 'Increase regular prices (fixed amount or percentage)', 'woocommerce' ),
			'regular_price_decrease' => __( 'Decrease regular prices (fixed amount or percentage)', 'woocommerce' ),
			'sale_price' 			 => __( 'Set sale prices', 'woocommerce' ),
			'sale_price_increase' 	 => __( 'Increase sale prices (fixed amount or percentage)', 'woocommerce' ),
			'sale_price_decrease' 	 => __( 'Decrease sale prices (fixed amount or percentage)', 'woocommerce' )
		);

		foreach ( WCPBC()->get_regions() as $zone_id => $zone ) {
			echo '<optgroup label="'. esc_attr( sprintf( __( '%s Pricing (%s)', 'wc-price-based-country' ), $zone['name'], get_woocommerce_currency_symbol( $zone['currency'] ) ) ) . '">';
			
			foreach ($variable_actions as $key => $label ) {
				echo '<option value="wcpbc_variable_bluck_edit" data-action="' . $key . '" data-zone-id="' .  esc_attr( $zone_id ) . '">' . $label . '</option>';
			}		
			echo '</optgroup>';
		}				
	}

	/**
	 * Process variations bulk edit
	 *
	 * @since 2.16
	 * @param string $bulk_action
	 * @param array $data
	 * @param int $product_id
	 * @param array $variations
	 */
	public static function bulk_edit_variations_default( $bulk_action, $data, $product_id, $variations ){
		
		if ( $bulk_action !== 'wcpbc_variable_bluck_edit' || ! isset( $data['value'] ) || empty( $data['action'] ) || empty( $data['zone_id'] ) || ! array_key_exists( $data['zone_id'], WCPBC()->get_regions() ) ){
			return;
		}
				
		if ( in_array( $data[ 'action' ], array( 'regular_price', 'sale_price' ) ) ) {
			self::variation_bulk_set_price( $variations, $data['zone_id'], $data[ 'action' ], wc_format_decimal( $data['value'] ) );
		}

		if ( strpos( $data['action'], '_increase' ) !== false || strpos( $data['action'], '_decrease' ) !== false ) {
			$field = str_replace( array( '_increase', '_decrease' ), '', $data['action'] );			
			if ( in_array( $field, array( 'regular_price', 'sale_price' ) ) ) {				
				$operator = str_replace( "{$field}_", '', $data['action'] ) == 'increase' ? '+' : '-';				
				self::variation_bulk_adjust_price( $variations, $data['zone_id'], $field, $data['value'], $operator );
			}
		}
	}

	/**
	 * Bulk set convenience function.
	 *
	 * @access private
	 * @since 2.16
	 * @param array $variations
	 * @param string $zone_id
	 * @param string $field
	 * @param string $value
	 */
	private static function variation_bulk_set_price( $variations, $zone_id, $field, $value ) {
		
		foreach ( $variations as $variation_id ) {
			update_post_meta( $variation_id, "_{$zone_id}_{$field}", $value );
			update_post_meta( $variation_id, "_{$zone_id}_price_method", 'manual' );			
		}
	}

	/**
	 * Bulk set convenience function.
	 *
	 * @access private
	 * @since 2.16
	 * @param array $variations
	 * @param string $zone_id
	 * @param string $field
	 * @param string $value
	 * @param string $operator
	 */
	private static function variation_bulk_adjust_price( $variations, $zone_id, $field, $value, $operator ) {
		
		foreach ( $variations as $variation_id ) {

			$field_value = get_post_meta( $variation_id, "_{$zone_id}_{$field}", true );			
			
			if ( '%' === substr( $value, -1 ) ) {
				$percent = wc_format_decimal( substr( $value, 0, -1 ) );				
				$field_value += ( ( $field_value / 100 ) * $percent ) * "{$operator}1";				
			} else {
				$field_value += wc_format_decimal( $value ) * "({$operator}1)";
			}

			update_post_meta( $variation_id, "_{$zone_id}_{$field}", $field_value );
			update_post_meta( $variation_id, "_{$zone_id}_price_method", 'manual' );			
		}
	}

	/**
	 * Enqueue admin scripts
	 *
	 * @return void
	 */	
	public static function load_admin_script() {	

		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

		wp_register_script( 'wc-price-based-country-pro-admin', WC_Product_Price_Based_Country_Pro::plugin_url() . 'assets/js/admin' . $suffix . '.js', array( 'woocommerce_admin' ), WC_Product_Price_Based_Country_Pro::$version, true );		
		wp_enqueue_script( 'wc-price-based-country-pro-admin' );
		wp_localize_script( 'wc-price-based-country-pro-admin' , 'wc_price_based_country_pro_admin_param', array(
			'i18n_enter_a_value' 				  => esc_js( __( 'Enter a value', 'woocommerce' ) ),
			'i18n_enter_a_value_fixed_or_percent' => esc_js( __( 'Enter a value (fixed or %)', 'woocommerce' ) )
		) );		
	}

	/**
	 * Enqueue admin styles
	 *
	 * @return void
	 */	
	public static function load_admin_styles() {
		// enqueue admin styles
		wp_enqueue_style( 'wc-price-based-country-pro-admin', WC_Product_Price_Based_Country_Pro::plugin_url() . 'assets/css/admin.css', array(), WC_Product_Price_Based_Country_Pro::$version );
	}		
}

endif;

WCPBC_Admin_Pro::init();