<?php

use Automattic\WooCommerce\Utilities\OrderUtil;

/**
* Class MC4WP_Ecommerce
*
* @since 4.0
*/
class MC4WP_Ecommerce
{

    /**
    * @const string
    */
    const META_KEY = 'mc4wp_updated_at';

    /**
    * @var MC4WP_Ecommerce_Object_Transformer
    */
    public $transformer;

    /**
    * @var string The ID of the store object in Mailchimp
    */
    private $store_id;

    /**
     * @var bool Whether this store has WooCommerce HPOS enabled
     * @since 4.10
     */
    private $hpos_enabled;

    const ERR_NO_ITEMS = 29001;
    const ERR_NO_EMAIL_ADDRESS = 29002;

    /**
    * Constructor
    *
    * @param string $store_id
    * @param MC4WP_Ecommerce_Object_Transformer $transformer
    */
    public function __construct($store_id, MC4WP_Ecommerce_Object_Transformer $transformer)
    {
        $this->store_id = $store_id;
        $this->transformer = $transformer;
        $this->hpos_enabled = class_exists(OrderUtil::class) && OrderUtil::custom_orders_table_usage_is_enabled();
    }

    /**
    * @param string $store_id
    */
    public function set_store_id($store_id)
    {
        $this->store_id = $store_id;
    }

    /**
    * Update the "last updated" settings to now.
    *
    * @param WC_Order|int $post_id
    */
    public function touch($post_id = 0)
    {
        if ($post_id instanceof WC_Order) {
        	/** @var WC_Order $order */
        	$order = $post_id;
        	$order->update_meta_data(self::META_KEY, date('c'));
        	$order->save();
        } else {
        	update_post_meta($post_id, self::META_KEY, date('c'));
        }

        mc4wp_ecommerce_update_settings(array( 'last_updated' => time() ));
    }

    /**
     * @param string $cart_id
     *
     * @return object
     * @throws MC4WP_API_Exception|Exception
     */
    public function get_cart($cart_id)
    {
        $api = $this->get_api();

        return $api->get_ecommerce_store_cart($this->store_id, $cart_id);
    }

    /**
     * Add OR update a cart in Mailchimp.
     *
     * @param string $cart_id
     * @param object|WC_Customer|WP_User $customer
     * @param array $cart_contents
     *
     * @return bool
     * @throws MC4WP_API_Exception|Exception
     */
    public function update_cart($cart_id, $customer, array $cart_contents)
    {
        /**
         * Determine if a cart will be sent to Mailchimp.
         *
         * @param bool $send
         * @param object|WC_Customer|WP_User $customer
         * @param array $cart_contents
         *
         * @return bool on true will be sent to Mailchimp otherwise not.
         */
        $send_to_mailchimp = apply_filters('mc4wp_ecommerce_send_cart_to_mailchimp', true, $customer, $cart_contents);
        if (! $send_to_mailchimp) {
            return false;
        }

        $api = $this->get_api();
        $store_id = $this->store_id;

        if (is_array($customer) && isset($customer['customer'])) {
            // For backwards compatibility with queue data from before MC4WP Premium v3.4
            $cart_data = $customer;
        } else {
            $customer_data = $this->transformer->customer($customer);
            $cart_data = $this->transformer->cart($customer_data, $cart_contents);
        }

        // add (or update) customer
        $customer_data = $api->add_ecommerce_store_customer($store_id, $cart_data['customer']);

        // replace customer object in cart data with array with just an id
        $cart_data['customer'] = array(
            'id' => $customer_data->id,
        );

        // add or update cart
        try {
            $cart_data = $api->update_ecommerce_store_cart($store_id, $cart_id, $cart_data);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            $cart_data = $api->add_ecommerce_store_cart($store_id, $cart_data);
        }

        $this->touch();

        return true;
    }

    /**
     * @param string $cart_id
     *
     * @return bool
     * @throws Exception
     */
    public function delete_cart($cart_id)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;
        $result = $api->delete_ecommerce_store_cart($store_id, $cart_id);
        $this->touch();
        return $result;
    }

    /**
     * @param WP_User|WC_Order|object $customer_data
     *
     * @return string
     * @throws MC4WP_API_Exception|Exception
     */
    private function add_or_update_customer($customer_data)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;

        // get customer data
        $customer_data = $this->transformer->customer($customer_data);

        // add or update customer
        $api->add_ecommerce_store_customer($store_id, $customer_data);

        $this->touch();

        return $customer_data['id'];
    }

    /**
     * @param WP_User|WC_Order|object $customer_data
     *
     * @return string
     * @throws MC4WP_API_Exception|Exception
     */
    public function update_customer($customer_data)
    {
        /**
         * Determine if a customer get sent to mailchimp.
         *
         * @param bool
         * @param WP_User|WC_Order|object
         *
         * @return bool on true it will be sent to Mailchimp on false not.
         */
        $send_to_mailchimp = apply_filters('mc4wp_ecommerce_send_customer_to_mailchimp', true, $customer_data);
        if (! $send_to_mailchimp) {
            return '';
        }

        $api = $this->get_api();
        $store_id = $this->store_id;

	    // allow short-circuiting this method by return false from the filter hook called in the transformer class
	    $customer_data = $this->transformer->customer($customer_data);
	    if (false === is_array($customer_data)) {
		    return false;
	    }

        // update customer
        $api->update_ecommerce_store_customer($store_id, $customer_data['id'], $customer_data);

        $this->touch();

        return $customer_data['id'];
    }

    /**
    * @param int|WC_Order $order
    * @return boolean
    * @throws MC4WP_API_Exception|Exception
    */
    public function update_order($order)
    {
        // get & validate order
        $order = wc_get_order($order);
        if (! $order instanceof WC_Order) {
            throw new Exception(sprintf("Order #%d is not a valid order ID.", $order));
        }

        /**
        * Filters whether the order should be sent to Mailchimp.
        *
        * @param boolean $send Whether to send the order to Mailchimp, defaults to true.
        * @param WC_Order $order The order object.
        * @return bool
        */
        $send_to_mailchimp = apply_filters('mc4wp_ecommerce_send_order_to_mailchimp', true, $order);
        if (! $send_to_mailchimp) {
            return false;
        }

        // add or update customer in Mailchimp
        $this->add_or_update_customer($order);

        // get order data
        $order_id = $this->transformer->order_id($order);

	    // allow short-circuiting this method by return false from the filter hook called in the transformer class
	    $data = $this->transformer->order($order);
        if (false === is_array($data)) {
        	return false;
        }

        // validate existence of products in order
        foreach ($data['lines'] as $key => $line) {
            $product = wc_get_product($line['product_id']);
            $product_variation = wc_get_product($line['product_variant_id']);
            if (! $product || ! $product_variation) {
                // product or variant does no longer exist, replace with a generic deleted product.
                $this->ensure_deleted_product();

                // replace ID with ID of the generic "deleted product"
                $data['lines'][$key]['product_id'] = 'deleted';
                $data['lines'][$key]['product_variant_id'] = 'deleted';
            }
        }

        // throw exception if order contains no lines
        if (empty($data['lines'])) {
            throw new Exception("Order contains no items.", self::ERR_NO_ITEMS);
        }

        // add OR update order in Mailchimp
        return $this->is_order_tracked($order) ? $this->order_update($order, $data) : $this->order_add($order, $data);
    }

    /**
    * @param int $order_id
    *
    * @return boolean
    *
    * @throws MC4WP_API_Exception|Exception
    */
    public function delete_order($order_id)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;

        try {
            $success = $api->delete_ecommerce_store_order($store_id, $order_id);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            // good, order already non-existing
            $success = true;
        }

   		if ($this->hpos_enabled) {
    		global $wpdb;
	   		$query = $wpdb->prepare("DELETE FROM {$wpdb->prefix}wc_orders_meta WHERE order_id = %d AND meta_key = %s", array( $order_id, self::META_KEY));
	  		$wpdb->query($query);
	  	} else {
			delete_post_meta($order_id, self::META_KEY);
	  	}

        $this->touch();

        return $success;
    }

    /**
    * @param WC_Order $order
    * @param array $data
    * @param bool $recurse
    *
    * @return bool
    *
    * @throws MC4WP_API_Exception|Exception
    */
    private function order_add(WC_Order $order, array $data, $recurse = true)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;

        // check for method existence first because wc pre-3.0
        $order_id = $this->transformer->order_id($order);
        $order_number = $this->transformer->order_number($order);

        try {
            $response = $api->add_ecommerce_store_order($store_id, $data);
        } catch (MC4WP_API_Exception $e) {
            // update order if it already exists
            if ($recurse && stripos($e->detail, 'order with the provided ID already exists') !== false) {
                return $this->order_update($order, $data, false);
            }

            // if campaign_id data is corrupted somehow, retry without campaign data.
            if (! empty($data['campaign_id']) && stripos($e->detail, 'campaign with the provided ID does not exist') !== false) {
                unset($data['campaign_id']);
                return $this->order_add($order, $data);
            }

            throw $e;
        }

        $this->touch($order);
        return true;
    }

    /**
     * @param WC_Order $order
     * @param array $data
     * @param bool $recurse
     * @return bool
     * @throws MC4WP_API_Exception|Exception
     */
    private function order_update(WC_Order $order, array $data, $recurse = true)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;

        // check for method existence first because wc pre-3.0
        $order_id = $this->transformer->order_id($order);
        $order_number = $this->transformer->order_number($order);

        try {
            // use order number here as that is what we send in order_add too.
            $response = $api->update_ecommerce_store_order($store_id, $order_number, $data);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            if ($recurse) {
                return $this->order_add($order, $data, false);
            }

            throw $e;
        } catch (MC4WP_API_Exception $e) {
            // if campaign_id data is corrupted somehow, retry without campaign data.
            if (! empty($data['campaign_id']) && stripos($e->detail, 'campaign with the provided ID does not exist') !== false) {
                unset($data['campaign_id']);
                return $this->order_update($order, $data);
            }

            throw $e;
        }


        $this->touch($order);
        return true;
    }

    /**
    * Since Mailchimp does not allow connecting two sites that share the same domain (they strip off the entire subdirectory part) we generate a unique domain here.
    *
    * @return string
    */
    private function get_site_domain()
    {
        $domain = str_ireplace(array( 'https://', 'http://', '://' ), '', get_home_url());

        if (is_multisite() && strpos($domain, '/') > strpos($domain, '.')) {
            $subdir_pos = strpos($domain, '/');
            $subdir = substr($domain, $subdir_pos + 1);
            $domain = substr($domain, 0, $subdir_pos);
            $domain = $subdir . '.' . $domain;
        }

        return $domain;
    }

    /**
    * Add or update store in Mailchimp.
    *
    * @param array $data
    * @throws MC4WP_API_Exception|Exception
    * @return object
    */
    public function update_store(array $data)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;

        $data['id'] = (string) $store_id;
        $data['platform'] = 'WooCommerce';
        $data['domain'] = $this->get_site_domain();
        $data['email_address'] = get_option('admin_email');
        $data['primary_locale'] = substr(get_locale(), 0, 2);
        $data['address'] = array(
            'address1' => get_option('woocommerce_store_address', ''),
            'address2' => get_option('woocommerce_store_address_2', ''),
            'city' => get_option('woocommerce_store_city', ''),
            'postal_code' => get_option('woocommerce_store_postcode', ''),
            'country_code' => get_option('woocommerce_default_country', ''),
        );

        // make sure we got a boolean value.
        if (isset($data['is_syncing'])) {
            $data['is_syncing'] = !!$data['is_syncing'];
        }

        /**
        * Filter the store data we send to Mailchimp.
        *
        * @param array $data
        */
        $data = apply_filters('mc4wp_ecommerce_store_data', $data);

        try {
            $res = $api->update_ecommerce_store($store_id, $data);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            $res = $api->add_ecommerce_store($data);
        } catch (MC4WP_API_Exception $e) {
            if ($e->status == 400 && stripos($e->detail, "list may not be changed") !== false) {
               	$this->delete_all_metadata();
                $api->delete_ecommerce_store($store_id);
                $res = $api->add_ecommerce_store($data);
            } else {
                throw $e;
            }
        }

        $this->touch();

        return $res;
    }

 	// delete all local tracking indicators (for both products and orders)
    private function delete_all_metadata() {
    	global $wpdb;

        delete_post_meta_by_key(MC4WP_Ecommerce::META_KEY);

        if ($this->hpos_enabled) {
        	$query = $wpdb->prepare("DELETE FROM {$wpdb->prefix}wc_orders_meta WHERE meta_key = %s", array( self::META_KEY));
  			$wpdb->query($query);
  		}
    }

    /**
     * @throws MC4WP_API_Exception|Exception
     */
    public function ensure_connected_site()
    {
        $api = $this->get_api();
        $client = $api->get_client();

        try {
            // first, query site to see if it exists
            $resource = sprintf('/connected-sites/%s', $this->store_id);
            $response = $client->get($resource, array());
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            // if it does not exist, add it
            $response = $client->post('/connected-sites', array(
                'foreign_id' => $this->store_id,
                'domain' => $this->get_site_domain(),
            ));
        }
    }

    /**
     * @throws MC4WP_API_Exception|Exception
     */
    public function verify_store_script_installation()
    {
        $api = $this->get_api();
        $client = $api->get_client();
        $resource = sprintf('/connected-sites/%s/actions/verify-script-installation', $this->store_id);
        $client->post($resource, array());
    }

    /**
    * Add or update a product + variants in Mailchimp.
    *
    * TODO: Mailchimp interface does not yet reflect product "updates".
    *
    * @param int|WC_Product $product Post object or post ID of the product.
    * @return boolean
    * @throws MC4WP_API_Exception|Exception
    */
    public function update_product($product)
    {
        $product = wc_get_product($product);

        // check if product exists
        if (! $product instanceof WC_Product) {
            throw new Exception(sprintf("#%d is not a valid product ID", $product));
        }

        $product_id = $this->transformer->product_id($product);

        // make sure product is not a product-variation
        if ($product instanceof WC_Product_Variation) {
            throw new Exception(sprintf("#%d is a variation of another product. Use the variable parent product instead.", $product_id));
        }

        $product_data = $this->transformer->product($product);
        $variants_data = $this->transformer->product_variants($product);
        return $this->is_object_tracked($product_id) ? $this->product_update($product, $product_data, $variants_data) : $this->product_add($product, $product_data, $variants_data);
    }

    /**
    * @param int $product_id
    * @return boolean
    *
    * @throws MC4WP_API_Exception|Exception
    */
    public function delete_product($product_id)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;

        try {
            $success = $api->delete_ecommerce_store_product($store_id, $product_id);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            // product or store already non-existing: good!
            $success = true;
        }

        delete_post_meta($product_id, self::META_KEY);

        $this->touch();

        return $success;
    }


    /**
    * @param WC_Product $product
    * @param array $product_data
    * @param array $variants_data
    * @param bool $recurse
    *
    * @return bool
    *
    * @throws MC4WP_API_Exception|Exception
    */
    private function product_add(WC_Product $product, array $product_data, $variants_data = array(), $recurse = true)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;
        $product_id = $this->transformer->product_id($product);

        try {
            $response = $api->add_ecommerce_store_product($store_id, $product_data);

            // update each variant separately to work around Mailchimp API timeout issues with many variations (as of Jan 04, 2019)
            foreach ($variants_data as $variant_data) {
                $api->add_ecommerce_store_product_variant($store_id, $product_id, $variant_data);
            }
        } catch (MC4WP_API_Exception $e) {
            // update product if it already exists remotely.
            if ($recurse && (stripos($e->detail, 'product with the provided ID already exists') || stripos($e->detail, 'variant with the provided ID already exists'))) {
                return $this->product_update($product, $product_data, $variants_data, false);
            }

            throw $e;
        }

        $this->touch($product_id);
        return true;
    }

    /**
     * @param WC_Product $product
     * @param array $product_data
     * @param array $variants_data
     * @param bool $recurse
     * @return bool
     * @throws MC4WP_API_Exception|Exception
     */
    private function product_update(WC_Product $product, array $product_data, $variants_data = array(), $recurse = true)
    {
        $api = $this->get_api();
        $store_id = $this->store_id;
        $product_id = $this->transformer->product_id($product);

        try {
            // this method was added in Mailchimp for WordPress v4.0.12
            if (method_exists($api, 'update_ecommerce_store_product')) {
                $response = $api->update_ecommerce_store_product($store_id, $product_id, $product_data);
            }

            // update each variant separately to work around Mailchimp API timeout issues with many variations (as of Jan 04, 2019)
            foreach ($variants_data as $variant_data) {
                $api->add_ecommerce_store_product_variant($store_id, $product_id, $variant_data);
            }
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            if ($recurse) {
                return $this->product_add($product, $product_data, $variants_data, false);
            }

            throw $e;
        }

        $this->touch($product_id);
        return true;
    }

    /**
     * Returns true if we previously sent this order to Mailchimp
     *
     * @param WC_Order|int $order
     * @return bool
     * @since 4.10
     */
    public function is_order_tracked($order) {
    	if (! $order instanceof WC_Order) {
    		$order = wc_get_order($order);
    	}

    	return !!$order->get_meta(self::META_KEY) || !!get_post_meta($order->get_id(), self::META_KEY, true);
    }

    /**
    * @param int $object_id
    *
    * @return bool
    */
    public function is_object_tracked($object_id)
    {
        return !! get_post_meta($object_id, self::META_KEY, true);
    }

    /**
     * @return \MC4WP_API_v3
     * @throws Exception
     */
    private function get_api()
    {
        return mc4wp('api');
    }

    /**
     * Ensures the existence of a deleted product in Mailchimp, to be used in orders referencing a no-longer existing product.
     *
     * @return void
     * @throws MC4WP_API_Exception|Exception
     */
    private function ensure_deleted_product()
    {
        static $exists = false;

        if ($exists) {
            return;
        }

        // create or update deleted product in Mailchimp
        $store_id = $this->store_id;
        $api = $this->get_api();

        $product_id = 'deleted';
        $product_title = '(deleted product)';
        $data = array(
            'id' => $product_id,
            'title' => $product_title,
            'variants' => array(
                array(
                    'id' => $product_id,
                    'title' => $product_title,
                    'inventory_quantity' => 0,
                )
            )
        );

        try {
            $response = $api->update_ecommerce_store_product($store_id, $product_id, $data);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            $response = $api->add_ecommerce_store_product($store_id, $data);
        }

        // set flag to short-circuit this function next time it runs
        $exists = true;
    }


    /**********************
    * 		Promo codes 	 *
    **********************/

    /**
     * Deletes the associated promo from the connected store.
     * @param int $coupon_id
     * @throws MC4WP_API_Exception|Exception
     */
    public function delete_promo($coupon_id)
    {
        $store_id = $this->store_id;
        $api = $this->get_api();

        // fail silently if API class does not have method. This means user should update their Mailchimp for WordPress version.
        if (! method_exists($api, 'delete_ecommerce_store_promo_rule')) {
            return;
        }

        try {
            // TODO: Check if we need to delete children of the promo rule?
            //$api->delete_ecommerce_store_promo_rule_promo_code( $store_id, $coupon_id, $coupon_id );
            $api->delete_ecommerce_store_promo_rule($store_id, $coupon_id);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            // good. promo was not there to begin with.
        }

        $this->touch();
    }

    /**
     * Adds or updates the associated promo in the connected store.
     * @param int $coupon_id
     * @throws MC4WP_API_Exception|Exception
     */
    public function update_promo($coupon_id)
    {
        $store_id = $this->store_id;
        $api = $this->get_api();

        // fail silently if API class does not have method. This means user should update their Mailchimp for WordPress version.
        if (! method_exists($api, 'delete_ecommerce_store_promo_rule')) {
            return;
        }

        $wc_coupon = new WC_Coupon($coupon_id);

        // fail silently if on WooCommerce v2.x
        if (! method_exists($wc_coupon, 'get_code')) {
            return;
        }

        // create promo rule
        $promo_rule_data = array(
            'id' => (string) $coupon_id,
            'title' => (string) $wc_coupon->get_code(),
            'description' => (string) $wc_coupon->get_description(),
            'enabled' => true,
            'amount' => (float) $wc_coupon->get_amount('edit'),
        );

        if (empty($promo_rule_data['description'])) {
            $promo_rule_data['description'] = (string) $wc_coupon->get_code();
        }

        // determine whether rule is enabled
        $expires = $wc_coupon->get_date_expires();
        if ($expires) {
            $promo_rule_data['ends_at'] = (string) $expires;

            if (current_time('timestamp', true) >= $expires->getTimestamp()) {
                $promo_rule_data['enabled'] = false;
            }
        }

        switch ($wc_coupon->get_discount_type()) {
            case 'fixed_product':
                $promo_rule_data['type'] = 'fixed';
                $promo_rule_data['target'] = 'per_item';
            break;

            case 'fixed_cart':
                $promo_rule_data['type'] = 'fixed';
                $promo_rule_data['target'] = 'total';
            break;

            case 'percent':
                $promo_rule_data['type'] = 'percentage';
                $promo_rule_data['target'] = 'total';
                $promo_rule_data['amount'] = (float) $wc_coupon->get_amount('edit') / 100;
            break;
        }

        /**
        * Filters the promo rule data before it is sent to Mailchimp
        *
        * @param array $promo_rule_data
        */
        $promo_rule_data = apply_filters('mc4wp_ecommerce_promo_rule_data', $promo_rule_data);

        try {
            $api->update_ecommerce_store_promo_rule($store_id, $promo_rule_data['id'], $promo_rule_data);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            $api->add_ecommerce_store_promo_rule($store_id, $promo_rule_data);
        }

        // create promo code (child of promo rule)
        $redemption_url = add_query_arg(array(
            'coupon_code' => urlencode($wc_coupon->get_code()),
        ), get_home_url());

        $promo_code_data = array(
            'id' => (string) $coupon_id,
            'code' => (string) $wc_coupon->get_code(),
            'redemption_url' => (string) $redemption_url,
            'usage_count' => (int) $wc_coupon->get_usage_count(),
            'enabled' => $promo_rule_data['enabled'],
        );

        /**
        * Filters the promo code data before it is sent to Mailchimp
        *
        * @param array $promo_code_data
        */
        $promo_code_data = apply_filters('mc4wp_ecommerce_promo_code_data', $promo_code_data);

        try {
            $api->update_ecommerce_store_promo_rule_promo_code($store_id, $promo_rule_data['id'], $promo_code_data['id'], $promo_code_data);
        } catch (MC4WP_API_Resource_Not_Found_Exception $e) {
            $api->add_ecommerce_store_promo_rule_promo_code($store_id, $promo_rule_data['id'], $promo_code_data);
        }

        // update stats on when we last updated
        $this->touch();
    }
}
