( function dateInit () {
	/**
     * @fileoverview This script manages a date and time scheduling interface,
     * including a calendar and time slot selector. It handles timezone conversions
     * from a site's configured timezone to the user's local timezone.
     */

	// --- Core Initialization ---

	/**
     * Initializes the date and time picker after the DOM is fully loaded.
     * This ensures all required elements are available before the script runs.
     */

	/**
     * @id ctc_fd_time_slot_group - Container for time slot selection grid
     * @id ctc_cal_modal - Main modal dialog for calendar and time picker
     * @id ctc_cal_modal_overlay - Full-screen overlay that darkens background
     * @id ctc_cal_grid - Grid container for calendar day elements
     * @id ctc_cal_month_year - Displays current month and year
     * @id ctc_cal_prev_month - Button to navigate to previous month
     * @id ctc_cal_next_month - Button to navigate to next month
     * @id ctc_cal_clear_button - Button to clear selected date and time
     * @id ctc_cal_ok_button - Button to confirm selection and close modal
     * @id ctc_fd_time_select - Grid container for time slot boxes
     * @id ctc_fd_time_slot_group_modal - Modal container for time slot selector
     * -----------------------------------------------------------------------------
     * @class ctc_fd_form_date - Main container for date and time inputs
     * @class ctc_fd_date_input - Date input field for 'scheduler' mode
     * @class ctc_fd_date_only_input - Date input field for 'date_only' mode
     * @class ctc_fd_time_input - Time input field
     * @class ctc_fd_time_slot_box - Individual selectable time slot element
     * @class ctc_user_tz - Hidden input for user's timezone
     * @class ctc_cal_day - Individual calendar day element
     * @class ctc_cal_day_header - Day of week header element
     * @class ctc_cal_scroll_content - Scrollable content area in modal
     * @class hidden - Utility class to hide elements
     * @class ctc_cal_day_disabled - Non-selectable calendar days
     * @class ctc_cal_day_past - Past calendar days
     * @class ctc_cal_day_selected - Currently selected calendar day
     * @class ctc_fd_time_slot_box_selected - Currently selected time slot
     */

	// todo: check.. format like there wont any error.. or seperate logic with a function..
	let userTimezone = '';

	/**
     * The user's local timezone offset.
     * @type {number}
     */
	const user_tz_offset = -new Date()
		.getTimezoneOffset();

	function initDate () {
		console.log( 'date.dev.js' );

		// --- Configuration & Variable Retrieval ---

		/**
         * The main configuration object, `ht_ctc_variables`, is a global variable
         * expected to be set by the WordPress plugin. It contains settings
         * for the date and time scheduler.
         * @typedef {object} HtCtcVariables
         * @property {number} [g_f_slot_time=60] - Interval for time slots in minutes.
         * @property {number} [g_f_max_days=365] - Maximum number of days in the future a user can book.
         * @property {number} [g_f_min_notice_hours=0] - Minimum number of hours' notice required for a booking.
         * @property {Array<Array<object>>} [g_f_scheduling_hours] - A 7-element array representing the hours of operation for each day of the week (Sun-Sat).
         * @property {string} [g_f_min_date] - The minimum selectable date, can be a specific date (YYYY-MM-DD) or a relative number of days (+N, -N).
         * @property {string} [g_f_max_date] - The maximum selectable date, can be a specific date (YYYY-MM-DD) or a relative number of days (+N, -N).
         */
		const ht_ctc_variables = window.ht_ctc_variables || {};

		/**
         * Determines the mode of the date field, either 'scheduler' with time slots
         * or 'date_only' for simple date selection.
         * @type {string}
         */
		const dateMode = document.querySelector( '.ctc_fd_form_date' )
			?.getAttribute( 'data-date-mode' ) || 'scheduler';
		console.log( 'Date Mode:', dateMode );

		/**
         * The interval in minutes for generating time slots. Defaults to 60 if not
         * a valid positive number.
         * @type {number}
         */
		const g_f_slot_time = Number.isNaN( parseInt( ht_ctc_variables.g_f_slot_time ) ) || parseInt( ht_ctc_variables.g_f_slot_time ) <= 0 ? 60 : parseInt( ht_ctc_variables.g_f_slot_time );

		/**
         * The maximum number of days in advance a user can book. Defaults to 365.
         * @type {number}
         */
		const g_f_max_days = Number.isNaN( parseInt( ht_ctc_variables.g_f_max_days ) ) ? 365 : parseInt( ht_ctc_variables.g_f_max_days );

		/**
         * The minimum number of hours notice required for a booking. Defaults to 0.
         * @type {number}
         */
		const g_f_min_notice_hours = Number.isNaN( parseInt( ht_ctc_variables.g_f_min_notice_hours ) ) ? 0 : parseInt( ht_ctc_variables.g_f_min_notice_hours );

		/**
         * The timezone offset of the WordPress site (admin).
         * @type {number}
         */
		const admin_tz_offset = Number.isNaN( parseFloat( window.ht_ctc_chat_var?.tz ) ) ? 0 : parseFloat( window.ht_ctc_chat_var.tz ) * 60;

		/**
         * Raw scheduling hours from the configuration. This array is normalized to
         * ensure it always has 7 elements.
         * @type {Array<Array<object>>}
         */
		const g_f_scheduling_hours = ht_ctc_variables.g_f_scheduling_hours || Array( 7 )
			.fill( [ { 'start': '00:00', 'end': '00:00' } ] );
		while ( g_f_scheduling_hours.length < 7 ) {
			g_f_scheduling_hours.push( [ { 'start': '00:00', 'end': '00:00' } ] );
		}

		// --- Timezone Conversion Utilities ---

		/**
         * Formats a timezone offset in minutes into a standard `+HH:MM` or `-HH:MM` string.
         * @param {number} offsetInMinutes The offset in minutes.
         * @returns {string} The formatted timezone string.
         */
		function getOffsetString ( offsetInMinutes ) {
			const sign = offsetInMinutes >= 0 ? '+' : '-';
			const absMinutes = Math.abs( offsetInMinutes );
			const hours = String( Math.floor( absMinutes / 60 ) )
				.padStart( 2, '0' );
			const minutes = String( absMinutes % 60 )
				.padStart( 2, '0' );
			return `${sign}${hours}:${minutes}`;
		}

		/**
         * Converts a time string from the admin's timezone to the user's local timezone.
         * It handles time shifts that cross midnight.
         * @param {string} timeStr - The time string in `HH:MM` format (admin timezone).
         * @returns {string} The converted time string in `HH:MM` format (user timezone).
         */
		function convertAdminTimeToUserTime ( timeStr ) {
			const [ hours, minutes ] = timeStr.replace( /\s/g, '' )
				.split( ':' )
				.map( Number );
			const timeDiffMinutes = user_tz_offset - admin_tz_offset;
			let totalMinutes = hours * 60 + minutes + timeDiffMinutes;

			// Handle day overflow/underflow
			if ( totalMinutes < 0 ) { totalMinutes += 24 * 60; }
			if ( totalMinutes >= 24 * 60 ) { totalMinutes -= 24 * 60; }

			const userHours = Math.floor( totalMinutes / 60 );
			const userMinutes = totalMinutes % 60;
			return `${String( userHours )
				.padStart( 2, '0' )}:${String( userMinutes )
				.padStart( 2, '0' )}`;
		}

		/**
         * Converts all scheduling hour slots for all days from the admin timezone
         * to the user's local timezone. It splits slots that cross midnight into two.
         * @param {Array<Array<object>>} slots - The array of scheduling slots in admin timezone.
         * @returns {Array<Array<object>>} The converted scheduling slots in user timezone.
         */
		function convertSlotsToUserTime ( slots ) {
			return slots.map( daySlots => {
				const convertedSlots = [];
				daySlots.forEach( slot => {
					if ( slot.start === '00:00' && slot.end === '00:00' ) {
						convertedSlots.push( { start: '00:00', end: '00:00' } );
						return;
					}
					const userStart = convertAdminTimeToUserTime( slot.start );
					const userEnd = convertAdminTimeToUserTime( slot.end );
					const [ startH, startM ] = userStart.split( ':' )
						.map( Number );
					const [ endH, endM ] = userEnd.split( ':' )
						.map( Number );
					if ( endH * 60 + endM <= startH * 60 + startM ) {
						convertedSlots.push( { start: userStart, end: '23:59' } );
						convertedSlots.push( { start: '00:00', end: userEnd } );
					} else {
						convertedSlots.push( { start: userStart, end: userEnd } );
					}
				} );
				return convertedSlots;
			} );
		}

		const userLocalSlots = convertSlotsToUserTime( g_f_scheduling_hours );

		// --- Main Configuration Object ---

		const CONFIG = {
			LIMIT_DAYS: g_f_max_days,
			MIN_NOTICE_HOURS: g_f_min_notice_hours,
			SLOT_INTERVAL_MINUTES: g_f_slot_time,
			SCHEDULING_HOURS: userLocalSlots,
			WP_TZ_OFFSET: user_tz_offset,
		};
		console.log( 'CONFIG', CONFIG );

		// --- Date Constraint Parsing ---

		/**
         * Parses a date constraint string into a Date object. The string can be
         * a relative number of days (+N, -N) or a specific date (YYYY-MM-DD).
         * @param {string} constraint - The date constraint string.
         * @returns {Date|null} The parsed Date object or null if invalid.
         */
		function parseDateConstraint ( constraint ) {
			if ( ! constraint ) { return null; }
			const relativeMatch = constraint.match( /^([+-]?\d+)$/ );
			if ( relativeMatch ) {
				const days = parseInt( relativeMatch[ 1 ] );
				const dateObj = new Date();
				dateObj.setDate( dateObj.getDate() + days );
				return dateObj;
			}
			const dateMatch = constraint.match( /^(\d{4})-(\d{1,2})-(\d{1,2})$/ );
			if ( dateMatch ) {
				const year = parseInt( dateMatch[ 1 ] );
				const month = parseInt( dateMatch[ 2 ] ) - 1;
				const day = parseInt( dateMatch[ 3 ] );
				return new Date( year, month, day );
			}
			return null;
		}
		const minDateConstraint = parseDateConstraint( ht_ctc_variables.g_f_min_date );
		const maxDateConstraint = parseDateConstraint( ht_ctc_variables.g_f_max_date );
		console.log( 'Min Date Constraint:', minDateConstraint );
		console.log( 'Max Date Constraint:', maxDateConstraint );

		// --- Date Field Initialization ---

		/**
         * Initializes all date fields found on the page with the appropriate
         * listeners and functionality.
         * @param {HTMLElement} dateField - The container for the date picker elements.
         */
		function initializeDateField ( dateField ) {
			console.log( 'Initializing date field:', dateField );
			const dateInput = dateField.querySelector( '.ctc_fd_date_input' );
			const dateOnlyInput = dateField.querySelector( '.ctc_fd_date_only_input' );
			const timeInput = dateField.querySelector( '.ctc_fd_time_input' );
			const timeZoneInput = dateField.querySelector( '.ctc_user_tz' );
			const timeSlotGroup = document.getElementById( 'ctc_fd_time_slot_group' );
			const calendarModal = document.getElementById( 'ctc_cal_modal' );
			const modalOverlay = document.getElementById( 'ctc_cal_modal_overlay' );
			const calendar = document.getElementById( 'ctc_cal_grid' );
			const monthYear = document.getElementById( 'ctc_cal_month_year' );
			const prevMonth = document.getElementById( 'ctc_cal_prev_month' );
			const nextMonth = document.getElementById( 'ctc_cal_next_month' );
			const clearButton = document.getElementById( 'ctc_cal_clear_button' );
			const okButton = document.getElementById( 'ctc_cal_ok_button' );
			const timeGrid = document.getElementById( 'ctc_fd_time_select' );

			const currentDate = new Date();
			let selectedDate = null;

			// Hide time input and its label by default.
			if ( timeInput ) {
				timeInput.disabled = true;
				timeInput.style.display = 'none';
				const timeLabel = timeInput.previousElementSibling;
				if ( timeLabel?.tagName === 'LABEL' ) {
					timeLabel.style.display = 'none';
				}
			}

			// Set min/max attributes for native date input in 'date_only' mode.
			if ( dateOnlyInput && dateMode === 'date_only' ) {
				if ( minDateConstraint ) {
					dateOnlyInput.min = minDateConstraint.toISOString()
						.split( 'T' )[ 0 ];
				}
				if ( maxDateConstraint ) {
					dateOnlyInput.max = maxDateConstraint.toISOString()
						.split( 'T' )[ 0 ];
				}
			}

			// --- Event Handlers & Core Functions ---

			/** Shows the calendar modal and renders the calendar. */
			function showCalendar () {
				if ( ! calendarModal || ! modalOverlay ) { return; }
				calendarModal.style.display = 'block';
				modalOverlay.style.display = 'block';
				renderCalendar();
				if ( ! selectedDate ) {
					document.querySelector( '.ctc_cal_scroll_content' )
						?.scrollTo( { top: 0, behavior: 'smooth' } );
				}
			}

			/** Hides the calendar modal. */
			function hideCalendar () {
				if ( ! calendarModal || ! modalOverlay ) { return; }
				calendarModal.style.display = 'none';
				modalOverlay.style.display = 'none';
			}

			/** Changes the displayed month in the calendar. */
			function changeMonth ( delta ) {
				currentDate.setMonth( currentDate.getMonth() + delta );
				renderCalendar();
			}

			/** Clears the selected date and time and hides related elements. */
			function clearSelection () {
				if ( dateInput ) { dateInput.value = ''; }
				if ( timeInput ) {
					timeInput.value = '';
					timeInput.disabled = true;
					timeInput.style.display = 'none';
					const timeLabel = timeInput.previousElementSibling;
					if ( timeLabel?.tagName === 'LABEL' ) { timeLabel.style.display = 'none'; }
				}
				if ( timeSlotGroup ) { timeSlotGroup.style.display = 'none'; }
				if ( timeGrid ) {
					timeGrid.innerHTML = '';
					timeGrid.classList.add( 'hidden' );
					timeGrid.disabled = true;
				}
				document.querySelectorAll( '.ctc_cal_day' )
					.forEach( day => day.classList.remove( 'selected' ) );
				hideCalendar();
			}

			/** Confirms the selection and hides the calendar. */
			function doneSelection () {
				hideCalendar();
			}

			/** Fills a hidden input with the user's timezone. */
			function fillTimeZoneField ( timeZoneInput ) {
				if ( timeZoneInput ) {
					timeZoneInput.value = userTimezone;
				}
			}

			/** Renders the calendar grid for the current month. */
			function renderCalendar () {
				if ( ! calendar || ! monthYear ) { return; }
				calendar.innerHTML = '';
				const now = new Date();
				const todayAtMidnight = new Date( now.getFullYear(), now.getMonth(), now.getDate() );
				const maxDate = new Date( todayAtMidnight );
				maxDate.setDate( todayAtMidnight.getDate() + CONFIG.LIMIT_DAYS );
				const weekdays = [ 'Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat' ];
				weekdays.forEach( day => {
					const div = document.createElement( 'div' );
					div.className = 'ctc_cal_day_header';
					div.textContent = day;
					calendar.appendChild( div );
				} );

				const year = currentDate.getFullYear();
				const month = currentDate.getMonth();
				const firstDay = new Date( year, month, 1 );
				const lastDay = new Date( year, month + 1, 0 );
				const daysInMonth = lastDay.getDate();
				const startingDay = firstDay.getDay();

				monthYear.textContent = currentDate.toLocaleString( 'default', { month: 'long', year: 'numeric' } );

				for ( let i = 0; i < startingDay; i++ ) { calendar.appendChild( document.createElement( 'div' ) ); }

				for ( let day = 1; day <= daysInMonth; day++ ) {
					const date = new Date( year, month, day );
					const dayDiv = document.createElement( 'div' );
					dayDiv.className = 'ctc_cal_day';
					dayDiv.textContent = day;

					const dateAtMidnight = new Date( date.getFullYear(), date.getMonth(), date.getDate() );
					const isPast = dateAtMidnight < todayAtMidnight;
					const isBeyondLimit = dateAtMidnight > maxDate;
					const hasSlots = dateMode === 'scheduler' ? hasAvailableSlots( date ) : true;
					const isWithinConstraint = ( minDateConstraint ? dateAtMidnight >= minDateConstraint : true ) && ( maxDateConstraint ? dateAtMidnight <= maxDateConstraint : true );

					if ( isPast || isBeyondLimit || ! hasSlots || ! isWithinConstraint ) {
						dayDiv.classList.add( 'disabled' );
						if ( isPast ) { dayDiv.classList.add( 'past' ); }
					} else {
						dayDiv.addEventListener( 'click', () => handleDateSelection( dayDiv, date ) );
					}

					if ( selectedDate && dateAtMidnight.getTime() === new Date( selectedDate.getFullYear(), selectedDate.getMonth(), selectedDate.getDate() )
						.getTime() ) {
						dayDiv.classList.add( 'selected' );
					}

					calendar.appendChild( dayDiv );
				}
			}

			/**
             * Checks if a given date has any available time slots based on the
             * minimum notice hours.
             * @param {Date} date - The date to check.
             * @returns {boolean} True if there are available slots, false otherwise.
             */
			function hasAvailableSlots ( date ) {
				const dayOfWeek = date.getDay();
				const hours = CONFIG.SCHEDULING_HOURS[ dayOfWeek ];
				if ( ! hours || hours.length === 0 ) { return true; }
				const hasValidTimes = hours.some( slot => ! ( slot.start === '00:00' && slot.end === '00:00' ) );
				if ( ! hasValidTimes ) { return false; }

				const now = new Date();
				const minNoticeTimestamp = now.getTime() + CONFIG.MIN_NOTICE_HOURS * 60 * 60 * 1000;
				const slots = getTimeSlots( date, hours );
				return slots.some( slot => {
					const [ hours, minutes ] = slot.split( ':' )
						.map( Number );
					const slotTime = new Date( date.getFullYear(), date.getMonth(), date.getDate(), hours, minutes );
					return slotTime.getTime() > minNoticeTimestamp;
				} );
			}

			/**
             * Generates a list of time slots for a given date and hour ranges.
             * @param {Date} date - The date for which to generate slots.
             * @param {Array<object>} hours - The start/end hour ranges.
             * @returns {Array<string>} An array of time strings in `HH:MM` format.
             */
			function getTimeSlots ( date, hours ) {
				const slots = [];
				hours.forEach( range => {
					const [ startHour, startMinute ] = range.start.split( ':' )
						.map( Number );
					const [ endHour, endMinute ] = range.end.split( ':' )
						.map( Number );
					let currentMinutes = startHour * 60 + startMinute;
					const endMinutes = endHour * 60 + endMinute;

					while ( currentMinutes + CONFIG.SLOT_INTERVAL_MINUTES <= endMinutes ) {
						const hour = Math.floor( currentMinutes / 60 );
						const minute = currentMinutes % 60;
						slots.push( `${String( hour )
							.padStart( 2, '0' )}:${String( minute )
							.padStart( 2, '0' )}` );
						currentMinutes += CONFIG.SLOT_INTERVAL_MINUTES;
					}
				} );
				return slots;
			}

			/**
             * Handles the date selection logic, updates the input field,
             * and populates the time slot selector.
             * @param {HTMLElement} dayDiv - The calendar day div that was clicked.
             * @param {Date} date - The selected date.
             */
			function handleDateSelection ( dayDiv, date ) {
				selectedDate = date;
				const formattedDate = `${date.getFullYear()}-${String( date.getMonth() + 1 )
					.padStart( 2, '0' )}-${String( date.getDate() )
					.padStart( 2, '0' )}`;
				console.log( 'Selected Date:', formattedDate );

				if ( dateInput ) { dateInput.value = formattedDate; }

				// Reset and show time input/label
				if ( timeInput ) {
					timeInput.value = '';
					timeInput.disabled = false;
					timeInput.style.display = 'block';
					const timeLabel = timeInput.previousElementSibling;
					if ( timeLabel?.tagName === 'LABEL' ) { timeLabel.style.display = 'block'; }
				}

				document.querySelectorAll( '.ctc_cal_day' )
					.forEach( day => day.classList.remove( 'selected' ) );
				dayDiv.classList.add( 'selected' );
				populateTimeSlots( date );
				document.querySelector( '.ctc_cal_scroll_content' )
					?.scrollTo( {
						top: document.querySelector( '.ctc_cal_scroll_content' ).scrollHeight * 0.15,
						behavior: 'smooth',
					} );
			}

			/**
             * Populates the time slot grid with available slots for the selected date.
             * @param {Date} date - The selected date.
             */
			function populateTimeSlots ( date ) {
				const timeSlotGroupModal = document.getElementById( 'ctc_fd_time_slot_group_modal' );
				if ( ! timeSlotGroupModal || ! timeGrid ) { return; }
				timeGrid.innerHTML = '';
				const hours = CONFIG.SCHEDULING_HOURS[ date.getDay() ];
				const slots = getTimeSlots( date, hours );
				const now = new Date();
				const minNoticeTimestamp = now.getTime() + CONFIG.MIN_NOTICE_HOURS * 60 * 60 * 1000;

				slots.forEach( slot => {
					const [ hours, minutes ] = slot.split( ':' )
						.map( Number );
					const slotTime = new Date( date.getFullYear(), date.getMonth(), date.getDate(), hours, minutes );
					if ( slotTime.getTime() > minNoticeTimestamp ) {
						const slotBox = document.createElement( 'div' );
						slotBox.className = 'ctc_fd_time_slot_box';
						slotBox.dataset.time = slot;
						slotBox.textContent = formatTime( slot );
						slotBox.addEventListener( 'click', () => handleTimeSelection( slot, slotBox ) );
						timeGrid.appendChild( slotBox );
					}
				} );
				timeSlotGroupModal.style.display = 'block';
			}

			/**
             * Handles the selection of a time slot.
             * @param {string} slot - The selected time slot in `HH:MM` format.
             * @param {HTMLElement} slotBox - The HTML element of the selected slot.
             */
			function handleTimeSelection ( slot, slotBox ) {
				console.log( 'handleTimeSelection called' );

				console.log( 'Selected Time Slot:', slot );

				// slotbox
				console.log( 'slotBox', slotBox );

				// timeInput
				console.log( 'timeInput', timeInput );

				document.querySelectorAll( '.ctc_fd_time_slot_box.selected' )
					.forEach( el => el.classList.remove( 'selected' ) );
				slotBox.classList.add( 'selected' );
				if ( timeInput ) { timeInput.value = formatTime( slot ); }

				if ( timeSlotGroup ) {
					timeSlotGroup.style.display = 'block';
				}

				// fill timezone field
				fillTimeZoneField( timeZoneInput );
			}

			/**
             * Formats a 24-hour time string into a 12-hour AM/PM format.
             * @param {string} time - The time string in `HH:MM` format.
             * @returns {string} The formatted time string.
             */
			function formatTime ( time ) {
				const [ hour, minute ] = time.split( ':' )
					.map( Number );
				const period = hour >= 12 ? 'PM' : 'AM';
				const displayHour = hour % 12 || 12;
				return `${displayHour}:${String( minute )
					.padStart( 2, '0' )} ${period}`;
			}

			// --- Event Listeners ---

			// click or foucus on date input to show calendar
			if ( dateInput && dateMode === 'scheduler' ) {
				dateInput.addEventListener( 'focus', showCalendar );
				dateInput.addEventListener( 'click', showCalendar );
			}

			// time input click to show calendar
			if ( timeInput ) { timeInput.addEventListener( 'click', showCalendar ); }

			// Hide calendar when clicking outside or on calendar controls
			if ( modalOverlay ) { modalOverlay.addEventListener( 'click', hideCalendar ); }

			// calendar controls - month navigation, clear, ok
			if ( prevMonth ) { prevMonth.addEventListener( 'click', () => changeMonth( -1 ) ); }
			if ( nextMonth ) { nextMonth.addEventListener( 'click', () => changeMonth( 1 ) ); }
			if ( clearButton ) { clearButton.addEventListener( 'click', clearSelection ); }
			if ( okButton ) { okButton.addEventListener( 'click', doneSelection ); }

			// Initial render
			if ( dateMode === 'scheduler' ) {
				renderCalendar();
			}
		}

		function getUserTimeZone () {
			try {
				return `${Intl.DateTimeFormat()
					.resolvedOptions().timeZone} (UTC${getOffsetString( user_tz_offset )})`;
			} catch ( error ) {
				return '';
			}
		}

		// --- Execution ---
		if ( document.querySelector( '.ctc_fd_form_date' ) ) {

			// todo: make sure this is correct, no chance of error.. or call in a function with try catch
			userTimezone = getUserTimeZone();

			document.querySelectorAll( '.ctc_fd_form_date' )
				.forEach( initializeDateField );
		}

	}

	// Call initDate when the DOM is ready
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', initDate );
	} else {
		initDate();
	}
} )();
