<?php
/**
 * Hooks and admin integrations for Click to Chat Pro.
 * Position type, greetings, scripts, localization and more.
 *
 * @package Click_To_Chat_Pro
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'HT_CTC_PRO_Admin_Hooks' ) ) :

	/**
	 * Admin hooks for Click to Chat Pro.
	 */
	class HT_CTC_PRO_Admin_Hooks {

		/**
		 * Constructor.
		 */
		public function __construct() {
			$this->hooks();
		}

		/**
		 * Hooks
		 *
		 * @return void
		 */
		public function hooks() {

			$this->admin_notice();

			$options = get_option( 'ht_ctc_chat_options' );

			// position type
			add_filter( 'ht_ctc_fh_position_type_values', array( $this, 'add_position_type_values' ) );

			// editor values
			add_filter( 'ht_ctc_fh_greetings_setting_editor_values', array( $this, 'editor_values' ) );

			/**
			 * Woo places
			 *
			 * Valid upto main version - 3.9.11
			 * removed this hook in 3.9.12, added this feature directly in main version itself.
			 *
			 * Safelty remove this hook and its related function in November 2022. expected that users will update both versions.
			 * (as now: no problem if both versions are not updated, issue will be if one plugin updated and other not)
			 */
			add_filter( 'ht_ctc_fh_admin_woo_places', array( $this, 'woo_places' ) );

			add_action( 'ht_ctc_ah_admin_chat_bottom_meta_box', array( $this, 'chat_meta_box' ), 10, 1 );
			add_action( 'ht_ctc_ah_admin_chat_meta_box_after_number', array( $this, 'chat_meta_box_after_number' ), 10, 1 );

			add_filter( 'ht_ctc_ah_admin_chat_number', array( $this, 'number' ) );
			add_filter( 'ht_ctc_ah_admin_chat_after_showhide', array( $this, 'after_showhide' ) );
			add_filter( 'ht_ctc_ah_admin_chat_more_options', array( $this, 'bh' ) );

			add_action( 'ht_ctc_ah_admin_header_status_badge', array( $this, 'header_badge' ) );

			// scripts
			// add_action( 'ht_ctc_ah_admin_scripts_end', array($this, 'admin_scripts') );
			add_action( 'ht_ctc_ah_admin_scripts_start', array( $this, 'admin_scripts' ) );

			// wp - admin enqueue scipts
			add_action( 'admin_enqueue_scripts', array( $this, 'admin_register_scripts' ), 20 );

			// fb conversation api - admin settings
			add_action( 'ht_ctc_ah_admin_after_fb_pixel', array( $this, 'fb_capi' ) );

			// google ads
			add_action( 'ht_ctc_ah_admin_google_ads', array( $this, 'google_ads' ) );

			// uses from front end. as ajax call, action declared here. (is_admin)
			add_action( 'wp_ajax_ctc_pro_is_user_logged_in', array( $this, 'is_user_logged_in' ) );
			add_action( 'wp_ajax_nopriv_ctc_pro_is_user_logged_in', array( $this, 'is_user_logged_in' ) );

			// updates
			// add_action( 'wp_ajax_ctc_pro_updates', [$this, 'plugin_updates'] );

			// ajax call for function update_r_sqx - for sqx
			$r_num_type = ( isset( $options['r_num_type'] ) ) ? esc_attr( $options['r_num_type'] ) : '';
			if ( 'sequential' === $r_num_type ) {
				add_action( 'wp_ajax_ctc_pro_update_r_sqx', array( $this, 'update_r_sqx' ) );
				add_action( 'wp_ajax_nopriv_ctc_pro_update_r_sqx', array( $this, 'update_r_sqx' ) );

				// ajax call for function get_r_sqx - for sqx
				add_action( 'wp_ajax_ctc_pro_get_r_sqx', array( $this, 'get_r_sqx' ) );
				add_action( 'wp_ajax_nopriv_ctc_pro_get_r_sqx', array( $this, 'get_r_sqx' ) );
			}

			// greetings
			add_filter( 'ht_ctc_fh_greetings_register', array( $this, 'greetings_register' ) );
			add_filter( 'ht_ctc_fh_greetings_templates', array( $this, 'greetings_templates' ) );
			add_filter( 'ht_ctc_fh_greetings_setting_values', array( $this, 'greetings_settings' ) );

			// meta box inputs.. adding greetings dialog. (classic editor. new editor is in class-ht-ctc-pro-hooks.php)
			add_filter( 'ht_ctc_fh_greetings_setting_meta_editor', array( $this, 'greetings_meta_editor' ) );

			// Admin - demo - add greetings templates
			// add_filter( 'ht_ctc_fh_admin_demo_greetings_templates', array($this, 'demo_greetings_templates') );

			add_filter( 'ht_ctc_fh_url_structure_d_list', array( $this, 'url_structure_d_list' ) );
			add_filter( 'ht_ctc_fh_url_structure_m_list', array( $this, 'url_structure_m_list' ) );

			add_action( 'ht_ctc_ah_url_structure_desktop', array( $this, 'custom_url_desktop' ) );
			add_action( 'ht_ctc_ah_url_structure_mobile', array( $this, 'custom_url_mobile' ) );

			$g1_pro_options = get_option( 'ht_ctc_greetings_pro_1' );
			$raw_email      = ( isset( $g1_pro_options['email'] ) ) ? esc_attr( $g1_pro_options['email'] ) : '';

			// greetings form data hanlde - ajax hook works only if valid email is added..
			// && is_email($email)
			if ( '' !== $raw_email ) {

				$email_list = explode( ',', $raw_email );

				// if is array and not empty
				if ( is_array( $email_list ) && ! empty( $email_list ) ) {
					// trim each email
					$email_list = array_map( 'trim', $email_list );

					// filter only valid emails
					$valid_emails = array_filter( $email_list, 'is_email' );

					// if valid emails
					if ( ! empty( $valid_emails ) ) {
						// add action for greetings form data
						add_action( 'wp_ajax_ctc_pro_greetings_form', array( $this, 'greetings_form_data' ) );
						add_action( 'wp_ajax_nopriv_ctc_pro_greetings_form', array( $this, 'greetings_form_data' ) );
					}
				}
			}

			// greetings localization
			add_action( 'ht_ctc_ah_admin_localization_greetings_page', array( $this, 'localization_greetings_page' ), 10, 1 );

			// multi agent localization (hook: after multi agent greetings updates)
			// it mightbe better to change this hook if this values are need to user in other places..
			add_action( 'update_option_ht_ctc_greetings_pro_2', array( $this, 'localization_strings' ) );

			// conversation api
			add_action( 'wp_ajax_ctc_pro_capi', array( $this, 'conversion_api' ) );
			add_action( 'wp_ajax_nopriv_ctc_pro_capi', array( $this, 'conversion_api' ) );
		}


		/**
		 * Conversion API.
		 */
		public function conversion_api() {

			// Verify nonce first.
			$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
			if ( ! $nonce || ! wp_verify_nonce( $nonce, 'ht_ctc_nonce' ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Nonce verification failed', 'click-to-chat-pro' ),
					)
				);
			}

			$post_data = ( isset( $_POST ) && $_POST ) ? map_deep( $_POST, 'sanitize_text_field' ) : array();

			// Bail if server super global is unavailable.
			if ( ! isset( $_SERVER ) ) {
				wp_send_json_error(
					array(
						'message' => __( 'Server data is missing', 'click-to-chat-pro' ),
					)
				);
				return;
			}

			$options = get_option( 'ht_ctc_othersettings' );

			$api_version = 'v19.0';

			$pixel_id        = isset( $options['pixel_id'] ) ? esc_attr( $options['pixel_id'] ) : '';
			$pixel_token     = isset( $options['pixel_token'] ) ? esc_attr( $options['pixel_token'] ) : '';
			$test_event_code = isset( $options['test_event_code'] ) ? esc_attr( $options['test_event_code'] ) : '';

			if ( '' === $pixel_id || '' === $pixel_token ) {
				wp_send_json_error(
					array(
						'message' => 'Pixel ID or Access Token is missing',
					)
				);
				return;
			}

			// required: event_name, event_time, action_source. and for website events: event_source_url, client_user_agent are required. and event_id is recommended for deduplication.
			// $event_name = isset($options['pixel_event_name']) ? esc_attr( $options['pixel_event_name'] ) : 'Click to Chat by HoliThemes';
			$event_name = isset( $post_data['pixel_event_name'] ) ? esc_attr( $post_data['pixel_event_name'] ) : 'Click to Chat by HoliThemes';
			// php time i.e. utc time. thats exactly what need to send
			$event_time    = time();
			$action_source = 'website';

			// $event_source_url = isset($_SERVER['HTTP_REFERER']) ? esc_attr( $_SERVER['HTTP_REFERER'] ) : '';
			$event_source_url = isset( $post_data['event_source_url'] ) ? esc_attr( $post_data['event_source_url'] ) : '';

			// client user agent e.g. Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.3
			$client_user_agent = isset( $_SERVER['HTTP_USER_AGENT'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) ) : '';

			$client_ip_address = isset( $_SERVER['REMOTE_ADDR'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) ) : '';

			// Recommended for deduplication
			$event_id = isset( $post_data['event_id'] ) ? esc_attr( $post_data['event_id'] ) : '';

			// $custom_data = isset($post_data['pixel_params']) ? ($post_data['pixel_params']) : [];
			$custom_data = array();

			// pixel_params values will be in array format. but added logic if it can be stringified JSON format.
			if ( isset( $post_data['pixel_params'] ) ) {
				if ( is_array( $post_data['pixel_params'] ) ) {
					$custom_data = map_deep( $post_data['pixel_params'], 'esc_attr' );
				} elseif ( is_string( $post_data['pixel_params'] ) ) {
					// Clean slashes if they exist
					$raw     = stripslashes( $post_data['pixel_params'] );
					$decoded = json_decode( $raw, true );
					if ( is_array( $decoded ) ) {
						$custom_data = map_deep( $decoded, 'esc_attr' );
					}
				}
			}

			$api_url = "https://graph.facebook.com/$api_version/$pixel_id/events?access_token=$pixel_token";

			// client_user_agent, client_ip_address have to be in user_data. can add more data like em (email), ph (phone), fn, ln, etc.
			$user_data = array(
				'client_user_agent' => $client_user_agent,
				'client_ip_address' => $client_ip_address,
			);

			$data = array(
				'event_name'       => $event_name,
				'event_time'       => $event_time,
				'action_source'    => $action_source,
				'event_id'         => $event_id,
				'event_source_url' => $event_source_url,
				'user_data'        => $user_data,
				// 'client_user_agent' => $client_user_agent,
				// 'client_ip_address' => $client_ip_address,
				'custom_data'      => $custom_data,
			);

			// Facebook Conversion API expects 'data' to be an array of event objects,
			// even if sending only one event — hence $data is wrapped in an array.
			$payload = array(
				'data' => array( $data ),
			);

			if ( ! empty( $test_event_code ) ) {
				$payload['test_event_code'] = $test_event_code;
			}

			$args = array(
				'body'        => wp_json_encode( $payload ),
				'headers'     => array( 'Content-Type' => 'application/json' ),
				'method'      => 'POST',
				'data_format' => 'body',
			);

			$response = wp_remote_post( $api_url, $args );

			if ( is_wp_error( $response ) ) {
				wp_send_json_error(
					array(
						'message' => 'Request failed',
						'error'   => $response->get_error_message(),
					)
				);
			}

			$response_body = wp_remote_retrieve_body( $response );

			wp_send_json_success(
				array(
					'message'  => 'Conversion event sent',
					'response' => $response_body,
				)
			);

			// Ends early - this wont runs
			wp_die();
		}


		/**
		 * Output header badge color field.
		 *
		 * @return void
		 */
		public function header_badge() {

			$options = get_option( 'ht_ctc_greetings_options' );

			$g_header_offline_status_color = isset( $options['g_header_offline_status_color'] ) ? esc_attr( $options['g_header_offline_status_color'] ) : '#06e376';

			?>
		<div class="input-field col s6" style="margin-bottom:2px;">
			<p calss="description" style="margin-bottom: 5px;">Offline Status Badge Color</p>
			<input class="ht-ctc-color g_header_offline_status_color" name="ht_ctc_greetings_options[g_header_offline_status_color]" data-default-color="#06e376" value="<?php echo esc_attr( $g_header_offline_status_color ); ?>" type="text">
		</div>
			<?php
		}

		/**
		 * AJAX handler for submitting the Greetings Form in Click to Chat Pro.
		 *
		 * This function:
		 * - Verifies nonce for security.
		 * - Retrieves configured form options from plugin settings.
		 * - Parses and sanitizes form data submitted by the user.
		 * - Constructs and sends an email to the configured admin email.
		 * - Sends a JSON success response back to the frontend.
		 *
		 * @return void
		 */
		public function greetings_form_data() {
			// Security: Check for valid nonce before processing
			check_ajax_referer( 'ht_ctc_pro_greetings_nonce', 'nonce' );

			// Get plugin options for Greetings Form (from `ht_ctc_greetings_pro_1`)
			$g1_pro_options = get_option( 'ht_ctc_greetings_pro_1' );

			// Get and sanitize admin email from options
			// sanitize_email
			$raw_email = ( isset( $g1_pro_options['email'] ) ) ? esc_attr( $g1_pro_options['email'] ) : '';

			// Proceed only if admin email is not empty
			// && is_email($email)
			if ( '' !== $raw_email ) {

				// Expects data in the format: [{name: "field_1", value: "Some Name"}, ...]
					$form_data = ( $_POST && isset( $_POST['form_data'] ) ) ? map_deep( wp_unslash( $_POST['form_data'] ), 'sanitize_text_field' ) : '';
				// $form_data = ($_POST && $_POST['form_data']) ? map_deep( $_POST['form_data'], 'esc_attr' ) : '';

				// Build a name=>value map for quick access to companion inputs like field_1, field_2, field_2_time, field_2_tz
				$form_data_map = array();
				if ( is_array( $form_data ) ) {
					foreach ( $form_data as $r ) {
						if ( isset( $r['name'] ) ) {
							$k                   = sanitize_key( $r['name'] );
							$v                   = isset( $r['value'] ) ? sanitize_text_field( $r['value'] ) : '';
							$form_data_map[ $k ] = $v;
						}
					}
				}

				// Get admin-configured field references (field_1, field_4, etc. - admin filed names might not be in order)
				$admin_fields = ( isset( $g1_pro_options['fields'] ) ) ? array_map( 'esc_attr', $g1_pro_options['fields'] ) : array();

				// Get email subject from options or fallback to default
				$subject = ( isset( $g1_pro_options['email_subject'] ) ) ? esc_attr( $g1_pro_options['email_subject'] ) : '';
				if ( '' === $subject ) {
					$subject = 'Click to Chat - Greetings Form';
				}

				// Initialize variables for email message and reply-to address
				$message     = '';
				$reply_email = '';
				$i           = 0;

				// Loop through form data and build the email message
				foreach ( $form_data as $field ) {

					// if field name string contains _time, _tz then skip. - time, timezone fields are handled with date field if date mode scheduler.
					// if date mode is scheduler, then expect 3 consecutive fields: date, time, timezone. (date field logic like other fields)
					if ( false !== strpos( $field['name'], '_time' ) || false !== strpos( $field['name'], '_tz' ) ) {
						continue;
					}

					if ( isset( $admin_fields[ $i ] ) ) {

						// Get field name (field_1, field_4, etc.)
						$s1 = esc_attr( $admin_fields[ $i ] );

						// Get type of field from admin settings
						$type = isset( $g1_pro_options[ $s1 ]['type'] ) ? esc_attr( $g1_pro_options[ $s1 ]['type'] ) : 'text';

						// Get display name of field from admin settings
						$name = isset( $g1_pro_options[ $s1 ]['name'] ) ? esc_attr( $g1_pro_options[ $s1 ]['name'] ) : $s1;

						// Get field value from sent data
						$value = esc_attr( $field['value'] );

						// If it's an email field and the value is valid, use it as reply-to
						if ( 'email' === $type && is_email( $value ) ) {
							$reply_email = $value;
						}

						// if type is date and date mode is scheduler, then expect 3 consecutive fields: date, time, timezone
						// if type is date and date mode is scheduler, fetch companions by name (not by index offsets)
						if ( 'date' === $type ) {

							$date_mode = ( isset( $g1_pro_options['form_date']['date_mode'] ) ) ? esc_attr( $g1_pro_options['form_date']['date_mode'] ) : '';

							if ( 'scheduler' === $date_mode ) {

								// current posted field name, e.g., field_2
								$current_field_name = isset( $field['name'] ) ? sanitize_key( $field['name'] ) : '';

								// derive companions by name
								$time_key = $current_field_name . '_time';
								$tz_key   = $current_field_name . '_tz';

								// pull from the lookup map (safer than positional offsets)
								$time = isset( $form_data_map[ $time_key ] ) ? esc_attr( $form_data_map[ $time_key ] ) : '';
								$tz   = isset( $form_data_map[ $tz_key ] ) ? esc_attr( $form_data_map[ $tz_key ] ) : '';

								// body
								$message .= "{$name}: {$value}, <br>";
								if ( '' !== $time ) {
									$message .= "-Time: {$time}, <br>";
								}
								if ( '' !== $tz ) {
									$message .= "-Timezone: {$tz}, <br>";
								}

								// as per your current behavior: do NOT replace {field_n} for scheduler dates

								// keep your existing index behavior
								++$i;
								continue;
							}
						}

						// Add field label and value to message body
						$message .= "$name: $value, <br>";

						// Replace {field_1}, {field_2}, etc., in subject line with user input values
						$subject = str_replace( '{field_' . ( $i + 1 ) . '}', $value, $subject );
					}
					++$i;
				}

				// Add additional metadata to email (like date and referring page)
				if ( isset( $_SERVER ) ) {
					$message .= '<br><br><br>';

					// Current date & time in site timezone
					$date     = wp_date( 'F j, Y, g:i a', time() );
					$message .= "Date (based on your WordPress Timezone): $date <br>";

					// Page where the form was submitted (HTTP referrer)
					if ( ! empty( $_SERVER['HTTP_REFERER'] ) ) {
						$v = esc_url_raw( wp_unslash( $_SERVER['HTTP_REFERER'] ) );

						// Decode URL if it has percent-encoded characters (if special chars)
						if ( preg_match( '/%[a-z][0-9]%/i', $v ) ) {
							$v = urldecode( $v );
						}

						$message .= "Page: $v <br>";
					}

					// if ($_SERVER['REMOTE_ADDR']) {
					// $v = $_SERVER['REMOTE_ADDR'];
					// $message .= "IP: $v <br>";
					// }

					// if ($_SERVER['HTTP_USER_AGENT']) {
					// $v = $_SERVER['HTTP_USER_AGENT'];
					// $message .= "User Agent: $v <br>";
					// }
				}

				// HTML headers for email
				$headers = array(
					'Content-Type: text/html; charset=UTF-8',
				);

				// Add reply-to header if a user email was captured
				if ( '' !== $reply_email ) {
					$headers[] = 'Reply-To: ' . $reply_email;
				}

				// html entity decode
				$subject = html_entity_decode( $subject );

				// email_list - multiple emails
				$email_list = explode( ',', $raw_email );

				// explode give array, but still check if array and not empty to use array_map
				if ( is_array( $email_list ) && ! empty( $email_list ) ) {

					// trim each email
					$email_list = array_map( 'trim', $email_list );

					// check if email is valid, loop through emails
					foreach ( $email_list as $e ) {
						if ( is_email( $e ) ) {

							$to = sanitize_email( $e );

							// Send the email to admin with form data
							wp_mail( $to, stripslashes( $subject ), stripslashes( $message ), $headers );

						}
					}
				}
			}

			// Send a success response
			$m = array(
				'works' => 'thanks',
			);
			wp_send_json_success( $m );

			// Properly terminate the AJAX call (ends early, this wont runs)
			wp_die();
		}

		/**
		 * Register greetings strings for localization plugins.
		 *
		 * @param array $new_input New input values.
		 * @return void
		 */
		public function localization_greetings_page( $new_input ) {

			foreach ( $new_input as $key => $value ) {

				// greetings form filed - localization
				if ( false !== strpos( $key, 'field_' ) && false === strpos( $key, 'field_count' ) ) {

					$name         = isset( $value['name'] ) ? $value['name'] : '';
					$placeholder  = isset( $value['placeholder'] ) ? $value['placeholder'] : '';
					$selectvalues = isset( $value['selectvalues'] ) ? $value['selectvalues'] : '';

					do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "greetings_form_$key" . '_name', $name );
					do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "greetings_form_$key" . '_placeholder', $placeholder );

					if ( ! empty( $selectvalues ) ) {
						do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "greetings_form_$key" . '_selectvalues', $selectvalues );
					}
				}

				// grettings multi agent - localization
				if ( false !== strpos( $key, 'agent_' ) && false === strpos( $key, 'agent_count' ) && false === strpos( $key, 'agent_offline' ) ) {
					$number      = isset( $value['number'] ) ? $value['number'] : '';
					$title       = isset( $value['title'] ) ? $value['title'] : '';
					$description = isset( $value['description'] ) ? $value['description'] : '';
					$pre_filled  = isset( $value['pre_filled'] ) ? $value['pre_filled'] : '';

					// multi_agent_1_number    key: agent_1 ...
					do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "multi_$key" . '_number', $number );
					do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "multi_$key" . '_title', $title );
					do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "multi_$key" . '_description', $description );
					do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "multi_$key" . '_pre_filled', $pre_filled );
				}
			}
		}

		/**
		 * Multilingual - general strings
		 *
		 * @uses Greetings multi agent feature..
		 */
		public function localization_strings() {

			$values = array(
				'ctc_hour'    => 'Hour',
				'ctc_hours'   => 'Hours',
				'ctc_minute'  => 'Minute',
				'ctc_minutes' => 'Minutes',
				'ctc_day'     => 'Day',
				'ctc_days'    => 'Days',
			);

			foreach ( $values as $key => $value ) {
				do_action( 'wpml_register_single_string', 'Click to Chat for WhatsApp', "$key", $value );
			}
		}


		/**
		 * AJAX: Return whether current user is logged in.
		 *
		 * @return void
		 */
		public function is_user_logged_in() {
			// // Verify nonce for security
			// if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ?? '' ) ), 'ht_ctc_nonce' ) ) {
			// wp_send_json_error( 'Invalid nonce' );
			// return;
			// }

			$is_logged_in = ( is_user_logged_in() ) ? 'yes' : 'no';
			wp_send_json_success( $is_logged_in );
		}


		/**
		 * AJAX handler to update the sequential index `r_sqx` in the database.
		 *
		 * - Validates the nonce to ensure the request is secure.
		 * - Sanitizes and stores the provided `r_sqx` value in the options table.
		 * - Returns a JSON response indicating success or failure.
		 *
		 * @return void Outputs a JSON response and terminates the script.
		 */
		public function update_r_sqx() {

				// Check nonce
			if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'ht_ctc_nonce' ) ) {
				wp_send_json_error( 'Invalid or expired nonce.' );
				return;
			}

			try {
					// Sanitize r_sqx by using esc_attr
					$r_sqx = isset( $_POST['r_sqx'] ) ? sanitize_text_field( wp_unslash( $_POST['r_sqx'] ) ) : '';
				// $r_sqx = esc_attr($r_sqx);

				// Check if r_sqx is a number, convert to integer
				if ( ! is_numeric( $r_sqx ) ) {
					$r_sqx = '0';
				}

				update_option( 'ht_ctc_r_sqx', $r_sqx );
				wp_send_json_success( 'r_sqx updated' );

			} catch ( Throwable $e ) {
				wp_send_json_error( 'Exception in update_r_sqx' );
			}

			wp_send_json_success( 'r_sqx' );
			// Properly terminate the AJAX call (ends early at try or catch, this wont runs)

			wp_die();
		}

		/**
		 * AJAX handler to fetch the current value of `r_sqx` from the database.
		 *
		 * - Validates the nonce to ensure the request is secure.
		 * - Retrieves the `r_sqx` option from the database (defaults to 0 if not set).
		 * - Returns the value as a JSON response.
		 *
		 * @return void Outputs a JSON response and terminates the script.
		 */
		public function get_r_sqx() {

				// Check nonce for security
			if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'ht_ctc_nonce' ) ) {
				wp_send_json_error( 'Invalid or expired nonce.' );
				return;
			}

			try {

				$r_sqx = get_option( 'ht_ctc_r_sqx', '' );
				$r_sqx = esc_attr( $r_sqx );

				if ( ! is_numeric( $r_sqx ) ) {
					$r_sqx = '0';
				}

				// plan not to touch db with get fn.
				// update_option('ht_ctc_r_sqx', $r_sqx);

				// Send JSON response and terminate properly
				wp_send_json_success( $r_sqx );

			} catch ( Throwable $e ) {
				wp_send_json_error( 'Exception in get_r_sqx' );
			}

			wp_send_json_success( 'get_r_sqx' );

			// Properly terminate the AJAX call (ends early at try or catch, this wont runs)
			wp_die();
		}

		/**
		 * FB conversion api - server side event
		 *
		 * Settings for Facebook Conversion API
		 * DB values:
		 *  Enable: checkbox: fb_conversion
		 *  Pixel_id: text: pixel_id
		 *  Token: textarea: pixel_token
		 */
		public function fb_capi() {

			$options = get_option( 'ht_ctc_othersettings' );
			$dbrow   = 'ht_ctc_othersettings';

			$fb_conversion_checkbox = ( isset( $options['fb_conversion'] ) ) ? esc_attr( $options['fb_conversion'] ) : '';
			$pixel_id               = isset( $options['pixel_id'] ) ? esc_attr( $options['pixel_id'] ) : '';
			$pixel_token            = isset( $options['pixel_token'] ) ? esc_textarea( $options['pixel_token'] ) : '';
			$test_event_code        = isset( $options['test_event_code'] ) ? esc_attr( $options['test_event_code'] ) : '';

			$fb_capi_error = '';
			if ( '1' === $fb_conversion_checkbox && ( '' === $pixel_id || '' === $pixel_token ) ) {
				$fb_capi_error = __( 'Pixel ID and Access Token are required to enable Meta Conversion API.', 'click-to-chat-pro' );
			}

			?>
		<p>
			<label>
				<input name="<?php echo esc_attr( $dbrow ); ?>[fb_conversion]" type="checkbox" value="1" <?php checked( $fb_conversion_checkbox, 1 ); ?> id="fb_conversion" />
				<span><?php esc_html_e( 'Facebook - Conversion Api - Server side Event (Beta)', 'click-to-chat-pro' ); ?></span>
			</label>
		</p>
		
		<div class="ctc_fb_capi_values ctc_init_display_none">

			<?php if ( ! empty( $fb_capi_error ) ) : ?>
				<p class="description" style="color:#d63638;"><?php echo esc_html( $fb_capi_error ); ?></p>
			<?php endif; ?>

			<p class="description">Send events to your server and forward them to the Meta Conversion API for improved attribution and conversion tracking.</p>

			<p class="description" style="margin-top: 10px;">
				<p style="margin: 0;">• Uses the same event name and parameters configured in the Meta Pixel settings above.</p>
				<p style="margin: 0;">• Sends enhanced data for better match rates and attribution:</p>
				<p style="margin-left: 20px;">• IP address (e.g., 192.168.1.1)</p>
				<p style="margin-left: 20px;">• Browser & device info (e.g., Chrome on macOS)</p>
				<p style="margin-left: 20px;">• Page URL (e.g., https://example.com/landing)</p>
				<!-- <div style="margin-left: 20px;">• User agent & referrer (for advanced diagnostics)</div> -->
			</p>

			<div class="row">
				<div class="input-field col s6">
					<p class="description"><?php esc_html_e( 'Facebook Pixel ID', 'click-to-chat-pro' ); ?></p>
				</div>
				<div class="input-field col s6">
					<input name="<?php echo esc_attr( $dbrow ); ?>[pixel_id]" value="<?php echo esc_attr( $pixel_id ); ?>" id="pixel_id" type="text" class="input-margin">
					<label for="pixel_id"><?php esc_html_e( 'Pixel ID', 'click-to-chat-pro' ); ?></label>
				</div>
			</div>

			<div class="row">
				<div class="input-field col s6">
					<p class="description"><?php esc_html_e( 'Access Token', 'click-to-chat-pro' ); ?></p>
				</div>
				<div class="input-field col s6">
					<!-- single line input only - so santize as text works here -->
					<textarea name="<?php echo esc_attr( $dbrow ); ?>[pixel_token]" id="pixel_token" class="materialize-textarea"><?php echo esc_textarea( $pixel_token ); ?></textarea>
					<label for="pixel_token"><?php esc_html_e( 'Access Token', 'click-to-chat-pro' ); ?></label>
				</div>
			</div>


			<div class="row">
				<div class="input-field col s6">
					<p class="description"><?php esc_html_e( 'Test Event Code', 'click-to-chat-pro' ); ?></p>
				</div>
				<div class="input-field col s6">
					<input name="<?php echo esc_attr( $dbrow ); ?>[test_event_code]" value="<?php echo esc_attr( $test_event_code ); ?>" id="test_event_code" type="text" class="input-margin">
					<label for="test_event_code"><?php esc_html_e( 'Test Event Code', 'click-to-chat-pro' ); ?></label>
					<p class="description" style="margin-top: 6px; font-size: 0.85rem; color: #757575;">
						<?php esc_html_e( 'This is only for testing purposes. Remove this value when going live.', 'click-to-chat-pro' ); ?>
					</p>
				</div>
			</div>


			<p class="description"><strong>(Beta feature)</strong> We’d love your feedback to improve this integration. Please <a href="https://holithemes.com/plugins/click-to-chat/support/" target="_blank">share your experience</a> and any issues you encounter.</p>
			<p class="description">Note: If you use a caching plugin, set the cache lifespan to less than 8 hours to ensure accurate event tracking.</p>
		</div>

		<br>
		<br>
			<?php
		}

		/**
		 * Google ads
		 * Conversion id, label
		 *
		 * @since 1.4 - checkbox.. compatible with v3.8
		 * @updated 2.5 added google ads params..
		 */
		public function google_ads() {

			$options = get_option( 'ht_ctc_othersettings' );
			$dbrow   = 'ht_ctc_othersettings';

			$ga_ads_checkbox       = ( isset( $options['ga_ads'] ) ) ? esc_attr( $options['ga_ads'] ) : '';
			$gads_conversion_id    = isset( $options['gads_conversion_id'] ) ? esc_attr( $options['gads_conversion_id'] ) : '';
			$gads_conversion_label = isset( $options['gads_conversion_label'] ) ? esc_attr( $options['gads_conversion_label'] ) : '';

			$g_ads_params = ( isset( $options['g_ads_params'] ) && is_array( $options['g_ads_params'] ) ) ? array_map( 'esc_attr', $options['g_ads_params'] ) : '';

			// count of g_ads params.. used for adding new params.. always raises..
			$g_ads_param_order = ( isset( $options['g_ads_param_order'] ) ) ? esc_attr( $options['g_ads_param_order'] ) : 1;
			// $key_gen = 1;
			?>

		<p>
			<label class="ctc_checkbox_label">
				<input name="<?php echo esc_attr( $dbrow ); ?>[ga_ads]" type="checkbox" value="1" <?php checked( $ga_ads_checkbox, 1 ); ?> id="ga_ads" />
				<span><?php esc_html_e( 'Google Ads Conversion', 'click-to-chat-pro' ); ?></span>
			</label>
		</p>

		<div class="ctc_g_ads_values ctc_init_display_none">
			<div class="row ctc_side_by_side">
				<div class="input-field col s6">
					<p class="description"><?php esc_html_e( 'Conversion ID', 'click-to-chat-pro' ); ?></p>
				</div>
				<div class="input-field col s6">
					<input name="<?php echo esc_attr( $dbrow ); ?>[gads_conversion_id]" value="<?php echo esc_attr( $gads_conversion_id ); ?>" id="gads_conversion_id" type="text" class="input-margin">
					<label for="gads_conversion_id"><?php esc_html_e( 'Conversion ID', 'click-to-chat-pro' ); ?></label>
				</div>
			</div>
			<div class="row ctc_side_by_side">
				<div class="input-field col s6">
					<p class="description"><?php esc_html_e( 'Conversion Label', 'click-to-chat-pro' ); ?></p>
				</div>
				<div class="input-field col s6">
					<input name="<?php echo esc_attr( $dbrow ); ?>[gads_conversion_label]" value="<?php echo esc_attr( $gads_conversion_label ); ?>" id="gads_conversion_label" type="text" class="input-margin">
					<label for="gads_conversion_label"><?php esc_html_e( 'Conversion Label', 'click-to-chat-pro' ); ?></label>
				</div>
			</div>

			<div class="row ctc_an_params ctc_g_ads_params ctc_sortable">

				<?php

				$num = '';

				if ( is_array( $g_ads_params ) && isset( $g_ads_params[0] ) ) {

					foreach ( $g_ads_params as $param ) {

						$param_options = ( isset( $options[ $param ] ) && is_array( $options[ $param ] ) ) ? map_deep( $options[ $param ], 'esc_attr' ) : '';

						$key   = ( isset( $param_options['key'] ) ) ? esc_attr( $param_options['key'] ) : '';
						$value = ( isset( $param_options['value'] ) ) ? esc_attr( $param_options['value'] ) : '';

						// if key and value not empty..
						if ( ! empty( $key ) && ! empty( $value ) ) {
							?>
							<div class="ctc_an_param g_ads_param row" style="margin-bottom:5px; display:flex; gap:5px; justify-content:center;">

								<input style="display: none;" name="ht_ctc_othersettings[g_ads_params][]" type="text" class="g_ads_param_order_ref_number" value="<?php echo esc_attr( $param ); ?>">

								<div class="input-field">
									<input name="ht_ctc_othersettings[<?php echo esc_attr( $param ); ?>][key]" value="<?php echo esc_attr( $key ); ?>" id="<?php echo esc_attr( $param . '_key' ); ?>" type="text" class="ht_ctc_g_ads_param_key input-margin">
									<label for="<?php echo esc_attr( $param . '_key' ); ?>"><?php esc_html_e( 'Event Parameter', 'click-to-chat-pro' ); ?></label>
								</div>

								<div class="input-field">
									<input name="ht_ctc_othersettings[<?php echo esc_attr( $param ); ?>][value]" value="<?php echo esc_attr( $value ); ?>" id="<?php echo esc_attr( $param ); ?>" type="text" class="ht_ctc_g_ads_param_value input-margin">
									<label for="<?php echo esc_attr( $param ); ?>"><?php esc_html_e( 'Value', 'click-to-chat-pro' ); ?></label>
								</div>

								<div class="input-field">
									<span style="color:#ddd; margin-left:auto; cursor:pointer;" class="an_param_remove dashicons dashicons-no-alt" title="Remove Parameter"></span>
								</div>


							</div>
							<?php
						}

						// $key_gen++;
					}
				}

				?>

				<!-- new fileds - for adding -->
				<!-- <div class="ctc_new_g_ads_param">
				</div> -->

				<!-- Add parameter - button -->
				<!-- <div style="text-align:center;">
					<div class="ctc_add_g_ads_param_button" style="display:inline-flex; margin: 10px 0px; cursor:pointer; font-size:16px; font-weight:500; padding: 8px; justify-content:center;">
						<span style="color: #039be5;" class="dashicons dashicons-plus-alt2" ></span>
						<span style="color: #039be5;">Add Parameter</span>
					</div>
				</div> -->
				
			</div>
		</div>

		<!-- snippets -->
		<div class="ctc_g_ads_param_snippets" style="display: none;">

			<!-- g_ads_param order. next key. (uses from js, saves in db) -->
			<input type="text" name="ht_ctc_othersettings[g_ads_param_order]" class="g_ads_param_order" value="<?php echo esc_attr( $g_ads_param_order ); ?>">

			
			<!-- snippet: add g_ads_param -->
			<div class="ctc_an_param g_ads_param ht_ctc_g_ads_add_param">

				<div class="row" style="display:flex; gap:5px; justify-content:center;">

					<input style="display: none;" type="text" class="g_ads_param_order_ref_number" value="<?php echo esc_attr( $g_ads_param_order ); ?>">

					<div class="input-field">
						<input type="text" placeholder="click" class="ht_ctc_g_ads_add_param_key input-margin">
						<label><?php esc_html_e( 'Event Parameter', 'click-to-chat-pro' ); ?></label>
					</div>

					<div class="input-field">
						<input type="text" placeholder="chat" class="ht_ctc_g_ads_add_param_value input-margin">
						<label><?php esc_html_e( 'Value', 'click-to-chat-pro' ); ?></label>
					</div>

					<div class="input-field">
						<span style="color:#ddd; margin-left:auto; cursor:pointer;" class="an_param_remove dashicons dashicons-no-alt" title="Remove Parameter"></span>
					</div>
					
				</div>

			</div>
			
		</div>
		
		<!-- todo:l we can add - click count, date, username, if woo... add product details, .... -->
		<!-- <p class="description" style="margin:0px 10px;">Variables: {title}, {url}, {number} replace page title, url, and number that are assigned to the widget.</p> -->

		<p class="description"><?php esc_html_e( 'Google Ads Conversion', 'click-to-chat-pro' ); ?> - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/google-ads-conversion/"><?php esc_html_e( 'more info', 'click-to-chat-pro' ); ?></a> </p>

			<?php
		}


		/**
		 * Admin notices
		 */
		public function admin_notice() {

			if ( defined( 'HT_CTC_VERSION' ) ) {

				if ( version_compare( HT_CTC_VERSION, HT_CTC_PRO_CTC_REQUIRED_VERSION, '<' ) ) {

					if ( version_compare( HT_CTC_VERSION, HT_CTC_PRO_CTC_REQUIRED_VERSION_TOWORK, '<' ) ) {
						// high priority notice - functions may stop working based on this
						add_action( 'admin_notices', array( $this, 'ctc_update_notice_important' ) );
					} else {
						// a plain notice - plugin wont stop working
						add_action( 'admin_notices', array( $this, 'ctc_update_notice' ) );
					}
				}
			}
		}

		/**
		 * CTC update notice
		 */
		public function ctc_update_notice() {
			?>
		<div class="notice notice-warning is-dismissible">
			<p><?php esc_html_e( "Please update the 'Click to Chat' plugin to the latest version, Ignore this message if already updated", 'click-to-chat-pro' ); ?>.</p>
		</div>
			<?php
		}

		/**
		 * CTC update notice important
		 */
		public function ctc_update_notice_important() {
			?>
		<div class="notice notice-error is-dismissible">
			<p><?php esc_html_e( "Please update the 'Click to Chat' plugin to the latest version, its important for some features to work, Ignore this message if already updated", 'click-to-chat-pro' ); ?>.</p>
		</div>
			<?php
		}

		/**
		 * Number
		 */
		public function number() {

			$options = get_option( 'ht_ctc_chat_options' );

			$number = ( isset( $options['number'] ) ) ? esc_attr( $options['number'] ) : '';

			// random numbers (array)
			$r_nums     = ( isset( $options['r_nums'] ) ) ? $options['r_nums'] : '';
			$count      = 1;
			$num        = '';
			$r_num_type = ( isset( $options['r_num_type'] ) ) ? esc_attr( $options['r_num_type'] ) : '';

			if ( is_array( $r_nums ) ) {
				// Filter out empty values from the array
				$r_nums = array_filter( $r_nums );
				// Re-index the array to ensure sequential keys
				$r_nums = array_values( $r_nums );
				// $r_nums = array_map('esc_attr', $r_nums );
				$count = count( $r_nums );
			}
			?>

			<div class="ht_ctc_numbers">
				<?php

				// Additional numbers
				if ( isset( $r_nums[0] ) ) {
					?>
					<div class="ctc_sortable">
					<?php
					for ( $i = 0; $i < $count; $i++ ) {
						$dbrow = "ht_ctc_chat_options[r_nums][$i]";
						$num   = esc_attr( $r_nums[ $i ] );

						if ( class_exists( 'HT_CTC_Formatting' ) && method_exists( 'HT_CTC_Formatting', 'wa_number' ) ) {
							$num = HT_CTC_Formatting::wa_number( $num );
							$num = "+$num";
						}

						?>
						<div class="additional-number" style="margin-bottom: 15px;">
							<p class="handle" style="display: flex;">
								<input type="text" name="<?php echo esc_attr( $dbrow ); ?>" data-name="<?php echo esc_attr( $dbrow ); ?>" class="intl_number browser-default" value="<?php echo esc_attr( $num ); ?>">
								<span style="color: #039be5; cursor: pointer;" class="remove_number dashicons dashicons-no-alt"></span>
							</p>
						</div>
						<?php
					}
					?>
					</div>
					<?php
				}

				?>

			</div>

			<div style="display:flex; align-items:center; width:100%; gap:10px; line-height:1.5;">
				<div class="add_number" style="display:flex; align-items:center; cursor:pointer;">
					<span class="dashicons dashicons-plus-alt2" style="font: size 20px; color:#039be5;"></span>
					<span style="color:#039be5;font-size:16px; font-weight:500; margin-left:5px;"><?php esc_html_e( 'Add Random Number', 'click-to-chat-pro' ); ?></span>
				</div>
				<span style="font-size:12px; white-space:nowrap; vertical-align:middle; line-height:1;">(<a target="_blank" href="https://holithemes.com/plugins/click-to-chat/random-number/">more info</a>)</span>
			</div>

			<!-- Random Number Type Selection -->
			<details class="ctc_details" style="margin: 12px 25px; font-size: 0.85em;">
				<summary style="cursor:pointer;">Random Number Type</summary>
				<div class="ctc_details_content">
					<p style="margin-bottom:12px;">
						<label for="r_num_type">
							<span class="description">How random numbers should be distributed.</span>
						</label><br>
						<select name="ht_ctc_chat_options[r_num_type]" class="select_r_num_type" style="margin-top: 6px;">
								<option value="pure" <?php selected( $r_num_type, 'pure' ); ?>>Pure Random</option>
							<option value="sequential" <?php selected( $r_num_type, 'sequential' ); ?>>Sequence</option>
						</select>
					</p>
					<div style="font-size: 13px; line-height: 1.6; color: #555;">
						<p><strong>Pure Random:</strong> Completely random selection with no predefined order or pattern.</p>
						<p><strong>Sequence:</strong> Numbers are selected in order to maintain a balanced distribution over time. <br> Triggers the server to fetch and update the latest value after the page load and user clicks on the widget.</p>
					</div>
				</div>
			</details>

				<p class="description" style="margin-top:12px; font-size:14px; color:#555;">Form filling and Multi-Agent features are on the <a target="_blank" href="<?php echo esc_url( admin_url( 'admin.php?page=click-to-chat-greetings' ) ); ?>" style="color: #039be5; font-weight: 500; text-decoration: none;">Greetings page</a></p>
			<div class="ctc_random_number_snippets ctc_init_display_none" style="display:none;">
				<div class="additional-number col s12" style="display:flex; padding-left:0px;">
					<div class="input-field col s12" style="margin:5px 0px; padding-left:0px;">
						<input type="text" class="browser-default ctc_add_auto_country_code" value="">
						<span style="cursor: pointer; color:#039be5;" class="remove_number dashicons dashicons-no-alt"></span>
					</div>
				</div>
			</div>

			<?php
		}


		/**
		 * Add 'absolute' to position type values.
		 *
		 * @param array $position_type_values Existing position type values.
		 * @return array Updated position type values including 'absolute'.
		 */
		public function add_position_type_values( $position_type_values ) {

			$position_type_values['absolute'] = 'Absolute';

			return $position_type_values;
		}

		/**
		 * All pages editor values that can be safely added.
		 *
		 * @param array $values Existing editor values.
		 * @return array Updated editor values including WooCommerce locations.
		 */
		public function editor_values( $values ) {

			$editor_values = array(
				'woo_single_header_content',
				'woo_single_main_content',
				'woo_single_bottom_content',
			);

			$values = array_merge( $values, $editor_values );

			return $values;
		}

		/**
		 * WooCommerce hook locations.
		 *
		 * @param array $woo_places Existing WooCommerce hook locations.
		 * @return array Updated WooCommerce hook locations including additional places.
		 */
		public function woo_places( $woo_places ) {

			$add_woo_places = array(
				'woocommerce_before_main_content'       => 'Before Main Content',
				'woocommerce_before_single_product'     => 'Before Product',
				'woocommerce_before_single_product_summary' => 'Before Product Summary',
				'woocommerce_single_product_summary'    => 'Product Summary',
				'woocommerce_before_add_to_cart_form'   => 'Before Add to Cart Form',
				'woocommerce_before_add_to_cart_button' => 'Before Cart Button',
				'woocommerce_after_add_to_cart_button'  => 'After Cart Button',
				'woocommerce_after_add_to_cart_form'    => 'After Add to Cart Form',
				'woocommerce_after_single_product'      => 'After Product',
			);

			$woo_places = array_merge( $woo_places, $add_woo_places );

			return $woo_places;
		}


		/**
		 * Meta box: position settings for the current post.
		 *
		 * @param WP_Post $current_post Current post object.
		 * @return string HTML for the meta box.
		 */
		public function chat_meta_box( $current_post ) {
			// Get saved meta values
			$ht_ctc_pagelevel = get_post_meta( $current_post->ID, 'ht_ctc_pagelevel', true );
			$dbrow            = 'ht_ctc_pagelevel';

			// Extract and sanitize saved values
			$timedelay          = isset( $ht_ctc_pagelevel['timedelay'] ) ? esc_attr( $ht_ctc_pagelevel['timedelay'] ) : '';
			$scroll             = isset( $ht_ctc_pagelevel['scroll'] ) ? esc_attr( $ht_ctc_pagelevel['scroll'] ) : '';
			$style_desktop      = isset( $ht_ctc_pagelevel['style_desktop'] ) ? esc_attr( $ht_ctc_pagelevel['style_desktop'] ) : '';
			$style_mobile       = isset( $ht_ctc_pagelevel['style_mobile'] ) ? esc_attr( $ht_ctc_pagelevel['style_mobile'] ) : '';
			$enable_greetings   = isset( $ht_ctc_pagelevel['enable_greetings'] ) ? esc_attr( $ht_ctc_pagelevel['enable_greetings'] ) : '';
			$greetings_template = isset( $ht_ctc_pagelevel['greetings_template'] ) ? esc_attr( $ht_ctc_pagelevel['greetings_template'] ) : '';

			// List of available greetings templates
			$g_templates = array(
				'no'              => '-- No Greetings Dialog --',
				'greetings-1'     => 'Greetings-1 - Customizable Design',
				'greetings-2'     => 'Greetings-2 - Content Specific',
				'greetings-pro-1' => 'Greetings - Form',
				'greetings-pro-2' => 'Multi Agent',
			);

			?>

		<div class="ht-ctc-meta-box">

				<h2 class="ht-ctc-meta-section-title">
					<?php esc_html_e( 'Settings', 'click-to-chat-pro' ); ?>
				</h2>

			<div class="ht-ctc-meta-field">
				<label for="ctc_time_delay"><?php esc_html_e( 'Time Delay', 'click-to-chat-pro' ); ?></label>
				<input type="number" id="ctc_time_delay" name="<?php echo esc_attr( $dbrow ); ?>[timedelay]" value="<?php echo esc_attr( $timedelay ); ?>">
			</div>

			<div class="ht-ctc-meta-field">
				<label for="ctc_scroll"><?php esc_html_e( 'Scroll Delay', 'click-to-chat-pro' ); ?></label>
				<input type="number" id="ctc_scroll" name="<?php echo esc_attr( $dbrow ); ?>[scroll]" value="<?php echo esc_attr( $scroll ); ?>">
			</div>

			<div class="ht-ctc-meta-field">
				<label for="style_desktop"><?php esc_html_e( 'Select Style (Desktop)', 'click-to-chat-pro' ); ?></label>
				<select id="style_desktop" name="<?php echo esc_attr( $dbrow ); ?>[style_desktop]">
					<option value="" <?php selected( $style_desktop, '' ); ?> ><?php esc_html_e( 'Select style', 'click-to-chat-pro' ); ?></option>
					<option value="1" <?php selected( $style_desktop, '1' ); ?> ><?php esc_html_e( 'Style-1', 'click-to-chat-pro' ); ?></option>
					<option value="2" <?php selected( $style_desktop, '2' ); ?> ><?php esc_html_e( 'Style-2', 'click-to-chat-pro' ); ?></option>
					<option value="3" <?php selected( $style_desktop, '3' ); ?> ><?php esc_html_e( 'Style-3', 'click-to-chat-pro' ); ?></option>
					<option value="3_1" <?php selected( $style_desktop, '3_1' ); ?> >&emsp;<?php esc_html_e( 'Style-3 Extend', 'click-to-chat-pro' ); ?></option>
					<option value="4" <?php selected( $style_desktop, '4' ); ?> ><?php esc_html_e( 'Style-4', 'click-to-chat-pro' ); ?></option>
					<option value="5" <?php selected( $style_desktop, '5' ); ?> ><?php esc_html_e( 'Style-5', 'click-to-chat-pro' ); ?></option>
					<option value="6" <?php selected( $style_desktop, '6' ); ?> ><?php esc_html_e( 'Style-6', 'click-to-chat-pro' ); ?></option>
					<option value="7" <?php selected( $style_desktop, '7' ); ?> ><?php esc_html_e( 'Style-7', 'click-to-chat-pro' ); ?></option>
					<option value="7_1" <?php selected( $style_desktop, '7_1' ); ?> >&emsp;<?php esc_html_e( 'Style-7 Extend', 'click-to-chat-pro' ); ?></option>
					<option value="8" <?php selected( $style_desktop, '8' ); ?> ><?php esc_html_e( 'Style-8', 'click-to-chat-pro' ); ?></option>
					<option value="99" <?php selected( $style_desktop, '99' ); ?> ><?php esc_html_e( 'Style-99 (Own Image/GIF)', 'click-to-chat-pro' ); ?></option>
				</select>
			</div>

			<div class="ht-ctc-meta-field">
				<label for="style_mobile"><?php esc_html_e( 'Select Style (Mobile)', 'click-to-chat-pro' ); ?></label>
				<select id="style_mobile" name="<?php echo esc_attr( $dbrow ); ?>[style_mobile]">
					<option value="" <?php selected( $style_mobile, '' ); ?> ><?php esc_html_e( 'Select style', 'click-to-chat-pro' ); ?></option>
					<option value="1" <?php selected( $style_mobile, '1' ); ?> ><?php esc_html_e( 'Style-1', 'click-to-chat-pro' ); ?></option>
					<option value="2" <?php selected( $style_mobile, '2' ); ?> ><?php esc_html_e( 'Style-2', 'click-to-chat-pro' ); ?></option>
					<option value="3" <?php selected( $style_mobile, '3' ); ?> ><?php esc_html_e( 'Style-3', 'click-to-chat-pro' ); ?></option>
					<option value="3_1" <?php selected( $style_mobile, '3_1' ); ?> >&emsp;<?php esc_html_e( 'Style-3 Extend', 'click-to-chat-pro' ); ?></option>
					<option value="4" <?php selected( $style_mobile, '4' ); ?> ><?php esc_html_e( 'Style-4', 'click-to-chat-pro' ); ?></option>
					<option value="5" <?php selected( $style_mobile, '5' ); ?> ><?php esc_html_e( 'Style-5', 'click-to-chat-pro' ); ?></option>
					<option value="6" <?php selected( $style_mobile, '6' ); ?> ><?php esc_html_e( 'Style-6', 'click-to-chat-pro' ); ?></option>
					<option value="7" <?php selected( $style_mobile, '7' ); ?> ><?php esc_html_e( 'Style-7', 'click-to-chat-pro' ); ?></option>
					<option value="7_1" <?php selected( $style_mobile, '7_1' ); ?> >&emsp;<?php esc_html_e( 'Style-7 Extend', 'click-to-chat-pro' ); ?></option>
					<option value="8" <?php selected( $style_mobile, '8' ); ?> ><?php esc_html_e( 'Style-8', 'click-to-chat-pro' ); ?></option>
					<option value="99" <?php selected( $style_mobile, '99' ); ?> ><?php esc_html_e( 'Style-99 (Own Image/GIF)', 'click-to-chat-pro' ); ?></option>
				</select>
			</div>

			<div class="ht-ctc-meta-field">
				<input type="checkbox" id="enable_greetings" name="<?php echo esc_attr( $dbrow ); ?>[enable_greetings]" value="1" <?php checked( $enable_greetings, 1 ); ?>>
				<label for="enable_greetings"><?php esc_html_e( 'Enable page level Greetings settings', 'click-to-chat-pro' ); ?></label>
			</div>

			<p class="ht-ctc-meta-description">
				<?php esc_html_e( 'Update and Refresh the page -', 'click-to-chat-pro' ); ?>
				<a href="https://holithemes.com/plugins/click-to-chat/greetings-page-level-settings/" target="_blank">More Info</a>
			</p>
		</div>
			<?php

			if ( '1' === $enable_greetings ) {
				$allowed_html = wp_kses_allowed_html( 'post' );

				$header_content = ( isset( $ht_ctc_pagelevel['header_content'] ) ) ? $ht_ctc_pagelevel['header_content'] : '';
				if ( '' !== $header_content ) {
					$header_content = html_entity_decode( wp_kses( $header_content, $allowed_html ) );
				}

				$main_content = ( isset( $ht_ctc_pagelevel['main_content'] ) ) ? $ht_ctc_pagelevel['main_content'] : '';
				if ( '' !== $main_content ) {
					$main_content = html_entity_decode( wp_kses( $main_content, $allowed_html ) );
				}

				$bottom_content = ( isset( $ht_ctc_pagelevel['bottom_content'] ) ) ? $ht_ctc_pagelevel['bottom_content'] : '';
				if ( '' !== $bottom_content ) {
					$bottom_content = html_entity_decode( wp_kses( $bottom_content, $allowed_html ) );
				}

				$g_call_to_action = isset( $ht_ctc_pagelevel['g_call_to_action'] ) ? esc_attr( $ht_ctc_pagelevel['g_call_to_action'] ) : '';

				if ( ! function_exists( 'ctc_meta_tinymce_mce_buttons_2' ) ) {
					/**
					 * Add font select and font size select to the second row of TinyMCE editor buttons.
					 *
					 * @param array $buttons Existing buttons in the second row.
					 * @return array Modified buttons including font select and font size select.
					 */
					function ctc_meta_tinymce_mce_buttons_2( $buttons ) {
						$key = array_search( 'forecolor', $buttons, true );
						if ( false && is_int( $key ) !== $key ) {
							array_splice( $buttons, $key + 1, 0, 'backcolor' );
						}
						array_unshift( $buttons, 'fontselect' );
						array_push( $buttons, 'fontsizeselect' );
						return $buttons;
					}
				}
				add_filter( 'mce_buttons_2', 'ctc_meta_tinymce_mce_buttons_2' );

				$editor_args = array(
					'textarea_rows' => 5,
					'media_buttons' => false,
					// 'tinymce'       => true,
					'tinymce'       => array(
						'textarea_rows'    => 10,
						'fontsize_formats' => '6px 8px 10px 12px 13px 14px 15px 16px 18px 20px 24px 28px 32px 36px',
						'content_style'    => 'body { background-color: #26a69a; }',
					),
					// 'quicktags' => true,
					// 'quicktags' => array( 'buttons' => 'em,strong,link' ),
					// 'teeny'         => true,
					// 'editor_height' => 150,
					// 'drag_drop_upload' => true,
				);
				?>
			<div class="ht-ctc-meta-box">
				<h2 class="ht-ctc-meta-section-title"><?php esc_html_e( 'Greetings Template', 'click-to-chat-pro' ); ?></h2>

				<div class="ht-ctc-meta-field">
					<label for="greetings_template"><?php esc_html_e( 'Select Template', 'click-to-chat-pro' ); ?></label>
					<select id="greetings_template" name="<?php echo esc_attr( $dbrow ); ?>[greetings_template]">
						<option value=""><?php esc_html_e( '-- Default --', 'click-to-chat-pro' ); ?></option>
						<?php foreach ( $g_templates as $k => $v ) : ?>
							<option value="<?php echo esc_attr( $k ); ?>" <?php selected( $greetings_template, $k ); ?>><?php echo esc_html( $v ); ?></option>
						<?php endforeach; ?>
					</select>
				</div>

				<div class="ht-ctc-meta-field">
					<label><?php esc_html_e( 'Greetings - Header Content', 'click-to-chat-pro' ); ?></label>
					<?php
					$args['textarea_name'] = 'ht_ctc_pagelevel[header_content]';
					wp_editor( $header_content, 'ctc_header_content', array_merge( $editor_args, $args ) );
					?>
				</div>

				<div class="ht-ctc-meta-field">
					<label><?php esc_html_e( 'Greetings - Main Content', 'click-to-chat-pro' ); ?></label>
						<?php
						$args['textarea_name'] = 'ht_ctc_pagelevel[main_content]';
						wp_editor( $main_content, 'ctc_main_content', array_merge( $editor_args, $args ) );
						?>
				</div>

				<div class="ht-ctc-meta-field">
					<label><?php esc_html_e( 'Greetings - Bottom Content', 'click-to-chat-pro' ); ?></label>
						<?php
						$args['textarea_name'] = 'ht_ctc_pagelevel[bottom_content]';
						wp_editor( $bottom_content, 'ctc_bottom_content', array_merge( $editor_args, $args ) );
						?>
				</div>

				<div class="ht-ctc-meta-field">
					<label for="g_call_to_action"><?php esc_html_e( 'Call to Action', 'click-to-chat-pro' ); ?></label>
					<input type="text" id="g_call_to_action" name="<?php echo esc_attr( $dbrow ); ?>[g_call_to_action]" value="<?php echo esc_attr( $g_call_to_action ); ?>">
				</div>
			</div>

				<?php
			}
		}




		/**
		 * Metabox: custom link
		 *
		 * Commented while calling. (planning to relese in later version)
		 *
		 * @param WP_Post $current_post Current post object.
		 */
		public function chat_meta_box_after_number( $current_post ) {
			$ht_ctc_pagelevel = get_post_meta( $current_post->ID, 'ht_ctc_pagelevel', true );

			$dbrow        = 'ht_ctc_pagelevel';
			$custom_url_d = ( isset( $ht_ctc_pagelevel['custom_url_d'] ) ) ? esc_attr( $ht_ctc_pagelevel['custom_url_d'] ) : '';
			$custom_url_m = ( isset( $ht_ctc_pagelevel['custom_url_m'] ) ) ? esc_attr( $ht_ctc_pagelevel['custom_url_m'] ) : '';

			$options = get_option( 'ht_ctc_chat_options' );

			// placeholders
			$ph_custom_url_d = '';
			$ph_custom_url_m = '';
			if ( is_array( $options ) ) {
				$ph_custom_url_d = ( isset( $options['custom_url_d'] ) ) ? esc_attr( $options['custom_url_d'] ) : '';
				$ph_custom_url_m = ( isset( $options['custom_url_m'] ) ) ? esc_attr( $options['custom_url_m'] ) : '';
			}

			?>
		<div class="ht-ctc-meta-box">
				<!-- <h2 class="ht-ctc-meta-section-title">Custom URL</h2> -->
			
			<p class="description"><?php esc_html_e( 'Custom URL have high priority then number', 'click-to-chat-pro' ); ?> - <a style="text-decoration: none" target="_blank" href="https://holithemes.com/plugins/click-to-chat/custom-url/">Custom URL</a></p>
			<div class="ht-ctc-meta-field">
				<label for="custom_url_d"><?php esc_html_e( 'Custom URL: Desktop', 'click-to-chat-pro' ); ?></label>
				<input name="<?php echo esc_attr( $dbrow ); ?>[custom_url_d]" value="<?php echo esc_attr( $custom_url_d ); ?>" id="custom_url_d" type="text" placeholder="<?php echo esc_attr( $ph_custom_url_d ); ?>">
				<!-- <span class="helper">E.g., WhatsApp Channel URL</span> -->
			</div>
			
			<div class="ht-ctc-meta-field">
				<label for="custom_url_m"><?php esc_html_e( 'Custom URL: Mobile', 'click-to-chat-pro' ); ?></label>
				<input name="<?php echo esc_attr( $dbrow ); ?>[custom_url_m]" value="<?php echo esc_attr( $custom_url_m ); ?>" id="custom_url_m" type="text" placeholder="<?php echo esc_attr( $ph_custom_url_m ); ?>">
			</div>
			
		</div>

			<?php
		}


		/**
		 * After_showhide - above show/hide display settings
		 *
		 * @since 1.1 - logged_in_only
		 * @updated 1.3 - all_users, logged_in, logged_out
		 */
		public function after_showhide() {
			$options = get_option( 'ht_ctc_chat_options' );
			$dbrow   = 'ht_ctc_chat_options';
			$type    = 'chat';

			$timedelay = ( isset( $options['timedelay'] ) ) ? esc_attr( $options['timedelay'] ) : '';
			$scroll    = ( isset( $options['scroll'] ) ) ? esc_attr( $options['scroll'] ) : '';

			$logged_in_only = ( isset( $options['logged_in_only'] ) ) ? esc_attr( $options['logged_in_only'] ) : '';

			// since 1.3
			$display_user_base = ( isset( $options['display_user_base'] ) ) ? esc_attr( $options['display_user_base'] ) : 'all_users';

			// previous version compatibility
			if ( isset( $options['logged_in_only'] ) ) {
				$display_user_base = 'logged_in';
			}
			?>

		<p class="description ht_ctc_pro_subtitle" style="margin-bottom: 10px; border-color: #8fbc8f;">PRO Settings: </p>

		<!-- display only to log in users -->
		<div class="row ctc_side_by_side">
			<div class="col s6">
				<p><?php esc_html_e( 'Display based on User Login Status', 'click-to-chat-pro' ); ?></p>
			</div>
			<div class="input-field col s6" style="margin-top: 0px;">
				<select name="<?php echo esc_attr( $dbrow ); ?>[display_user_base]" class="display_select_user_base">
					<option value="all_users" <?php selected( $display_user_base, 'all_users' ); ?> ><?php esc_html_e( 'All Users', 'click-to-chat-pro' ); ?></option>
					<option value="logged_in" <?php selected( $display_user_base, 'logged_in' ); ?> ><?php esc_html_e( 'Logged-in Users only', 'click-to-chat-pro' ); ?></option>
					<option value="logged_out" <?php selected( $display_user_base, 'logged_out' ); ?> ><?php esc_html_e( 'Logged-out Users only (not logged-in)', 'click-to-chat-pro' ); ?></option>
				</select>
				<p class="description">Display based on website visitor login Status</p>
				<p class="description">All Users: Logged-in and not-logged-in users</p>
			</div>
		</div>

		<!-- time delay -->
		<div class="row ctc_side_by_side">
			<div class="col s6">
				<p><?php esc_html_e( 'Time Delay', 'click-to-chat-pro' ); ?></p>
			</div>
			<div class="input-field col s6">
				<input name="<?php echo esc_attr( $dbrow ); ?>[timedelay]" value="<?php echo esc_attr( $timedelay ); ?>" id="timedelay" type="number" min="0" class="" >
				<label for="timedelay"><?php esc_html_e( 'Time Delay', 'click-to-chat-pro' ); ?></label>
				<p class="description"><?php esc_html_e( 'Display After this number of seconds. E.g. Add 10 to display after 10 seconds', 'click-to-chat-pro' ); ?></p>
			</div>
		</div>

		<!-- after page scroll percentage -->
		<div class="row ctc_side_by_side">
			<div class="col s6">
				<p><?php esc_html_e( 'Scroll', 'click-to-chat-pro' ); ?></p>
			</div>
			<div class="input-field col s6">
				<input name="<?php echo esc_attr( $dbrow ); ?>[scroll]" value="<?php echo esc_attr( $scroll ); ?>" id="scroll" type="number" min="0" class="" >
				<label for="scroll"><?php esc_html_e( 'Scroll Delay', 'click-to-chat-pro' ); ?></label>
				<p class="description"><?php esc_html_e( 'Display After user scrolled this percentage of page. E.g. Add 10 to display after user scrolled 10% of page', 'click-to-chat-pro' ); ?></p>
			</div>
		</div>


			<?php

			/**
			 * Display based on country code
			 *
			 * @var display_countries select all/selected
			 * @var display_countries_list selected countries list..
			 *
			 * @since 2.6
			 */

			// include countries
			include_once HT_CTC_PRO_PLUGIN_DIR . 'inc/tools/class-ht-ctc-pro-commons.php';
			$countries = array();

			// if HT_CTC_PRO_Commons::countries exists..
			if ( class_exists( 'HT_CTC_PRO_Commons' ) && method_exists( 'HT_CTC_PRO_Commons', 'countries_list' ) ) {
				$countries = HT_CTC_PRO_Commons::countries_list();
			}

			// all / selected
			$display_countries = ( isset( $options['display_countries'] ) ) ? esc_attr( $options['display_countries'] ) : 'all';

			// retun array. uses to check if selected.
			$display_countries_list = ( isset( $options['display_countries_list'] ) ) ? $options['display_countries_list'] : array();
			$display_countries_list = ( is_array( $display_countries_list ) && ! empty( $display_countries_list ) ) ? array_map( 'esc_attr', $display_countries_list ) : array();
			?>

		<!-- Countries -->
		<div class="row ctc_side_by_side">

			<div class="col s6">
				<p>Display based on country <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/display-based-on-country/"><span class="dashicons dashicons-external"></span></a></p>
			</div>

			<div class="col s6">

				<div class="">
					<select name="<?php echo esc_attr( $dbrow ); ?>[display_countries]" class="select_display_countries">
						<option value="all" <?php selected( $display_countries, 'all' ); ?> >All Countries</option>
						<option value="selected" <?php selected( $display_countries, 'selected' ); ?> >Only selected Countries</option>
					</select>
				</div>

				<div class="ctc_display_countries_base" style="margin: 20px 0px;">
					<select name="<?php echo esc_attr( $dbrow ); ?>[display_countries_list][]" class="display_countries_list browser-default" multiple id="">
						<?php
						foreach ( $countries as $k => $v ) {
							?>
							<option value="<?php echo esc_attr( $k ); ?>" <?php selected( in_array( $k, $display_countries_list, true ), true ); ?> ><?php echo esc_html( $v ); ?></option>
							<?php
						}
						?>
					</select>
				</div>

			</div>

		</div>
	
			<?php
		}


		/**
		 * Business Hours (bh)
		 */
		public function bh() {
			$options = get_option( 'ht_ctc_chat_options' );
			$dbrow   = 'ht_ctc_chat_options';
			$type    = 'chat';

			$off_num = ( isset( $options['off_num'] ) ) ? esc_attr( $options['off_num'] ) : '';
			$off_cta = ( isset( $options['off_cta'] ) ) ? esc_attr( $options['off_cta'] ) : '';

			// default/main values
			$d_number = ( isset( $options['number'] ) ) ? esc_attr( $options['number'] ) : '';
			$d_cta    = ( isset( $options['call_to_action'] ) ) ? esc_attr( $options['call_to_action'] ) : '';

			$bh        = ( isset( $options['bh'] ) ) ? esc_attr( $options['bh'] ) : '';
			$monday    = ( isset( $options['monday'] ) ) ? esc_attr( $options['monday'] ) : '';
			$tuesday   = ( isset( $options['tuesday'] ) ) ? esc_attr( $options['tuesday'] ) : '';
			$wednesday = ( isset( $options['wednesday'] ) ) ? esc_attr( $options['wednesday'] ) : '';
			$thursday  = ( isset( $options['thursday'] ) ) ? esc_attr( $options['thursday'] ) : '';
			$friday    = ( isset( $options['friday'] ) ) ? esc_attr( $options['friday'] ) : '';
			$saturday  = ( isset( $options['saturday'] ) ) ? esc_attr( $options['saturday'] ) : '';
			$sunday    = ( isset( $options['sunday'] ) ) ? esc_attr( $options['sunday'] ) : '';

			?>

		<ul class="collapsible" id="ht_ctc_bh">
		<li class="active">
		<div class="collapsible-header"><?php esc_html_e( 'PRO Settings: Business hours - Offline/Online', 'click-to-chat-pro' ); ?>
		<span class="right_icon dashicons dashicons-arrow-down-alt2"></span>
		</div>
		<div class="collapsible-body">

			<!-- business hours -->
			<div class="row ctc_side_by_side">
				<div class="col s6" style="padding-top: 14px;">
					<p><?php esc_html_e( 'Business Hours', 'click-to-chat-pro' ); ?> <br> (<?php esc_html_e( 'Online/Offline', 'click-to-chat-pro' ); ?>)</p>
				</div>
				<div class="input-field col s6">
					<select name="<?php echo esc_attr( $dbrow ); ?>[bh]" class="select_bh">
						<option value="always" <?php selected( $bh, 'always' ); ?> ><?php esc_html_e( 'Always Open (Online)', 'click-to-chat-pro' ); ?></option>
						<option value="timebase" <?php selected( $bh, 'timebase' ); ?> ><?php esc_html_e( 'Selected Days, Hours', 'click-to-chat-pro' ); ?></option>
					</select>
				</div>
			</div>

			<p class="description bh_time">Current Site Time: <code><?php echo esc_html( current_time( 'mysql' ) ); ?></code> ( Settings -> General - Timezone )</p>
			<p class="description bh_time" style="margin: 15px 0px; text-align:center; font: bold;">After set the Business Hours - please check <a href="#basedon_business_hours">Settings based on Business Hours</a> (Hide widget, change number, ..)</p>
			<br class="bh_time">

			<div class="bh_time_schedule bh_time">

				<?php

				// since v2.15: multiple time slots for each week day
				$days = array(
					'monday',
					'tuesday',
					'wednesday',
					'thursday',
					'friday',
					'saturday',
					'sunday',
				);

				foreach ( $days as $day ) {

					$st = $day . '_st';
					$et = $day . '_et';

					// if checked set to 1. else blank.  the value can be saved as 'on' or '1'. i.e. checked. if not set then off. (previous version compatibility)
					$day_checked = ( isset( $options[ $day ] ) ) ? '1' : '';
					$day_times   = ( isset( $options[ $day . '_times' ] ) ) ? $options[ $day . '_times' ] : '';

					$bh_add_time_here_class = ( '1' === $day_checked ) ? '' : 'ctc_init_display_none';
					?>

					<div class="bh_schedule" style="margin-bottom: 12px; border: 1px solid #dddddd; padding: 20px;" data-day="<?php echo esc_attr( $day ); ?>">
						<div class="flex" style="display:flex; align-items: center; gap: 12px;">

							<!-- days checkbox -->
							<div class="input-field" style="min-width: 130px; margin: 1px 0px;">
								<p>
									<label>
										<input class="bh_day_checkbox" name="<?php echo esc_attr( $dbrow ); ?>[<?php echo esc_attr( $day ); ?>]" type="checkbox" value="1" <?php checked( $day_checked, 1 ); ?> id="<?php echo esc_attr( 'bh_' . $day ); ?>">
										<span><?php echo esc_html( ucfirst( $day ) ); ?></span>
									</label>
								</p>
							</div>

							<div style="display: flex; flex-direction: column;">
							<?php
							if ( is_array( $day_times ) ) { // && $day_times[0]
								?>
								<div class="bh_add_time_here <?php echo esc_attr( $bh_add_time_here_class ); ?> ">
								<?php
								$set_count = 1;
								foreach ( $day_times as $set ) {

									$start_time = ( isset( $set['st'] ) ) ? esc_attr( $set['st'] ) : '';
									$end_time   = ( isset( $set['et'] ) ) ? esc_attr( $set['et'] ) : '';

									/**
									 * Is start or time is set then only end..
									 * and if only one is added then other one is 0: 00;
									 */
									if ( '' === $start_time && '' === $end_time ) {
										continue;
									}
									// by above if - alteast one is added....

									if ( '' === $start_time || '' === $end_time ) {
										$start_time = ( '' !== $start_time ) ? $start_time : '0: 00';
										$end_time   = ( '' !== $end_time ) ? $end_time : '0: 00';
									}
									?>
									<div class="bh_add_time" style="display: flex; flex-direction: column;">
										<div style="display: flex;">
											<div class="input-field ctc_time_<?php echo esc_attr( $day ); ?>" style="min-width: 130px; margin: 1px 0px;">
												<input name="<?php echo esc_attr( $dbrow ); ?>[<?php echo esc_attr( $day . '_times' ); ?>][set_<?php echo esc_attr( $set_count ); ?>][st]" placeholder="" value="<?php echo esc_attr( $start_time ); ?>" id="<?php echo esc_attr( 'bh_' . $day . '_times_set_' . $set_count . '_st' ); ?>" type="text" class="ctc_j_timepicker_start">
												<span class="helper-text">Start time</span>
											</div>
											<div class="input-field ctc_time_<?php echo esc_attr( $day ); ?>" style="min-width: 130px; margin: 1px 10px;">
												<input name="<?php echo esc_attr( $dbrow ); ?>[<?php echo esc_attr( $day . '_times' ); ?>][set_<?php echo esc_attr( $set_count ); ?>][et]" placeholder="" value="<?php echo esc_attr( $end_time ); ?>" id="<?php echo esc_attr( 'bh_' . $day . '_times_set_' . $set_count . '_et' ); ?>" type="text" class="ctc_j_timepicker_end">
												<span class="helper-text">End time</span>
											</div>
											<span style="color:#dddddd;float:right;cursor:pointer;" class="ht_ctc_pro_bh_remove_time_set_link dashicons dashicons-no-alt" title="Remove Time"></span>
										</div>
										<!-- <span class="agent_set_time_range" style="color:#aeaeae;">Time set for __H __M from __:__ to __:__ </span> -->
									</div>
									<?php
									++$set_count;
								}
								?>
									<div class="bh_add_time_set" style="display: flex; flex-direction: column;">
									</div>
									<div class="today_schedule"><span class="bh_online_24_content" style="display: none;">24 Hours online&emsp;(or)&emsp;</span><span class="bh_click_to_add_time" style="color: #039be5; cursor: pointer; float: right;">Add Time</span></div>
								</div>
								<?php
							} else {
								?>
								<div class="bh_add_time_here <?php echo esc_attr( $bh_add_time_here_class ); ?>">
									<div class="bh_add_time_set" style="display: flex; flex-direction: column;">
									</div>
									<div class="today_schedule"><span class="bh_online_24_content">24 Hours online&emsp;(or)&emsp;</span><span class="bh_click_to_add_time" style="color: #039be5; cursor: pointer; float: right;">Add Time</span></div>
								</div>
								<?php
							}
							?>
							</div>
						</div>
					</div>


					<?php

				}

				?>

				<!-- snippet: bh add time -->
				<div class="bh_snippets" style="display:none">
					<div class="bh_add_time" style="display: flex;">
						<div class="input-field" style="min-width: 130px; margin: 1px 0px;">
							<input placeholder="" type="text" class="bh_ctc_j_timepicker_start add_time_st">
							<span class="helper-text">Start time</span>
						</div>
						<div class="input-field" style="min-width: 130px; margin: 1px 10px;">
							<input placeholder="" type="text" class="bh_ctc_j_timepicker_end add_time_et">
							<span class="helper-text">End time</span>
						</div>
						<span style="color:#dddddd; float:right; cursor:pointer;" class="ht_ctc_pro_bh_remove_time_set_link dashicons dashicons-no-alt" title="Remove Agent"></span>
					</div>
				</div>

			</div>

			
			<p class="bh_time" style="margin-bottom: 59px"></p>

			<hr class="bh_time" id="basedon_business_hours" style="margin-bottom: 59px;">

			<!-- offline - Hide -->
			<div class="row bh_time">
				<div class="col s6">
						<p class="description"><?php esc_html_e( 'Hide when Offline', 'click-to-chat-pro' ); ?></p>
				</div>
				
				<div class="col s6">

					<?php
					if ( isset( $options['off_hide'] ) ) {
						?>
						<p>
							<label>
								<input name="<?php echo esc_attr( $dbrow ); ?>[off_hide]" type="checkbox" value="1" <?php checked( $options['off_hide'], 1 ); ?> id="off_hide" class="off_hide"/>
									<span><?php esc_html_e( 'Hide WhatsApp When Offline', 'click-to-chat-pro' ); ?></span>
							</label>
						</p>
						<?php
					} else {
						?>
						<p>
							<label>
								<input name="<?php echo esc_attr( $dbrow ); ?>[off_hide]" type="checkbox" value="1" id="off_hide" class="off_hide"/>
									<span><?php esc_html_e( 'Hide WhatsApp When Offline', 'click-to-chat-pro' ); ?></span>
							</label>
						</p>
						<?php
					}
					?>
					<p class="description"><?php esc_html_e( 'Hide WhatsApp When offline', 'click-to-chat-pro' ); ?> - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/offline-hide/">more info</a> </p>
				</div>
			</div>
			
			<!-- offline - number change -->
			<div class="row bh_time offline_hide">
				<div class="input-field col s12">
					<input name="<?php echo esc_attr( $dbrow ); ?>[off_num]" value="<?php echo esc_attr( $off_num ); ?>" id="off_num" type="text" placeholder="<?php echo esc_attr( $d_number ); ?>" class="input-margin tooltipped" data-position="top" data-tooltip="Country Code + Number">
						<label for="off_num"><?php esc_html_e( 'Offline: Number', 'click-to-chat-pro' ); ?></label>
						<p class="description"><?php esc_html_e( 'Change Number when offline, leave blank to not change WhatsApp number when offline', 'click-to-chat-pro' ); ?> </p>
					<!-- - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/offline-number/">more info</a> -->
				</div>
			</div>


			<!-- Offline - Call to Action -->
			<div class="row bh_time offline_hide">
				<div class="input-field col s12">
					<input name="<?php echo esc_attr( $dbrow ); ?>[off_cta]" value="<?php echo esc_attr( $off_cta ); ?>" id="off_cta" type="text" placeholder="<?php echo esc_attr( $d_cta ); ?>" class="input-margin">
						<label for="off_cta"><?php esc_html_e( 'Offline: Call to Action', 'click-to-chat-pro' ); ?></label>
						<p class="description"><?php esc_html_e( 'Change Call to Action when offline, leave blank to not change call to action when offline', 'click-to-chat-pro' ); ?> </p>
					<!-- - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/offline-call-to-action/">more info</a> -->
				</div>
			</div>
			

			<p class="description" style="margin-bottom: 40px;">Settings based on Business Hours - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/business-hours-online-offline/">more info</a></p>


		</div>
		</li>
		</ul>

			<?php
		}

		/**
		 * Admin scripts hook
		 */
		public function admin_scripts() {

			$os = get_option( 'ht_ctc_othersettings' );

			// js
			$js = 'admin_pro.js';

			if ( defined( 'HT_CTC_PRO_DEBUG_MODE' ) ) {
				$js = 'admin_pro.dev.js';
			}

			if ( defined( 'HT_CTC_PRO_PLUGIN_FILE' ) ) {
				wp_enqueue_script( 'ctc_admin_pro_js', plugins_url( "admin/admin_assets/js/$js", HT_CTC_PRO_PLUGIN_FILE ), array( 'jquery' ), HT_CTC_PRO_VERSION, true );
			}
		}

		/**
		 * Admin register scripts
		 *
		 * @param string $hook Hook name.
		 */
		public function admin_register_scripts( $hook ) {

			if ( defined( 'HT_CTC_PRO_PLUGIN_FILE' ) ) {

				if ( 'click-to-chat_page_click-to-chat-greetings' === $hook ) {
					// wp media uploader
					wp_enqueue_media();

					/**
					 * Intl input - for multi agent
					 * registered at main plugin - class-ht-ctc-admin-scripts
					 *
					 * IntlTelInput files are at main plugin
					 */
					wp_enqueue_style( 'ctc_admin_intl_css' );
					wp_enqueue_script( 'ctc_admin_intl_js' );

					$args = true;
					global $wp_version;
					// if wp version is not null and is greater than 6.3
					if ( $wp_version && version_compare( $wp_version, '6.3', '>=' ) ) {
						$args = array(
							'in_footer' => true,
							'strategy'  => 'defer',
						);
					}

					wp_enqueue_script( 'ctc_pro_admin_greetings_js', plugins_url( 'admin/admin_assets/js/greetings.js', HT_CTC_PRO_PLUGIN_FILE ), array( 'jquery' ), HT_CTC_PRO_VERSION, $args );

				}

				if ( 'toplevel_page_click-to-chat' === $hook ) {
					// sumoselect
					wp_enqueue_style( 'ctc_admin_sumoselect_css', plugins_url( 'admin/admin_assets/lib/sumoselect/sumoselect.min.css', HT_CTC_PRO_PLUGIN_FILE ), '', HT_CTC_PRO_VERSION );
					wp_enqueue_script( 'ctc_admin_sumoselect_js', plugins_url( 'admin/admin_assets/lib/sumoselect/jquery.sumoselect.min.js', HT_CTC_PRO_PLUGIN_FILE ), array( 'jquery' ), HT_CTC_PRO_VERSION, true );
				}

				if ( 'click-to-chat_page_click-to-chat-greetings' === $hook || 'toplevel_page_click-to-chat' === $hook ) {
					// jQuery timepicker
					// should not load in main settings - where md timepicker is using.. todo: now md timepicker have to remove like..
					wp_enqueue_style( 'ctc_admin_pro_timepicker_css', plugins_url( 'admin/admin_assets/lib/timepicker/jquery.timepicker.min.css', HT_CTC_PRO_PLUGIN_FILE ), '', HT_CTC_PRO_VERSION );
					wp_enqueue_script( 'ctc_admin_pro_timepicker_js', plugins_url( 'admin/admin_assets/lib/timepicker/jquery.timepicker.min.js', HT_CTC_PRO_PLUGIN_FILE ), array( 'jquery' ), HT_CTC_PRO_VERSION, true );
				}
			}
		}


		/**
		 * Greetings Dialog
		 *
		 * Actions
		 *  Time
		 *  Scroll
		 *
		 * @param array $values Values array.
		 */
		public function greetings_settings( $values ) {

			$options  = get_option( 'ht_ctc_chat_options' );
			$pro_path = HT_CTC_PRO_PLUGIN_DIR . 'admin/components';

			$inputs = array(
				array(
					'template'     => 'heading',
					'title'        => 'Actions',
					'parent_class' => 'greetings_actions ctc_greetings_settings ctc_g_1 ctc_g_2',
				),
				array(
					'template'     => 'content',
					'description'  => '<div style="margin:0px 14px 0px 14px;"><p class="description"><a href="https://holithemes.com/plugins/click-to-chat/greetings-actions/">Greetings Actions:</a> Displays greetings based on specific triggers</p><p class="description"><strong style="font-weight: 500;">Click:</strong> Clicked on any element with Class name: \'ctc_greetings\'</p><p class="description"><strong style="font-weight: 500;">Viewport [PRO]:</strong> An element is in/reached viewport (25% margin) with Class name: \'ctc_greetings_now\'</p></div>',
					'parent_class' => 'pr_g_time_action greetings_actions ctc_greetings_settings ctc_g_1 ctc_g_2',
				),
				array(
					'title'        => 'Time',
					'db'           => 'g_time_action',
					'template'     => 'number',
					'min'          => '0',
					'label'        => 'Time Action',
					'description'  => 'Automatically displays after this number of seconds',
					'parent_class' => 'pr_g_time_action greetings_actions ctc_greetings_settings ctc_g_1 ctc_g_2 ctc_no_demo',
				),
				array(
					'title'        => 'Scroll',
					'db'           => 'g_scroll_action',
					'template'     => 'number',
					'min'          => '0',
					'label'        => 'Scroll Action',
					'description'  => 'Automatically displays after this percentage of page scroll',
					'parent_class' => 'pr_g_scroll_action greetings_actions ctc_greetings_settings ctc_g_1 ctc_g_2 ctc_no_demo',
				),
				array(
					'title'        => "If the user closes Greetings, Don't reopen based on time, scroll",
					'db'           => 'g_no_reopen',
					'template'     => 'checkbox',
					'label'        => 'Disable time, scroll Actions',
					'description'  => "If User closes the greetings dialog on any page, then greetings won't reopen based on time, scroll action",
					'parent_class' => 'pr_g_scroll_action greetings_actions ctc_greetings_settings ctc_g_1 ctc_g_2 ctc_no_demo',
				),
			);

			foreach ( $inputs as $input ) {
				// array_push($values['main']['inputs'], $input);
				array_push( $values['greetings_settings']['inputs'], $input );
			}

			$admin_email = get_option( 'admin_email' );

			$g_form_fallback_values = array(
				'empty'                => '1',
				'fields'               => array(
					'field_1',
					'field_2',
				),
				'field_1'              => array(
					'type'             => 'text',
					'name'             => 'Name',
					'placeholder'      => 'Name',
					'add_to_prefilled' => '1',
				),
				'field_2'              => array(
					'type'             => 'email',
					'name'             => 'Email',
					'placeholder'      => 'Email',
					'add_to_prefilled' => '1',
				),
				'field_count'          => '4',
				'email'                => $admin_email,
				'webhook'              => '',
				'header_bg_color'      => '#075e54',
				'main_bg_color'        => '#ece5dd',
				'message_box_bg_color' => '',
				'cta_style'            => '7_1',
			);

			$g_multiagent_fallback_values = array(
				'empty'                => '1',
				'header_bg_color'      => '#00a884', // #075e54
				'main_bg_color'        => '#ece5dd',
				'message_box_bg_color' => '#dcf8c6',
				'cta_style'            => '7_1',
			);

			$main_number = ( isset( $options['number'] ) ) ? esc_attr( $options['number'] ) : '';

			// if main number added then add to agent fallback
			if ( '' !== $main_number ) {
				$g_multiagent_fallback_values['agents']      = array(
					'agent_1',
				);
				$g_multiagent_fallback_values['agent_1']     = array(
					'fallback_values' => 'yes',
					'ref_name'        => '',
					'enable'          => '1',
					'number'          => $main_number,
					'title'           => '',
					'description'     => '',
					'pre_filled'      => '',
					'agent_image_id'  => '',
					'agent_image_url' => '',
					'timings'         => 'always',
				);
				$g_multiagent_fallback_values['agent_count'] = '2';
			}

			// Greetings Dialog PRO - 1 - Form
			$g_pro_1 = array(
				'greetings_pro_1' => array(
					'id'              => 'ht_ctc_greetings_pro_1',
					'title'           => 'Greetings - Form',
					'dbrow'           => 'ht_ctc_greetings_pro_1',
					'class'           => 'ctc_greetings_settings pr_ht_ctc_greetings_pro_1',
					'fallback_values' => $g_form_fallback_values,
					'inputs'          => array(
						array(
							'template' => 'collapsible_start',
							'title'    => 'Greetings Dialog - Form',
						),
						array(
							'db'       => 'empty',
							'template' => 'empty',
						),
						array(
							'title'       => '',
							'db'          => '',
							'template'    => 'admin-greetings-pro-1',
							'path'        => $pro_path,
							'description' => '',
						),
						array(
							'title'       => 'Email',
							'db'          => 'email',
							'template'    => 'text',
							'label'       => 'Email Notification',
							'placeholder' => 'e.g. admin@example.com , admin2@example.com',
							'description' => 'Get notification at these email addresses when the user send the form - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/greetings-form/#email_notification">more info</a> <br> If you are using cache plugins, set cache lifespan to less than 8 hours or use <strong>webhooks feature</strong>',
						),
						array(
							'title'        => 'Webhook',
							'db'           => 'webhook',
							'template'     => 'text',
							'label'        => 'Webhook address',
							'description'  => 'Triggers this Webhook URL with form data when the user send the form - <a target="_blank" href="https://holithemes.com/plugins/click-to-chat/greetings-form/#webhooks">more info</a>',
							'parent_class' => 'pr_g_p_1_webhook',
						),
						array(
							'title'         => 'Header - Background Color',
							'db'            => 'header_bg_color',
							'template'      => 'color',
							'default_color' => '#075e54',
							'description'   => 'Header - Background Color',
							'parent_class'  => 'pr_g_p_1_header_bg_color',
						),
						array(
							'title'         => 'Form - Background Color',
							'db'            => 'main_bg_color',
							'template'      => 'color',
							'default_color' => '#ece5dd',
							'description'   => 'Form - Background Color',
							'parent_class'  => 'pr_g_p_1_main_bg_color',
						),
						array(
							'title'         => 'Message Box - Background Color',
							'db'            => 'message_box_bg_color',
							'template'      => 'color',
							'default_color' => '#dcf8c6',
							'description'   => 'Main Content as a Message Box with Background Color',
							'parent_class'  => 'pr_g_p_1_message_box_bg_color',
						),
						array(
							'title'    => 'Email Notification Subject, Call to Action - button type',
							'template' => 'element_details_start',
						),
						array(
							'title'       => 'Email Notification: Subject',
							'db'          => 'email_subject',
							'template'    => 'text',
							'label'       => 'Email Notification: Subject',
							'placeholder' => 'Click to Chat - Greetings Form',
							'description' => 'If blank: Click to Chat - Greetings Form. Use placeholders like {field_1}, {field_2} , etc.. to insert user input into the subject. ',
						),
						array(
							'title'        => __( 'Call to Action - button type', 'click-to-chat-pro' ),
							'db'           => 'cta_style',
							'template'     => 'select',
							'description'  => 'Call to Action - button type (Color settings at Click to Chat -> Customize)',
							'list'         => array(
								'1'   => 'Themes Button (style-1)',
								'7_1' => 'Button with WhatsApp Icon (style-7 Extend)',
							),
							'parent_class' => 'pr_g_p_1_cta_style',
						),
						array(
							'template' => 'element_details_end',
						),
						array(
							'template'    => 'collapsible_end',
							'description' => "<a href='https://holithemes.com/plugins/click-to-chat/greetings-form/' target='_blank'>Greetings Form</a>",
						),
					),
				),
			);

			$g_pro_2 = array(
				'greetings_pro_2' => array(
					'id'              => 'ht_ctc_greetings_pro_2',
					'title'           => 'Greetings - Multi Agent',
					'dbrow'           => 'ht_ctc_greetings_pro_2',
					'class'           => 'ctc_greetings_settings pr_ht_ctc_greetings_pro_2',
					'fallback_values' => $g_multiagent_fallback_values,
					'inputs'          => array(
						array(
							'template' => 'collapsible_start',
							'title'    => 'Greetings Dialog - Multi Agent',
						),
						array(
							'db'       => 'empty',
							'template' => 'empty',
						),
						array(
							'title'       => '',
							'db'          => '',
							'template'    => 'admin-greetings-pro-2',
							'path'        => $pro_path,
							'description' => '',
						),
						array(
							'title'        => __( 'Offline Agents', 'click-to-chat-pro' ),
							'db'           => 'agent_offline',
							'template'     => 'select',
							'list'         => array(
								'chat'   => 'Chat when offline (Displays agent with offline message)',
								'nochat' => 'Disable chat (Displays agent with offline message)',
								'hide'   => 'Hide',
							),
							'parent_class' => 'pr_g_p_2_offline_agents',
						),
						array(
							'title'        => __( 'Initial Display', 'click-to-chat-pro' ),
							'db'           => 'initial_display',
							'template'     => 'select',
							'list'         => array(
								'agents'  => 'Agents',
								'content' => 'Greetings content',
							),
							'description'  => '',
							'parent_class' => 'pr_g_p_2_inital_display',
						),
						array(
							'title'         => 'Header - Background Color',
							'db'            => 'header_bg_color',
							'template'      => 'color',
							'default_color' => '#075e54',
							'description'   => 'Header - Background Color',
							'parent_class'  => 'pr_g_p_2_header_bg_color',
						),
						array(
							'title'         => 'Main Content - Background Color',
							'db'            => 'main_bg_color',
							'template'      => 'color',
							'default_color' => '#ece5dd',
							'description'   => 'Main Content - Background Color',
							'parent_class'  => 'pr_g_p_2_main_bg_color ctc_init_display_none',
						),
						array(
							'title'         => 'Message Box - Background Color',
							'db'            => 'message_box_bg_color',
							'template'      => 'color',
							'default_color' => '#dcf8c6',
							'description'   => 'Main Content as a Message Box with Background Color',
							'parent_class'  => 'pr_g_p_2_message_box_bg_color ctc_init_display_none',
						),
						array(
							'title'        => __( 'Call to Action - button type', 'click-to-chat-pro' ),
							'db'           => 'cta_style',
							'template'     => 'select',
							'description'  => 'Call to Action - button type (Color settings at Click to Chat -> Customize)',
							'list'         => array(
								'1'   => 'Themes Button (style-1)',
								'7_1' => 'Button with WhatsApp Icon (style-7 Extend)',
							),
							'parent_class' => 'pr_g_p_1_cta_style',
						),
						array(
							'template'    => 'collapsible_end',
							'description' => "<a href='https://holithemes.com/plugins/click-to-chat/multi-agent/' target='_blank'>Greetings - Multi Agent</a>",
						),
					),
				),
			);

			// Greetings Dialog PRO - 1 - Form

			// $values = $values + $g_pro_1;

			// -- if printr($values) the greetings g_pro_1 - greetings_pro_1 is added as [0] or so.. instead of key..

			// Insert $g_pro_1 just before the last element of $values.
			// Parameters:
			// - $values : target array
			// - -1      : offset, means "count from the end → one before the last element"
			// - 0       : length, remove 0 elements (so we only insert, nothing removed)
			// - $g_pro_1: the value to insert
			array_splice( $values, -1, 0, $g_pro_1 );
			array_splice( $values, -1, 0, $g_pro_2 );

			return $values;
		}


		/**
		 * Admin demo - greetings templates
		 */
		// function demo_greetings_templates( $values ) {

		// $g_templates = [
		// 'greetings-pro-1' => plugin_dir_path( HT_CTC_PRO_PLUGIN_FILE ) . 'public/greetings/greetings-pro-1.php',
		// 'greetings-pro-2' => plugin_dir_path( HT_CTC_PRO_PLUGIN_FILE ) . 'public/greetings/greetings-pro-2.php',
		// ];

		// if ( is_array($values) ) {
		// $values = array_merge( $values, $g_templates );
		// }

		// return $values;
		// }

		/**
		 * Greetings meta editor
		 *
		 * @param array $values Values array.
		 */
		public function greetings_meta_editor( $values ) {

			$values = array(
				'header_content',
				'main_content',
				'bottom_content',
			);

			return $values;
		}


		/**
		 * Register Settings.. greetings page
		 *
		 * @param array $values Values array.
		 */
		public function greetings_register( $values ) {

			$pro_values = array(
				'ht_ctc_greetings_pro_1',
				'ht_ctc_greetings_pro_2',
			);

			$values = array_merge( $values, $pro_values );

			return $values;
		}

		/**
		 * Greetings templates..
		 *
		 * @param array $values Values array.
		 */
		public function greetings_templates( $values ) {

			// keys are like the file names (inlude 'pro' keyword in the key)
			$pro_values = array(
				'greetings-pro-1' => 'Greetings - Form',
				'greetings-pro-2' => 'Multi Agent',
				'greetings-pro-3' => 'Form with Multi Agent (Beta)',
			);

			$values = array_merge( $values, $pro_values );

			return $values;
		}

		/**
		 * Custom URL: Desktop
		 *
		 * @param array $values Values array.
		 */
		public function url_structure_d_list( $values ) {

			$pro_values = array(
				'custom_url' => __( 'Custom URL', 'click-to-chat-pro' ),
			);

			$values = array_merge( $values, $pro_values );

			return $values;
		}

		/**
		 * Custom URL: Mobile
		 *
		 * @param array $values Values array.
		 */
		public function url_structure_m_list( $values ) {

			$pro_values = array(
				'custom_url' => __( 'Custom URL', 'click-to-chat-pro' ),
			);

			$values = array_merge( $values, $pro_values );

			return $values;
		}

		/**
		 * Custom URL: Desktop
		 */
		public function custom_url_desktop() {

			$options = get_option( 'ht_ctc_chat_options' );
			$dbrow   = 'ht_ctc_chat_options';

			$custom_url_d = ( isset( $options['custom_url_d'] ) ) ? esc_attr( $options['custom_url_d'] ) : '';
			?>

		<div class="row custom_url_desktop ctc_init_display_none">
			<div class="col s6">
					<p><?php esc_html_e( 'Desktop', 'click-to-chat-pro' ); ?>: <?php esc_html_e( 'Custom URL', 'click-to-chat-pro' ); ?></p>
			</div>
			<div class="input-field col s6">
				<input placeholder="" name="<?php echo esc_attr( $dbrow ); ?>[custom_url_d]" value="<?php echo esc_attr( $custom_url_d ); ?>" id="custom_url_d" type="text">
					<label><?php esc_html_e( 'Desktop', 'click-to-chat-pro' ); ?>: <?php esc_html_e( 'Custom URL', 'click-to-chat-pro' ); ?></label>
					<p class="description"><?php esc_html_e( 'Add Full URL for Desktop', 'click-to-chat-pro' ); ?></p>
			</div>
		</div>
		
			<?php
		}

		/**
		 * Custom URL: Mobile
		 */
		public function custom_url_mobile() {

			$options = get_option( 'ht_ctc_chat_options' );
			$dbrow   = 'ht_ctc_chat_options';

			$custom_url_m = ( isset( $options['custom_url_m'] ) ) ? esc_attr( $options['custom_url_m'] ) : '';
			?>

		<div class="row custom_url_mobile ctc_init_display_none">
			<div class="col s6">
					<p><?php esc_html_e( 'Mobile', 'click-to-chat-pro' ); ?>: <?php esc_html_e( 'Custom URL', 'click-to-chat-pro' ); ?></p>
			</div>
			<div class="input-field col s6">
				<input placeholder="" name="<?php echo esc_attr( $dbrow ); ?>[custom_url_m]" value="<?php echo esc_attr( $custom_url_m ); ?>" id="custom_url_m" type="text">
					<label><?php esc_html_e( 'Mobile', 'click-to-chat-pro' ); ?>: <?php esc_html_e( 'Custom URL', 'click-to-chat-pro' ); ?></label>
					<p class="description"><?php esc_html_e( 'Add Full URL for Mobile', 'click-to-chat-pro' ); ?></p>
			</div>
		</div>
		
			<?php
		}

		// // plugins updates
		// function plugin_updates() {

		// check_ajax_referer( 'ht_ctc_pro_nonce', 'ht_ctc_pro_nonce' );

		// $url = "https://holithemes.com/shop/?edd_action=get_version&item_id=5502&license=82e81b1969fa05cf12d9fd111d9ee48b&url=https://s1.techponder.com/";
		// $response = wp_remote_post( $url );

		// $license = trim( get_option( 'ht_ctc_pro_license_key' ) );

		// $api_params = array(
		// 'edd_action' => 'get_version',
		// 'license'    => $license,
		// 'item_id'    => HT_CTC_PRO_SL_ITEM_ID,
		// 'url'        => home_url()
		// );

		// $response = wp_remote_post( HT_CTC_PRO_SL_STORE_URL, array( 'timeout' => 45, 'sslverify' => false, 'body' => $api_params ) );

		// $license_data = $response;

		// $installed_version = HT_CTC_PRO_VERSION;
		// $latest_version = '';

		// make sure the response came back okay
		// if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {

		// if ( is_wp_error( $response ) ) {
		// $message = $response->get_error_message();
		// } else {
		// $message = __( 'An error occurred, please try again. (not 200)' );
		// }

		// } else {

		// $license_data = json_decode( wp_remote_retrieve_body( $response ) );

		// if ( false === $license_data->success ) {

		// switch( $license_data->error ) {

		// case 'expired' :

		// $message = sprintf(
		// __( 'Your license key expired on %s.' ),
		// date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
		// );
		// break;

		// case 'disabled' :
		// case 'revoked' :

		// $message = __( 'Your license key has been disabled.' );
		// break;

		// case 'missing' :

		// $message = __( 'Invalid license.' );
		// break;

		// case 'invalid' :
		// case 'site_inactive' :

		// $message = __( 'Your license is not active for this URL.' );
		// break;

		// case 'item_name_mismatch' :

		// $message = sprintf( __( 'This appears to be an invalid license key for %s.' ), HT_CTC_PRO_SL_ITEM_ID );
		// break;

		// case 'no_activations_left':

		// $message = __( 'Your license key has reached its activation limit.' );
		// break;

		// default :

		// $message = __( 'An error occurred, please try again. (default) 1. might be another product license key' );
		// break;
		// }

		// } else {
		// if ($license_data->new_version) {
		// $latest_version = $license_data->new_version;
		// }
		// }

		// }

		// if not an error/issue
		// if ( empty( $message ) ) {

		// $message = "your plugin is up to date: $latest_version";
		// if ('' == $latest_version) {
		// $message = "unable to get the latest version, please contact plugin support";
		// }
		// if ($latest_version !== $installed_version) {
		// $message = "Please update the plugin";
		// }

		// }

		// $r = array(
		// 'message' => $message,
		// 'license_data' => $license_data,
		// );

		// $data = 'test';
		// wp_send_json_success($r);

		// ends early, this wont runs
		// wp_die();

		// }
	}

	new HT_CTC_PRO_Admin_Hooks();

endif; // END class_exists check
