<?php


namespace RTMTrade\OpenCart\AdminApi\Mapping;


use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Configuration\AutoMapperConfigInterface;
use AutoMapperPlus\MappingOperation\Operation;
use RTMTrade\OpenCart\AdminApi\Model\Address;
use RTMTrade\OpenCart\AdminApi\Model\PutOrderStatusHistory;
use RTMTrade\OpenCart\AdminApi\Model\SimpleCurrency;
use RTMTrade\OpenCart\AdminApi\Model\LanguageDescriptor;
use RTMTrade\OpenCart\AdminApi\Model\MethodDescriptor;
use RTMTrade\OpenCart\AdminApi\Model\Order;
use RTMTrade\OpenCart\AdminApi\Model\OrderCustomer;
use RTMTrade\OpenCart\AdminApi\Model\OrderHttpInfo;
use RTMTrade\OpenCart\AdminApi\Model\OrderInvoice;
use RTMTrade\OpenCart\AdminApi\Model\OrderProduct;
use RTMTrade\OpenCart\AdminApi\Model\OrderProductOption;
use RTMTrade\OpenCart\AdminApi\Model\OrderStatusHistory;
use RTMTrade\OpenCart\AdminApi\Model\OrderTotal;
use RTMTrade\OpenCart\AdminApi\Model\SimpleOrder;
use stdClass;

class OrderMapper implements ObjectMapperInterface
{

    public static function registerMappings(AutoMapperConfigInterface $config)
    {
        self::registerMethodDescriptor($config);
        self::registerOrderProductOption($config);
        self::registerOrderTotal($config);
        self::registerOrderProduct($config);
        self::registerOrderStatusHistory($config);
        self::registerOrderCustomer($config);
        self::registerOrderHttpInfo($config);
        self::registerOrder($config);
        self::registerSimpleOrder($config);
    }

    public static function registerSimpleOrder(AutoMapperConfigInterface $config)
    {
        $config->registerMapping(stdClass::class, SimpleOrder::class)
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d H:i:s', 'date_added')));
    }

    public static function registerOrderTotal(AutoMapperConfigInterface $config)
    {
        $config->registerMapping(stdClass::class, OrderTotal::class)
            ->forMember('id', Operation::fromProperty('order_total_id'))
        ;
    }

    protected static function registerMethodDescriptor(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, MethodDescriptor::class);
    }

    public static function registerOrderProductOption(AutoMapperConfigInterface $config)
    {
        $config->registerMapping(stdClass::class, OrderProductOption::class)
            ->forMember('productOptionId',
                Operation::mapFrom(self::asInt('product_option_id')))
            ->forMember('productOptionValueId',
                Operation::mapFrom(self::asInt('product_option_value_id')))
            ->forMember('optionId',
                Operation::mapFrom(self::asInt('option_id')))
            ->forMember('optionValueId',
                Operation::mapFrom(self::asInt('option_value_id')))
        ;
    }

    protected static function registerOrderProduct(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, OrderProduct::class)
            // map id
            ->forMember('id',
                Operation::fromProperty('order_product_id'))
            // map specific formatted object
            ->forMember('options',
                ObjectMapper::mapMultipleFromProperty(OrderProductOption::class, 'option'))
            // map misspelled field
            ->forMember('priceFormatted',
                Operation::fromProperty('price_formated'))
            // map misspelled field
            ->forMember('totalFormatted',
                Operation::fromProperty('total_formated'));
    }

    protected static function registerOrderStatusHistory(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, OrderStatusHistory::class)
            ->forMember('id', Operation::fromProperty('order_history_id'))
            // format dateAdded
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('d/m/Y H:i:s', 'date_added')))
            ->reverseMap();

        $config->registerMapping(stdClass::class, PutOrderStatusHistory::class)
            ->forMember('notify', Operation::mapFrom(fn(stdClass $source) => $source->notify ? 1 : 0))
            ->reverseMap()
        ;
    }

    protected static function registerOrder(AutoMapperConfigInterface $config): void
    {
        $config->registerMapping(stdClass::class, Order::class)
            ->forMember('id', Operation::fromProperty('order_id'))
            ->forMember('totalPrice', Operation::fromProperty('total'))
            // map collection
            ->forMember('products',
                Operation::mapCollectionTo(OrderProduct::class))
            // map invoice fields to object
            ->forMember('invoiceNumber',
                Operation::mapFrom(fn(object $source) => new OrderInvoice($source->invoice_no ?? null, $source->invoice_prefix ?? null)))
            // map language fields to object
            ->forMember('language',
                Operation::mapFrom(fn(object $source) => new LanguageDescriptor($source->language_id ?? null, $source->language_code ?? null, $source->language_filename ?? null, $source->language_directory ?? null)))
            // map currency fields to object
            ->forMember('currency',
                ObjectMapper::flatMap(SimpleCurrency::class))
            // map http info fields to object
            ->forMember('httpInfo',
                ObjectMapper::flatMap(OrderHttpInfo::class))
            // format dateAdded
            ->forMember('dateAdded',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d H:i:s', 'date_added')))
            // format dateModified
            ->forMember('dateModified',
                Operation::mapFrom(ObjectMapper::dateMapper('Y-m-d H:i:s', 'date_modified')))
            // map shipping and payment addresses and methods to objects
            ->forMember('shippingAddress',
                Operation::mapFrom(fn(object $source, AutoMapperInterface $mapper) =>
                $mapper->map(ObjectMapper::remapAddressNames($source, 'shipping'), Address::class)))
            ->forMember('paymentAddress',
                Operation::mapFrom(fn(object $source, AutoMapperInterface $mapper) => $mapper->map(ObjectMapper::remapAddressNames($source, 'payment'), Address::class)))
            ->forMember('shippingMethod',
                Operation::mapFrom(fn(object $source, AutoMapperInterface $mapper) => $mapper->map(ObjectMapper::remapMethodDescriptorArray($source, 'shipping'), MethodDescriptor::class)))
            ->forMember('paymentMethod',
                Operation::mapFrom(fn(object $source, AutoMapperInterface $mapper) => $mapper->map(ObjectMapper::remapMethodDescriptorArray($source, 'payment'), MethodDescriptor::class)))
            // map customer fields to object
            ->forMember('customer',
                ObjectMapper::flatMap(OrderCustomer::class))
            // map collection
            ->forMember('histories',
                Operation::mapCollectionTo(OrderStatusHistory::class))
            ->forMember('totals',
                Operation::mapCollectionTo(OrderTotal::class))
        ;
    }

    public static function registerOrderCustomer(AutoMapperConfigInterface $config)
    {
        $config->registerMapping(stdClass::class, OrderCustomer::class);
    }

    public static function registerOrderHttpInfo(AutoMapperConfigInterface $config)
    {
        $config->registerMapping(stdClass::class, OrderHttpInfo::class);
    }

    protected static function asInt(string $field): \Closure
    {
        return function ($source) use ($field) {
            return intval($source->{$field});
        };
    }
}