<?php

namespace RTMTrade\OpenCart\AdminApi\Api;

use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;
use InvalidArgumentException;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RTMTrade\OpenCart\AdminApi\Model\Login;
use RTMTrade\OpenCart\AdminApi\Model\ResetPassword;
use RTMTrade\OpenCart\AdminApi\Model\User;
use stdClass;

class UserApi extends BaseHttpApi
{
    public function __construct(ClientInterface $client, Configuration $config, HeaderSelector $headerSelector, AutoMapperInterface $mapper)
    {
        parent::__construct($client, $config, $headerSelector, $mapper);
    }

    /**
     * Forgotten password
     *
     * @param string $email
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function passwordForgotten(string $email): ApiResponse
    {
        [$response] = $this->passwordForgottenWithHttpInfo($email);
        return $response;
    }

    /**
     * Forgotten password
     *
     * @param string $email
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function passwordForgottenWithHttpInfo(string $email): array
    {
        $request = $this->createPasswordForgottenRequest($email);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'forgotten'
     *
     * @param string $email
     * @return Request
     */
    protected function createPasswordForgottenRequest(string $email): Request
    {

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'POST',
            $this->config->getBaseUrl() . '/forgotten',
            $headers,
            \GuzzleHttp\json_encode(['email' => $email])
        );
    }

    /**
     * Logs user into the system
     *
     * @param Login $loginObject Login object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function login(Login $loginObject): ApiResponse
    {
        [$response] = $this->loginWithHttpInfo($loginObject);
        return $this->mapper->map($response->getData(), User::class);
    }

    /**
     * Logs user into the system
     *
     * @param Login $loginObject Login object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function loginWithHttpInfo(Login $loginObject): array
    {
        $request = $this->createLoginRequest($loginObject);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'loginUser'
     *
     * @param Login $loginObject Login object (required)
     *
     * @return Request
     * @throws UnregisteredMappingException
     */
    protected function createLoginRequest(Login $loginObject): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );
        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'POST',
            $this->config->getBaseUrl() . '/login',
            $headers,
            \GuzzleHttp\json_encode($this->getMapper()->map($loginObject, stdClass::class))
        );
    }

    /**
     * Logs out current logged in user session
     *
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function logout(): ApiResponse
    {
        [$response] = $this->logoutWithHttpInfo();
        return $response;
    }

    /**
     * Logs out current logged in user session
     *
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function logoutWithHttpInfo(): array
    {
        $request = $this->createLogoutRequest();

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'logoutUser'
     *
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createLogoutRequest(): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'POST',
            $this->config->getBaseUrl() . '/logout',
            $headers
        );
    }

    /**
     * Reset password
     *
     * @param ResetPassword $resetPasswordObject Reset password object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function resetPassword(ResetPassword $resetPasswordObject): ApiResponse
    {
        [$response] = $this->resetPasswordWithHttpInfo($resetPasswordObject);
        return $response;
    }

    /**
     * Reset password
     *
     * @param ResetPassword $resetPasswordObject Reset password object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function resetPasswordWithHttpInfo(ResetPassword $resetPasswordObject): array
    {
        $request = $this->createResetPasswordRequest($resetPasswordObject);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'reset'
     *
     * @param ResetPassword $resetPasswordObject Reset password object (required)
     *
     * @return Request
     * @throws UnregisteredMappingException
     */
    protected function createResetPasswordRequest(ResetPassword $resetPasswordObject): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );
        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'POST',
            $this->config->getBaseUrl() . '/reset',
            $headers,
            \GuzzleHttp\json_encode($this->mapper->map($resetPasswordObject, stdClass::class))
        );
    }

    /**
     * Get user account details
     *
     *
     * @return User
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function getCurrentUserDetails(): User
    {
        [$response] = $this->getCurrentUserDetailsWithHttpInfo();
        return $this->mapper->map($response->getData(), User::class);
    }

    /**
     * Get user account details
     *
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function getCurrentUserDetailsWithHttpInfo(): array
    {
        $request = $this->createGetCurrentUserDetailsRequest();

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'userDetails'
     *
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createGetCurrentUserDetailsRequest(): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'GET',
            $this->config->getBaseUrl() . '/user',
            $headers
        );
    }

    public function getUserTokenByEmail(string $email): ApiResponse
    {
        [$response] = $this->getUserTokenByEmailWithHttpInfo($email);
        return $response;
    }

    public function getUserTokenByEmailWithHttpInfo(string $email): array
    {
        $request = $this->createGetUserTokenByEmailRequest($email);

        return $this->sendRequest($request);
    }

    /**
     * Gets the user login token by its email address
     *
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createGetUserTokenByEmailRequest(string $email): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'GET',
            $this->config->getBaseUrl() . '/user/token/' . $email,
            $headers
        );
    }
}
