<?php

namespace RTMTrade\OpenCart\AdminApi\Api;

use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use DateTime;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Uri;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RTMTrade\OpenCart\AdminApi\Model\InvoiceNumber;
use RTMTrade\OpenCart\AdminApi\Model\Order;
use RTMTrade\OpenCart\AdminApi\Model\OrderStatusHistory;
use RTMTrade\OpenCart\AdminApi\Model\PutOrderStatusHistory;
use RTMTrade\OpenCart\AdminApi\Model\SimpleOrder;
use RTMTrade\OpenCart\AdminApi\ObjectSerializer;
use stdClass;

class OrderApi extends AbstractApi
{
    public function __construct(ClientInterface $client, Configuration $config, HeaderSelector $headerSelector, AutoMapperInterface $mapper, LoggerInterface $logger = null)
    {
        parent::__construct($client, $config, $headerSelector, $mapper, $logger);
        $this->resourceClass = SimpleOrder::class;
        $this->resourceName = 'orders';

        $this->allowedOperations = [
            'add',
            'get',
            'update',
            'delete',
            'list',
            'pagination',
            'addSubresourceById' => [
                'invoice' => null
            ],
            'updateSubresourceById' => [
                'invoice' => InvoiceNumber::class
            ],
            'listSubresourceById' => [
                'added_from' => Order::class,
                'added_on' => Order::class,
                'id_larger_than' => Order::class,
                'id_lower_than' => Order::class,
                'modified_from' => Order::class,
                'modified_on' => Order::class,
                'status' => Order::class
            ]
        ];
    }

    public function get(int $id): ?object
    {
        $this->resourceClass = Order::class;
        $order = parent::get($id);
        $this->resourceClass = SimpleOrder::class;
        return $order;
    }

    /**
     * Custom route 'orderadmin'
     *
     * @param object $object
     * @return Request
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    protected function createAddRequest(object $object): Request
    {
        return parent::createAddRequest($object)
            ->withUri(new Uri("{$this->config->getBaseUrl()}/orderadmin"));
    }

    /**
     * Generate invoice number to the orders
     *
     * @param int $orderId Order Id (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function addInvoice(int $orderId): ApiResponse
    {
        [$response] = $this->addInvoiceWithHttpInfo($orderId);
        return $response;
    }

    /**
     * Generate invoice number to the orders
     *
     * @param int $orderId Order Id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function addInvoiceWithHttpInfo(int $orderId): array
    {
        $request = $this->createAddSubresourceByIdRequest($orderId, 'invoice');

        return $this->sendRequest($request);
    }

    /**
     * Set custom invoice number to the orders
     *
     * @param int $id Order Id (required)
     * @param InvoiceNumber $invoiceNumberObject Invoice number object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function invoiceUpdate(int $id, InvoiceNumber $invoiceNumberObject): ApiResponse
    {
        [$response] = $this->invoiceUpdateWithHttpInfo($id, $invoiceNumberObject);
        return $response;
    }

    /**
     * Set custom invoice number to the orders
     *
     * @param int $id Order Id (required)
     * @param InvoiceNumber $invoiceNumberObject Invoice number object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function invoiceUpdateWithHttpInfo(int $id, InvoiceNumber $invoiceNumberObject): array
    {
        $request = $this->createUpdateSubresourceByIdRequest($id, 'invoice', $invoiceNumberObject);

        return $this->sendRequest($request);
    }

    /**
     * Add order history
     *
     * @param int $id Order Id (required)
     * @param PutOrderStatusHistory $historyObject History object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function addOrderHistory(int $id, PutOrderStatusHistory $historyObject): ApiResponse
    {
        [$response] = $this->addOrderHistoryWithHttpInfo($id, $historyObject);
        return $response;
    }

    /**
     * Add order history
     *
     * @param int $id Order Id (required)
     * @param PutOrderStatusHistory $historyObject History object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function addOrderHistoryWithHttpInfo(int $id, PutOrderStatusHistory $historyObject): array
    {
        $request = $this->createAddOrderHistoryRequest($id, $historyObject);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'orderhistoryUpdate'
     *
     * @param int $id Order Id (required)
     * @param OrderStatusHistory $historyObject History object (required)
     *
     * @return Request
     * @throws UnregisteredMappingException
     */
    protected function createAddOrderHistoryRequest(int $id, PutOrderStatusHistory $historyObject): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );
        $headers = $this->addDefaultHeaders($headers);

        // $_tempBody is the method argument, if present
        $httpBody = \GuzzleHttp\json_encode($this->mapper->map($historyObject, stdClass::class));

        $uri = "{$this->config->getBaseUrl()}/orderhistory/{$id}";

        return new Request('PUT', $uri, $headers, $httpBody);
    }

    /**
     * Get order details by customer ID
     *
     * @param int $customerId Customer Id (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listByCustomerId(int $customerId): array
    {
        [$response] = $this->listByCustomerIdWithHttpInfo($customerId);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get order details by customer ID
     *
     * @param int $id Customer Id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listByCustomerIdWithHttpInfo(int $id): array
    {
        $request = $this->createListBySubresourceRequest('user', $id);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders filtered by date parameter (between)
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listAddedBetween(DateTime $addedFrom, DateTime $addedTo): array
    {
        [$response] = $this->listAddedBetweenWithHttpInfo($addedFrom, $addedTo);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by date parameter (between)
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listAddedBetweenWithHttpInfo(DateTime $addedFrom, DateTime $addedTo): array
    {
        $request = $this->createListAddedBetweenRequest($addedFrom, $addedTo);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'listAddedBetween'
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createListAddedBetweenRequest(DateTime $addedFrom, DateTime $addedTo): Request
    {
        $resourcePath = sprintf("/%s/added_from/%s/added_to/%s", $this->resourceName, ObjectSerializer::toPathValue($addedFrom), ObjectSerializer::toPathValue($addedTo));

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        $headers = $this->addDefaultHeaders($headers);

        return new Request(
            'GET',
            $this->config->getBaseUrl() . $resourcePath,
            $headers,
        );
    }

    /**
     * Get list of orders filtered by date parameter (from)
     *
     * @param DateTime $addedFrom From date added (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listAddedFrom(DateTime $addedFrom): array
    {
        [$response] = $this->listAddedFromWithHttpInfo($addedFrom);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by date parameter (from)
     *
     * @param DateTime $addedFrom From date added (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listAddedFromWithHttpInfo(DateTime $addedFrom): array
    {
        $request = $this->createListBySubresourceRequest('added_from', $addedFrom);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders filtered by  date parameter (on)
     *
     * @param DateTime $addedOn Orders added on date (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listAddedOn(DateTime $addedOn): array
    {
        [$response] = $this->listAddedOnWithHttpInfo($addedOn);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by  date parameter (on)
     *
     * @param DateTime $addedOn Orders added on date (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listAddedOnWithHttpInfo(DateTime $addedOn): array
    {
        $request = $this->createListBySubresourceRequest('added_on', $addedOn);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders with details
     *
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function listDetails(): array
    {
        [$response] = $this->listDetailsWithHttpInfo();

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details
     *
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsWithHttpInfo(): array
    {
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $request = $this->createListRequest();

        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders with details filtered by date parameter (between)
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsAddedBetween(DateTime $addedFrom, DateTime $addedTo): array
    {
        [$response] = $this->listDetailsAddedBetweenWithHttpInfo($addedFrom, $addedTo);

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by date parameter (between)
     *
     * @param DateTime $addedFrom From date added (required)
     * @param DateTime $addedTo To date added (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsAddedBetweenWithHttpInfo(DateTime $addedFrom, DateTime $addedTo): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listAddedBetweenWithHttpInfo($addedFrom, $addedTo);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with details filtered by date parameter (from)
     *
     * @param DateTime $addedFrom From date added (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsAddedFrom(DateTime $addedFrom): array
    {
        [$response] = $this->listDetailsAddedFromWithHttpInfo($addedFrom);

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by date parameter (from)
     *
     * @param DateTime $addedFrom From date added (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsAddedFromWithHttpInfo(DateTime $addedFrom): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listAddedFromWithHttpInfo($addedFrom);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with details filtered by  date parameter (on)
     *
     * @param DateTime $addedOn Orders added on date (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsAddedOn(DateTime $addedOn): array
    {
        [$response] = $this->listDetailsAddedOnWithHttpInfo($addedOn);

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by  date parameter (on)
     *
     * @param DateTime $addedOn Orders added on date (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsAddedOnWithHttpInfo(DateTime $addedOn): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listAddedOnWithHttpInfo($addedOn);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with id larger than {id}
     *
     * @param int $id Order id (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsIdLargerThan(int $id): array
    {
        [$response] = $this->listDetailsIdLargerThanWithHttpInfo($id);

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with id larger than {id}
     *
     * @param int $id Order id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsIdLargerThanWithHttpInfo(int $id): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listIdLargerThanWithHttpInfo($id);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with id lower than {id}
     *
     * @param int $id Order id (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsIdLowerThan(int $id): array
    {
        [$response] = $this->listDetailsIdLowerThanWithHttpInfo($id);

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with id lower than {id}
     *
     * @param int $id Order id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsIdLowerThanWithHttpInfo(int $id): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listIdLowerThanWithHttpInfo($id);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with details filtered by modified date parameter (between)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     * @param DateTime $modifiedTo To date modified (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsModifiedBetween(DateTime $modifiedFrom, DateTime $modifiedTo): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        [$response] = $this->listDetailsModifiedBetweenWithHttpInfo($modifiedFrom, $modifiedTo);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by modified date parameter (between)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     * @param DateTime $modifiedTo To date modified (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsModifiedBetweenWithHttpInfo(DateTime $modifiedFrom, DateTime $modifiedTo): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $arr = $this->listModifiedBetweenWithHttpInfo($modifiedFrom, $modifiedTo);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $arr;
    }

    /**
     * Get list of orders with details filtered by modified date parameter (from)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsModifiedFrom(DateTime $modifiedFrom): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        [$response] = $this->listDetailsModifiedFromWithHttpInfo($modifiedFrom);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by modified date parameter (from)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsModifiedFromWithHttpInfo(DateTime $modifiedFrom): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listModifiedFromWithHttpInfo($modifiedFrom);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with details filtered by modified date parameter (on)
     *
     * @param DateTime $modifiedOn Orders modified on date (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsModifiedOn(DateTime $modifiedOn): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        [$response] = $this->listDetailsModifiedOnWithHttpInfo($modifiedOn);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by modified date parameter (on)
     *
     * @param DateTime $modifiedOn Orders modified on date (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsModifiedOnWithHttpInfo(DateTime $modifiedOn): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listModifiedOnWithHttpInfo($modifiedOn);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with details filtered by status id
     *
     * @param string $statusId Order status id or ids (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsByStatusId(string $statusId): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        [$response] = $this->listDetailsByStatusIdWithHttpInfo($statusId);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with details filtered by status id
     *
     * @param string $statusId Order status id or ids (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listDetailsByStatusIdWithHttpInfo(string $statusId): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->listByStatusIdWithHttpInfo($statusId);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders with id larger than {id}
     *
     * @param int $id Order id (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listIdLargerThan(int $id): array
    {
        [$response] = $this->listIdLargerThanWithHttpInfo($id);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with id larger than {id}
     *
     * @param int $id Order id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listIdLargerThanWithHttpInfo(int $id): array
    {
        $request = $this->createListBySubresourceRequest('id_larger_than', $id);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders with id lower than {id}
     *
     * @param int $id Order id (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listIdLowerThan(int $id): array
    {
        [$response] = $this->listIdLowerThanWithHttpInfo($id);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders with id lower than {id}
     *
     * @param int $id Order id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listIdLowerThanWithHttpInfo(int $id): array
    {
        $request = $this->createListBySubresourceRequest('id_lower_than', $id);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders filtered by modified date parameter (between)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     * @param DateTime $modifiedTo To date modified (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listModifiedBetween(DateTime $modifiedFrom, DateTime $modifiedTo): array
    {
        [$response] = $this->listModifiedBetweenWithHttpInfo($modifiedFrom, $modifiedTo);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by modified date parameter (between)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     * @param DateTime $modifiedTo To date modified (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listModifiedBetweenWithHttpInfo(DateTime $modifiedFrom, DateTime $modifiedTo): array
    {
        $request = $this->createListModifiedBetweenRequest($modifiedFrom, $modifiedTo);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'listModifiedBetween'
     *
     * @param DateTime $modifiedFrom From date modified (required)
     * @param DateTime $modifiedTo To date modified (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createListModifiedBetweenRequest(DateTime $modifiedFrom, DateTime $modifiedTo): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $resourcePath = sprintf("/orders/modified_from/%s/modified_to/%s",
            ObjectSerializer::toPathValue($modifiedFrom),
            ObjectSerializer::toPathValue($modifiedTo)
        );

        $uri = $this->config->getBaseUrl() . $resourcePath;
        return new Request('GET', $uri, $headers);
    }

    /**
     * Get list of orders filtered by modified date parameter (from)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listModifiedFrom(DateTime $modifiedFrom): array
    {
        [$response] = $this->listModifiedFromWithHttpInfo($modifiedFrom);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by modified date parameter (from)
     *
     * @param DateTime $modifiedFrom From date modified (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listModifiedFromWithHttpInfo(DateTime $modifiedFrom): array
    {
        $request = $this->createListBySubresourceRequest('modified_from', $modifiedFrom);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders filtered by modified date parameter (on)
     *
     * @param DateTime $modifiedOn Orders modified on date (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listModifiedOn(DateTime $modifiedOn): array
    {
        [$response] = $this->listModifiedOnWithHttpInfo($modifiedOn);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by modified date parameter (on)
     *
     * @param DateTime $modifiedOn Orders modified on date (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listModifiedOnWithHttpInfo(DateTime $modifiedOn): array
    {
        $request = $this->createListBySubresourceRequest('modified_on', $modifiedOn);

        return $this->sendRequest($request);
    }

    /**
     * Get list of orders
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function paginationDetailed(int $limit, int $page): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $orders = $this->pagination($limit, $page);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $orders;
    }

    /**
     * Get list of orders
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function paginationDetailedWithHttpInfo(int $limit, int $page): array
    {
        // set resource name to orders/details
        $name = $this->resourceName;
        $this->resourceName = 'orders/details';
        $this->resourceClass = Order::class;

        $apiResponse = $this->paginationWithHttpInfo($limit, $page);

        // reset resource name
        $this->resourceName = $name;
        $this->resourceClass = SimpleOrder::class;

        return $apiResponse;
    }

    /**
     * Get list of orders filtered by status id
     *
     * @param string $statusId Order status id or ids (required)
     *
     * @return Order[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listByStatusId(string $statusId): array
    {
        [$response] = $this->listByStatusIdWithHttpInfo($statusId);
        return $this->mapper->mapMultiple($response->getData(), Order::class);
    }

    /**
     * Get list of orders filtered by status id
     *
     * @param string $statusId Order status id or ids (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listByStatusIdWithHttpInfo(string $statusId): array
    {
        $request = $this->createListBySubresourceRequest('status', $statusId);

        return $this->sendRequest($request);
    }

    /**
     * Update order by ID
     *
     * @param int $orderId Order Id (required)
     * @param int $statusId Order status object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function updateStatusByStatusId(int $orderId, int $statusId): ApiResponse
    {
        [$response] = $this->updateStatusByStatusIdWithHttpInfo($orderId, $statusId);
        return $response;
    }

    /**
     * Update order by ID
     *
     * @param int $id Order Id (required)
     * @param int $statusId Order status object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function updateStatusByStatusIdWithHttpInfo(int $id, int $statusId): array
    {
        $request = $this->createUpdateStatusByStatusIdRequest($id, $statusId);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'updateStatusByStatusId'
     *
     * @param int $id Order Id (required)
     * @param int $statusId Order status object (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createUpdateStatusByStatusIdRequest(int $id, int $statusId): Request
    {
        $httpBody = \GuzzleHttp\json_encode(['status' => $statusId]);

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/orders/{$id}";
        return new Request('PUT', $uri, $headers, $httpBody);
    }

    /**
     * Update order status by order ID
     *
     * @param int $id Order Id (required)
     * @param string $statusName
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function updateStatusByName(int $id, string $statusName): ApiResponse
    {
        [$response] = $this->updateStatusByStatusNameWithHttpInfo($id, $statusName);
        return $response;
    }

    /**
     * Update order status by order ID
     *
     * @param int $id Order Id (required)
     * @param string $statusName Order status object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function updateStatusByStatusNameWithHttpInfo(int $id, string $statusName): array
    {
        $request = $this->createUpdateStatusByNameRequest($id, $statusName);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'updateStatusByName'
     *
     * @param int $id Order Id (required)
     * @param string $statusName Order status object (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createUpdateStatusByNameRequest(int $id, string $statusName): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $httpBody = \GuzzleHttp\json_encode(['status' => $statusName]);

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/order_status/{$id}";
        return new Request('PUT', $uri, $headers, $httpBody);
    }

    /**
     * Update order tracking number by order ID
     *
     * @param int $id Order Id (required)
     * @param string $trackingNumber Order tracking number object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function updateTrackingNumber(int $id, string $trackingNumber): ApiResponse
    {
        [$response] = $this->updateTrackingNumberWithHttpInfo($id, $trackingNumber);
        return $response;
    }

    /**
     * Update order tracking number by order ID
     *
     * @param int $id Order Id (required)
     * @param string string Order tracking number object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function updateTrackingNumberWithHttpInfo(int $id, string $trackingNumber): array
    {
        $request = $this->createUpdateTrackingNumberRequest($id, $trackingNumber);

        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'updateTrackingNumber'
     *
     * @param int $id Order Id (required)
     * @param string $trackingNumber
     * @return Request
     */
    protected function createUpdateTrackingNumberRequest(int $id, string $trackingNumber): Request
    {
        $httpBody = \GuzzleHttp\json_encode(['tracking' => $trackingNumber]);

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/trackingnumber/{$id}";
        return new Request('PUT', $uri, $headers, $httpBody);
    }
}
