<?php


namespace RTMTrade\OpenCart\AdminApi\Api;


use App\Misc\Inflect;
use AutoMapperPlus\AutoMapperInterface;
use AutoMapperPlus\Exception\UnregisteredMappingException;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\MultipartStream;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Utils;
use InvalidArgumentException;
use Psr\Log\LoggerInterface;
use RTMTrade\OpenCart\AdminApi\ApiException;
use RTMTrade\OpenCart\AdminApi\Configuration;
use RTMTrade\OpenCart\AdminApi\HeaderSelector;
use RTMTrade\OpenCart\AdminApi\Mapping\ObjectMapper;
use RTMTrade\OpenCart\AdminApi\Model\ApiResponse;
use RTMTrade\OpenCart\AdminApi\ObjectSerializer;
use SplFileObject;
use stdClass;

/**
 * Class AbstractApi
 * @package RTMTrade\OpenCart\AdminApi\Api
 *
 * @template T the resource class
 */
abstract class AbstractApi extends BaseHttpApi
{
    /**
     * @psalm-param class-string<T>
     */
    protected string $resourceClass;

    /**
     * @var string
     */
    protected string $resourceName;

    protected array $allowedOperations;

    protected ?LoggerInterface $logger;

    /**
     * @param ClientInterface $client
     * @param Configuration $config
     * @param HeaderSelector $headerSelector
     * @param AutoMapperInterface $mapper
     * @param LoggerInterface|null $logger
     */
    public function __construct(
        ClientInterface $client,
        Configuration $config,
        HeaderSelector $headerSelector,
        AutoMapperInterface $mapper,
        LoggerInterface $logger = null
    )
    {
        parent::__construct($client, $config, $headerSelector, $mapper);
        $this->logger = $logger;

        $this->allowedOperations = [
            'get',
            'add',
            'update',
            'delete',
            'list',
            'pagination',
            'bulkAdd',
            'bulkUpdate',
            'bulkDelete'
        ];
    }

    /**
     * @return string
     */
    public function getResourceName(): string
    {
        return $this->resourceName;
    }

    /**
     * Add new resource to store
     *
     * @param T $object resource object (required)
     *
     * @return array of ['id' => val]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function add($object): array
    {
        [$response] = $this->addWithHttpInfo($object);
        return $response->getData();
    }

    /**
     * Get resource by ID
     *
     * @param int $id resource Id (required)
     *
     *
     * @return T|null
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function get(int $id): ?object
    {
        /** @var ApiResponse $response */
        [$response] = $this->getWithHttpInfo($id);

        return $this->mapper->map($response->getData(), $this->resourceClass);
    }

    /**
     * Update resource by ID
     *
     * @param int $id resource Id (required)
     * @param T $resourceObject resource object (required)
     *
     * @return ApiResponse
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function update(int $id, object $resourceObject): ApiResponse
    {
        /** @var ApiResponse $response */
        [$response] = $this->updateWithHttpInfo($id, $resourceObject);
        return $response;
    }

    /**
     * Delete resource by ID
     *
     * @param int $id resource Id (required)
     *
     * @return ApiResponse
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function delete(int $id): ApiResponse
    {
        [$response] = $this->deleteWithHttpInfo($id);
        return $response;
    }

    /**
     * Get resource list
     *
     * @return T[]
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function list(): array
    {
        [$response] = $this->listWithHttpInfo();
        return $this->mapper->mapMultiple($response->getData(), $this->resourceClass);
    }

    /**
     * Get list of resources by page and limit
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return T[]
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function pagination(int $limit, int $page): array
    {
        [$response] = $this->paginationWithHttpInfo($limit, $page);
        return $this->mapper->mapMultiple($response->getData(), $this->resourceClass);
    }

    /**
     * @param string $subresource
     * @param int $limit
     * @param int $page
     * @return object[]
     * @throws ApiException
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function paginationSubresource(string $subresource, int $limit, int $page): array
    {
        [$response] = $this->paginationSubresourceWithHttpInfo($subresource, $limit, $page);
        $targetClass = $this->allowedOperations['paginationSubresource'][$subresource] ?? $this->resourceClass;
        return $this->mapper->mapMultiple($response->getData(), $targetClass);
    }

    /**
     * Add new resource
     *
     * @param T $object resource object (required)
     *
     * @return array of ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function addWithHttpInfo($object): array
    {
        $request = $this->createAddRequest($object);
        return $this->sendRequest($request);
    }


    /**
     * Get resource by ID
     *
     * @param int $id resource Id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function getWithHttpInfo(int $id): array
    {
        $request = $this->createGetRequest($id);

        return $this->sendRequest($request);
    }


    /**
     * Update resource by ID
     *
     * @param int $id resource Id (required)
     * @param T $object resource object (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws UnregisteredMappingException
     * @throws GuzzleException
     */
    public function updateWithHttpInfo(int $id, object $object): array
    {
        $request = $this->createUpdateRequest($id, $object);

        return $this->sendRequest($request);
    }

    /**
     * Delete resource by ID
     *
     * @param int $id resource Id (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function deleteWithHttpInfo(int $id): array
    {
        $request = $this->createDeleteRequest($id);
        return $this->sendRequest($request);
    }

    /**
     * Get resource list
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function listWithHttpInfo(): array
    {
        $request = $this->createListRequest();
        return $this->sendRequest($request);
    }

    /**
     * Get list of resources by page and limit
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function paginationWithHttpInfo(int $limit, int $page): array
    {
        $request = $this->createPaginationRequest($limit, $page);

        return $this->sendRequest($request);
    }

    /**
     * @param string $subresource
     * @param int $limit
     * @param int $page
     * @return array
     * @throws ApiException
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function paginationSubresourceWithHttpInfo(string $subresource, int $limit, int $page): array
    {
        $request = $this->createPaginationSubresourceRequest($subresource, $limit, $page);
        return $this->sendRequest($request);
    }


    /**
     * Create request for operation 'get'
     *
     * @param int $id Resource Id (required)
     *
     * @return Request
     * @throws InvalidArgumentException
     */
    protected function createGetRequest(int $id): Request
    {
        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$id}";

        return $this->createHttpGetRequest($uri);
    }

    /**
     * Create request for operation 'add'
     *
     * @param object $object resource object (required)
     *
     * @return Request
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    protected function createAddRequest(object $object): Request
    {
        if (!in_array('add', $this->allowedOperations)) {
            throw new ApiException("'add' operation not allowed for " . get_class($this), 0);
        }
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $httpBody = \GuzzleHttp\json_encode($this->mapper->map($object, stdClass::class));

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}";

        return new Request('POST', $uri, $headers, $httpBody);
    }

    /**
     * Create request for operation 'delete'
     *
     * @param int $id Resource Id (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createDeleteRequest(int $id): Request
    {
        if (!in_array('delete', $this->allowedOperations)) {
            throw new ApiException("'delete' operation not allowed for " . get_class($this), 0);
        }
        // set json content type
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        // add default headers
        $headers = $this->addDefaultHeaders($headers);


        // build uri
        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$id}";

        return new Request('DELETE', $uri, $headers);
    }


    /**
     * Create request for operation 'update'
     *
     * @param int $id resource Id (required)
     * @param object $updateObject resource object (required)
     *
     * @return Request
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    protected function createUpdateRequest(int $id, object $updateObject): Request
    {
        if (!in_array('update', $this->allowedOperations)) {
            throw new ApiException("'update' operation not allowed for " . get_class($this), 0);
        }
        // set accept and content-type headers to json
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );
        // add default headers
        $headers = $this->addDefaultHeaders($headers);

        // map object
        if (!($updateObject instanceof stdClass)) {
            $updateObject = $this->mapper->map($updateObject, stdClass::class);
        }
        $httpBody = \GuzzleHttp\json_encode($updateObject);

        // build uri
        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$id}";

        return new Request('PUT', $uri, $headers, $httpBody);
    }


    /**
     * Create request for operation 'list'
     *
     * @return Request
     * @throws ApiException
     */
    protected function createListRequest(): Request
    {
        if (!in_array('list', $this->allowedOperations)) {
            throw new ApiException("'list' operation not allowed for " . get_class($this), 0);
        }
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}";

        return new Request('GET', $uri, $headers);
    }

    /**
     * Create request for operation 'pagination'
     *
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createPaginationRequest(int $limit, int $page): Request
    {
        if (!in_array('pagination', $this->allowedOperations)) {
            throw new ApiException("'pagination' operation not allowed for " . get_class($this), 0);
        }
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/limit/{$limit}/page/{$page}";

        return new Request('GET', $uri, $headers);
    }

    /**
     * Create request for operation 'pagination'
     *
     * @param string $subresource
     * @param int $limit Limit (required)
     * @param int $page Page (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createPaginationSubresourceRequest(string $subresource, int $limit, int $page): Request
    {
        $operationName = 'paginationSubresource';
        if ($this->validateSubresourceIsAllowed($operationName, $subresource)) {
            // if operation is not allowed
            throw new ApiException("'$operationName' operation not allowed for subresource {$subresource} in " . get_class($this), 0);
        }
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}/limit/{$limit}/page/{$page}";

        return new Request('GET', $uri, $headers);
    }

    /**
     * Create request for operation 'bulkAdd'
     *
     * @param T[] $objects Array of resource objects (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createBulkAddRequest(array $objects): Request
    {
        if (!in_array('bulkAdd', $this->allowedOperations)) {
            throw new ApiException("'bulkAdd' operation not allowed for " . get_class($this), 0);
        }
        // verify the required parameter 'resourceObject' is set
        if (empty($objects)) {
            throw new InvalidArgumentException(
                'Missing the required parameter $resourceObject when calling resourceBulkAdd'
            );
        }

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );
        $headers = $this->addDefaultHeaders($headers);

        $httpBody = \GuzzleHttp\json_encode($this->mapper->mapMultiple($objects, stdClass::class));

        $uri = "{$this->config->getBaseUrl()}/bulk_{$this->resourceName}";
        return new Request('POST', $uri, $headers, $httpBody);
    }

    /**
     * Create request for operation 'bulkUpdate'
     *
     * @param T[] $updateObjects Array of resource objects (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createBulkUpdateRequest(array $updateObjects): Request
    {
        if (!in_array('bulkUpdate', $this->allowedOperations)) {
            throw new ApiException("'bulkUpdate' operation not allowed for " . get_class($this), 0);
        }
        if (empty($updateObjects)) {
            throw new InvalidArgumentException(
                'Update objects cannot be empty'
            );
        }

        $headers = $this->headerSelector->selectHeaders(
            [],
            ['application/json']
        );
        $headers = $this->addDefaultHeaders($headers);

        $httpBody = \GuzzleHttp\json_encode(ObjectMapper::asDeleteObject($updateObjects, $this->resourceName));

        $uri = "{$this->config->getBaseUrl()}/bulk_{$this->resourceName}";
        return new Request('PUT', $uri, $headers, $httpBody);
    }


    /**
     * Create request for operation 'getBySubresource'
     *
     * @param string $subresource
     * @param mixed $value subresource value (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createGetBySubresourceRequest(string $subresource, $value = null): Request
    {
        $this->validateSubresource('getBySubresource', $subresource);

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}" . (is_null($value) ?: "/{$value}");
        return new Request('GET', $uri, $headers);
    }

    /**
     * Create request for operation 'listBySubresource'
     *
     * @param string $subresource
     * @param mixed $value the subresource value
     *
     * @return Request
     * @throws ApiException
     */
    protected function createListBySubresourceRequest(string $subresource, $value): Request
    {
        $operationName = 'listBySubresource';
        if ($this->validateSubresourceIsAllowed($operationName, $subresource)) {
            // if operation is not allowed
            throw new ApiException("'$operationName' operation not allowed for subresource {$subresource} in " . get_class($this), 0);
        }

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );
        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}/" . ObjectSerializer::toPathValue($value);

        return new Request('GET', $uri, $headers);
    }

    /**
     * Create request for operation 'subresourceBulkUpdate'
     *
     * @param string $subresource
     * @param object[] $subresourceUpdateObjects Array of resource quantity objects (required)
     *
     * @return Request
     */
    protected function createSubresourceBulkUpdateRequest(string $subresource, array $subresourceUpdateObjects): Request
    {
        // todo: allowedOperations subset
        // verify the required parameter '$subresourceUpdateObjects' is set
        if (!empty($subresourceUpdateObjects)) {
            throw new InvalidArgumentException(
                'Missing the required parameter $arrayOfresourceQuantityObjects when calling resourceBulkUpdate'
            );
        }

        $headers = $this->headerSelector->selectHeaders(
            [],
            ['application/json']
        );
        $headers = $this->addDefaultHeaders($headers);

        $httpBody = \GuzzleHttp\json_encode($this->mapper->mapMultiple($subresourceUpdateObjects, stdClass::class));

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}";
        return new Request('PUT', $uri, $headers, $httpBody);
    }

    /**
     * Add multiple resources
     *
     * @param T[] $objects Array of resource objects (required)
     *
     * @return ApiResponse
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function bulkAdd(array $objects): ApiResponse
    {
        [$response] = $this->bulkAddWithHttpInfo($objects);
        return $response;
    }

    /**
     * Add multiple resources
     *
     * @param T[] $objects Array of objects (required)
     *
     * @return array of \RTMTrade\OpenCart\AdminApi\Model\ApiResponse, HTTP status code, HTTP response headers (array of strings)
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function bulkAddWithHttpInfo(array $objects): array
    {
        $request = $this->createBulkAddRequest($objects);
        return $this->sendRequest($request);
    }

    /**
     * Update multiple resource
     *
     * @param T[] $objects Array of resource objects (required)
     *
     * @return void
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function bulkUpdate(array $objects)
    {
        $this->bulkUpdateWithHttpInfo($objects);
    }

    /**
     * Update multiple resources
     *
     * @param T[] $objects Array of resource objects (required)
     *
     * @return array of null, HTTP status code, HTTP response headers (array of strings)
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function bulkUpdateWithHttpInfo(array $objects): array
    {
        $request = $this->createBulkUpdateRequest($objects);
        return $this->sendRequest($request);
    }


    /**
     * Update multiple resources
     *
     * @param object[] $objects Array of resource objects (required)
     *
     * @return void
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function bulkDelete(array $objects)
    {
        $this->bulkUpdateWithHttpInfo($objects);
    }

    /**
     * Update multiple resources
     *
     * @param int[] $ids Array of resource objects (required)
     *
     * @return array of null, HTTP status code, HTTP response headers (array of strings)
     * @throws InvalidArgumentException
     * @throws ApiException on non-2xx response
     * @throws GuzzleException
     * @throws UnregisteredMappingException
     */
    public function bulkDeleteWithHttpInfo(array $ids): array
    {
        $request = $this->createBulkDeleteRequest($ids);
        return $this->sendRequest($request);
    }

    /**
     * Create request for operation 'delete'
     *
     * @param int[] $ids Resource Ids (required)
     *
     * @return Request
     * @throws ApiException
     */
    protected function createBulkDeleteRequest(array $ids): Request
    {
        if (!in_array('bulkDelete', $this->allowedOperations)) {
            throw new ApiException("'bulkDelete' operation not allowed for " . get_class($this));
        }
        // set json content type
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        // add default headers
        $headers = $this->addDefaultHeaders($headers);

        $httpBody = \GuzzleHttp\json_encode([$this->resourceName => $ids]);

        // build uri
        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}";

        return new Request('DELETE', $uri, $headers, $httpBody);
    }

    /**
     * @param int $objectId
     * @param string $subresource
     * @param object|null $object
     * @param bool $idBeforeSubresource
     * @return Request
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    public function createAddSubresourceByIdRequest(int $objectId, string $subresource, object $object = null, bool $idBeforeSubresource = false): Request
    {
        $this->validateSubresource('addSubresourceById', $subresource, $object);

        $httpBody = $object === null
            ? null
            : \GuzzleHttp\json_encode($this->mapper->map($object, stdClass::class));

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $headers = $this->addDefaultHeaders($headers);

        if ($idBeforeSubresource) {
            $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$objectId}/{$subresource}";
        } else {
            $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}/{$objectId}";
        }

        return new Request('POST', $uri, $headers, $httpBody);
    }


    /**
     * Create request for operation 'updateSubresourceById'
     *
     * @param string $subresource
     * @param object|null $updateObject resource object (required)
     * @return Request
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    protected function createUpdateSubresourceRequest(string $subresource,
                                                      object $updateObject = null): Request
    {
        $this->validateSubresource('updateSubresource', $subresource, $updateObject);

        $httpBody = null;
        if ($updateObject !== null) {
            $httpBody = \GuzzleHttp\json_encode($this->mapper->map($updateObject, stdClass::class));
        }

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $headers = $this->addDefaultHeaders($headers);

        $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}";

        return new Request('PUT', $uri, $headers, $httpBody);
    }

    /**
     * Create request for operation 'updateSubresourceById'
     *
     * @param int $objectId resource Id (required)
     * @param string $subresource
     * @param object|null $updateObject resource object (required)
     * @param bool $idBeforeSubresource
     * @return Request
     * @throws ApiException
     * @throws UnregisteredMappingException
     */
    protected function createUpdateSubresourceByIdRequest(int $objectId,
                                                          string $subresource,
                                                          object $updateObject = null,
                                                          bool $idBeforeSubresource = false): Request
    {
        $this->validateSubresource('updateSubresourceById', $subresource, $updateObject);

        $httpBody = null;
        if ($updateObject !== null) {
            $httpBody = \GuzzleHttp\json_encode($this->mapper->map($updateObject, stdClass::class));
        }

        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            ['application/json']
        );

        $headers = $this->addDefaultHeaders($headers);

        if ($idBeforeSubresource) {
            $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$objectId}/{$subresource}";
        } else {
            $uri = "{$this->config->getBaseUrl()}/{$this->resourceName}/{$subresource}/{$objectId}";
        }

        return new Request('PUT', $uri, $headers, $httpBody);
    }

    /**
     * Create request for operation 'updateImage'
     *
     * @param string $pathPrefix
     * @param SplFileObject $file (required)
     * @param string $pathSuffix
     *
     * @return Request
     */
    protected function createPostFileRequest(string $pathPrefix, SplFileObject $file, string $pathSuffix = ''): Request
    {
        return $this->createFileRequest('POST', "{$this->config->getBaseUrl()}/{$pathPrefix}/images" . (empty($pathSuffix) ?: "/$pathSuffix"), $file);
    }

    /**
     * Create request for operation 'updateImage'
     *
     * @param string $pathPrefix prefix for 'images'
     * @param SplFileObject $file (required)
     * @param string $pathSuffix suffix for 'images'
     * @return Request
     */
    protected function createPutFileRequest(string $pathPrefix, SplFileObject $file, string $pathSuffix = ''): Request
    {
        return $this->createFileRequest(
            'PUT',
            "{$this->config->getBaseUrl()}/{$pathPrefix}/images" . (empty($pathSuffix) ?: "/$pathSuffix"),
            $file
        );
    }

    /**
     * @param string $method
     * @param string $uri
     * @param SplFileObject $file
     * @return Request
     */
    protected function createFileRequest(string $method, string $uri, SplFileObject $file): Request
    {
        $headers = $this->headerSelector->selectHeadersForMultipart(
            ['application/json']
        );

        // for HTTP post (form)
        $httpBody = new MultipartStream(['file' => Utils::tryFOpen(ObjectSerializer::toFormValue($file), 'rb')]);

        $headers = $this->addDefaultHeaders($headers);

        return new Request($method, $uri, $headers, $httpBody);
    }

    /**
     * @param string $operationName
     * @param string $subresource
     * @param object|null $object
     * @return bool true if class is valid or is null
     *              false if subresource entry does not exists or entry class doesn't match $object's class
     */
    protected function validateSubresourceClass(string $operationName, string $subresource, ?object $object = null): bool
    {
        return isset($this->allowedOperations[$operationName][$subresource]) // exists?
            && $this->allowedOperations[$operationName][$subresource] !== null // class is not null
            && $this->allowedOperations[$operationName][$subresource] !== get_class($object);
    }

    /**
     * @param string $operationName
     * @param string $subresource
     * @return bool true if subresource entry exists and operation entry is 'false' (allow all) or the subresource entry is set
     *              false if subresource entry does not exists or the operations entry for the subresource doesnt exist
     */
    protected function validateSubresourceIsAllowed(string $operationName, string $subresource): bool
    {
        return isset($this->allowedOperations[$operationName])
            && ($this->allowedOperations[$operationName] === true
                || isset($this->allowedOperations[$operationName][$subresource]));
    }

    /**
     * Preconditions for subresource operations.
     *
     * Throws exceptions if not valid.
     *
     * @param string $operationName
     * @param string $subresource
     * @param object|null $object
     * @throws ApiException if subresource is not allowed, or subresource class is not valid with the given $object
     */
    protected function validateSubresource(string $operationName, string $subresource, ?object $object = null): void
    {
        if ($this->validateSubresourceIsAllowed($operationName, $subresource)) {
            // if operation is not allowed
            throw new ApiException("'$operationName' operation not allowed for subresource {$subresource} in "
                . get_class($this), 0);
        } elseif ($this->validateSubresourceClass($operationName, $subresource, $object)) {
            // if operation is allowed, but object cannot be null
            throw new ApiException("'$operationName' operation is allowed for subresource {$subresource} in "
                . get_class($this) . ", but object should not be null", 0);
        }
    }

    protected function createHttpGetRequest(string $uri, $body = null): Request
    {
        $headers = $this->headerSelector->selectHeaders(
            ['application/json'],
            []
        );

        $headers = $this->addDefaultHeaders($headers);

        return new Request('GET', $uri, $headers, $body);
    }
}